# auto generated by update_py.py

import queue
import threading
import time
import copy
from typing import Dict, List
from enum import Enum
from dataclasses import dataclass, field
from datetime import datetime, timedelta

from tlclient.linker.fist import Fist
from tlclient.linker.timer import Timer
from tlclient.linker.pb_msg import frame_pb
from tlclient.trader.constant import ExchangeID, OffsetFlag, OrderType, Side, OrderStatus
from tlclient.trader.pb_msg import message_pb
from tlclient.trader.client import Client
from tlclient.trader.timer_helper import TimerHelper
from tlclient.trader.combo_client import handle_exception


class ComboOrderState(Enum):
    WAIT_INSERT = 1
    WAIT_ADDITIONAL_INSERT = 2
    DONE = 3
    CANCELED = 4


@dataclass
class ComboResult:
    symbol: str
    traded_vol: float = 0
    traded_price: float = 0
    insert_count: int = 0
    cancel_count: int = 0


@dataclass
class FSMData:
    combo_id: int
    state: ComboOrderState
    order_id: int = 0


@dataclass
class OrderInfo:
    '''combo order中单个订单信息'''
    combo_id: int  # 回指combo id
    trading_vol: float  # 该order正在进行的交易量
    traded_vol: float = 0  # 已完成的交易量
    slice_index: int = 0  # 分片index，从0开始

    _start_time: datetime = field(default_factory=Timer.datetime, init=False)
    _is_cancelled: bool = field(default=False, init=False)


@dataclass
class ComboOrderInfo:
    '''
        combo order下单参数
    '''
    account_id: str  # 账号/tg name
    code: str  # combo分组，后续可以取消属于某组的所有combo order

    ticker: str  # symbol，如rb2010, 601816等
    exchange: ExchangeID  # 交易所
    unit: int  # 交易量
    direction: Side  # 买卖方向
    offset: OffsetFlag  # 开平方向
    slice_time: int  # 分片单位时间长度（秒）
    slice_number: int  # 分片数量
    sub_account: str = ''  # sub account，不涉及保持默认即可

    _order_infos: Dict[int, OrderInfo] = field(default_factory=dict, init=False)
    _result: Dict[str, ComboResult] = field(default_factory=dict, init=False)
    _start_time: datetime = field(default_factory=Timer.datetime, init=False)
    _done_time: datetime = field(default=None, init=False)
    _combo_order_status: OrderStatus = field(default=OrderStatus.UNKNOWN, init=False)


class ComboClient(Client, TimerHelper):

    _CANCEL_ORDER_PREFIX_TIME = 3

    def __init__(self, name: str, env_name: str, master_addr: str, curve_server_key=''):
        super().__init__(name=name, env_name=env_name, addr=master_addr, curve_server_key=curve_server_key)
        super(Fist, self).__init__(self.logger)

        # combo id start from this
        self._combo_id_base: int = 1
        # retrieve combo id by order_id
        self._order_id_to_combo: Dict[int, int] = {}
        # retrieve combo info by combo_id
        self._combo_id_to_combo_info: Dict[int, ComboOrderInfo] = {}
        # latest mkt snap of tickers
        self._mkt_snap = {}
        self._gl = threading.Lock()
        self._fsm_q = queue.Queue()
        # rzw TODO: prefer to set thread's daemon = False
        self._fsm_t = threading.Thread(
            target=self._fsm_func, name='fsm', daemon=True
        )
        self._fsm_t.start()
        self._fsm_timer_t = threading.Thread(
            target=self._fsm_timer_func, name='timer', daemon=True
        )
        self._fsm_timer_t.start()

    def get_mkt_price(self, symbol: str, side: Side):
        with self._gl:
            try:
                if side == Side.BUY:
                    price = self._mkt_snap[symbol].ask_price[0]
                else:
                    price = self._mkt_snap[symbol].bid_price[0]
                return price
            except KeyError:
                pass

    def _cancel_order_with_lock(self, combo_id: int, order_id: str):
        combo_order_info = self._combo_id_to_combo_info[combo_id]
        if self.cancel_order(tg_name=combo_order_info.account_id, order_id=order_id) == -1:
            # rzw TODO
            self.logger.warn(f'[fsm_func] cancel order failed (order_id){order_id} (combo_id){combo_id}')
        else:
            symbol = combo_order_info.ticker
            try:
                combo_order_info._result[symbol].cancel_count += 1
            except KeyError:
                combo_order_info._result[symbol] = ComboResult(symbol=symbol, cancel_count=1)

            self._gl.release()
            self.on_req_order_cancel(order_id)
            self._gl.acquire()

    def _cancel_a_combo_order_with_lock(self, combo_id, order_id_to_skip=None):
        # rzw TODO: prefer to deal with these in fsm
        combo_order_info = self._combo_id_to_combo_info[combo_id]
        if combo_order_info._done_time is not None:
            return
        order_id_cancelled: List[int] = []
        for order_id, order in combo_order_info._order_infos.items():
            # means this order has done, no need to cancel
            if order.traded_vol == combo_order_info.unit:
                continue
            if (order_id_to_skip is None or order_id != order_id_to_skip):
                if self.cancel_order(tg_name=combo_order_info.account_id, order_id=order_id) == -1:
                    # rzw TODO
                    self.logger.warn(f'[timer_func] cancel order failed when cancelling all unfinished orders of (combo_id){combo_id} (order_id){order_id}')
                else:
                    self.logger.info(f'[timer_func] cancel order {order_id} of combo order {combo_id} (order_info){order} (vol){combo_order_info.unit}')
                    order_id_cancelled.append(order_id)
                    symbol = combo_order_info.ticker
                    try:
                        combo_order_info._result[symbol].cancel_count += 1
                    except KeyError:
                        combo_order_info._result[symbol] = ComboResult(symbol=symbol, cancel_count=1)

        combo_order_info._done_time = Timer.datetime()

        self._gl.release()
        for order_id in order_id_cancelled:
            self.on_req_order_cancel(order_id)
        self._gl.acquire()

    @handle_exception
    def _fsm_func(self):
        self.logger.info(f"{self.fist_name}'s fsm thread started!")
        while not self.is_stopped():
            fsm_data = self._fsm_q.get()
            self.logger.info(f'[fsm_func] got a fsm data {fsm_data}')
            with self._gl:
                try:
                    combo_order_info = self._combo_id_to_combo_info[fsm_data.combo_id]
                except KeyError:
                    self.logger.warn(f'[fsm_func] no this combo id {fsm_data.combo_id} when got the fsm data')
                    self._fsm_q.task_done()
                    continue
                if fsm_data.state == ComboOrderState.WAIT_INSERT:
                    symbol = combo_order_info.ticker
                    if symbol not in self._mkt_snap:
                        self.logger.warning(f'[fsm_func] no market snap for symbol {symbol}, try it again later')
                        self.subscribe_snap(combo_order_info.exchange, symbol)
                        self.insert_func_after(2, lambda: self._fsm_q.put(FSMData(combo_id=fsm_data.combo_id, state=ComboOrderState.WAIT_INSERT)))
                    else:
                        exchange = combo_order_info.exchange
                        volume = combo_order_info.unit
                        side = combo_order_info.direction
                        offset_flag = combo_order_info.offset
                        # init price
                        if side == Side.BUY:
                            price = self._mkt_snap[symbol].bid_price[0]
                        else:
                            price = self._mkt_snap[symbol].ask_price[0]
                        order_id = self.insert_order(tg_name=combo_order_info.account_id, exchange=exchange,
                                                     ticker=symbol, price=price, volume=volume,
                                                     order_type=OrderType.LIMIT, direction=side,
                                                     offset_flag=offset_flag)
                        self.logger.info(f'[fsm_func] order inserted (order_id){order_id} (combo_id){fsm_data.combo_id} (vol){combo_order_info.unit} (price){price} (symbol){symbol}')
                        self._order_id_to_combo[order_id] = fsm_data.combo_id
                        # store every inserted order's info
                        combo_order_info._order_infos[order_id] = OrderInfo(combo_id=fsm_data.combo_id, trading_vol=combo_order_info.unit)
                        try:
                            combo_order_info._result[symbol].insert_count += 1
                        except KeyError:
                            combo_order_info._result[symbol] = ComboResult(symbol=symbol, insert_count=1)
                        self._gl.release()
                        self.on_req_order_insert(order_id, exchange, symbol, price, volume, OrderType.LIMIT, side, offset_flag)
                        self._gl.acquire()
                elif fsm_data.state == ComboOrderState.WAIT_ADDITIONAL_INSERT:
                    order_info = combo_order_info._order_infos[fsm_data.order_id]

                    # calculate the new price
                    slice_index = order_info.slice_index + 1
                    if slice_index+1 == combo_order_info.slice_number:
                        # 如果是最后一单，直接挂在涨停/跌停价
                        if combo_order_info.direction == Side.BUY:
                            price: float = self._mkt_snap[combo_order_info.ticker].upper_limit_price
                        else:
                            price: float = self._mkt_snap[combo_order_info.ticker].lower_limit_price
                    elif slice_index < (combo_order_info.slice_number-1) / 2:
                        # 前1/2的订单挂在己方价上
                        if combo_order_info.direction == Side.BUY:
                            price: float = self._mkt_snap[combo_order_info.ticker].bid_price[0]
                        else:
                            price: float = self._mkt_snap[combo_order_info.ticker].ask_price[0]
                    else:
                        # 后1/2除最后一单外挂在对手价上
                        if combo_order_info.direction == Side.BUY:
                            price: float = self._mkt_snap[combo_order_info.ticker].ask_price[0]
                        else:
                            price: float = self._mkt_snap[combo_order_info.ticker].bid_price[0]

                    trading_vol = order_info.trading_vol
                    traded_vol = order_info.traded_vol
                    del combo_order_info._order_infos[fsm_data.order_id]

                    exchange = combo_order_info.exchange
                    side = combo_order_info.direction
                    offset_flag = combo_order_info.offset
                    order_id = self.insert_order(tg_name=combo_order_info.account_id, exchange=exchange,
                                                 ticker=combo_order_info.ticker, price=price,
                                                 volume=trading_vol, order_type=OrderType.LIMIT, direction=side,
                                                 offset_flag=offset_flag)
                    self.logger.info(f'[fsm_func] additional order {order_id} inserted at price {price} (combo_id){fsm_data.combo_id} (traded_vol){traded_vol} (vol){trading_vol} (slice_index){slice_index}')
                    self._order_id_to_combo[order_id] = fsm_data.combo_id
                    # store every inserted order's info
                    combo_order_info._order_infos[order_id] = OrderInfo(combo_id=fsm_data.combo_id, trading_vol=trading_vol, slice_index=slice_index, traded_vol=traded_vol)
                    try:
                        combo_order_info._result[symbol].insert_count += 1
                    except KeyError:
                        combo_order_info._result[symbol] = ComboResult(symbol=symbol, insert_count=1)

                    self._gl.release()
                    self.on_req_order_insert(order_id, exchange, symbol, price, trading_vol, OrderType.LIMIT, side, offset_flag)
                    self._gl.acquire()
                elif fsm_data.state == ComboOrderState.DONE:
                    # can not clean datas immediately here, just set a done time and then do the cleaning job in timer thread
                    combo_order_info._done_time = Timer.datetime()
            self._fsm_q.task_done()

    @handle_exception
    def _fsm_timer_func(self):
        self.logger.info(f"{self.fist_name}'s fsm timer thread started!")
        while not self.is_stopped():
            combo_done_list = []
            with self._gl:
                for combo_id, combo in self._combo_id_to_combo_info.items():
                    now = Timer.datetime()
                    if combo._done_time is not None:
                        if now - combo._done_time >= timedelta(minutes=3):
                            self.logger.info(f'[timer_func] combo order has done, clean all data of it now (combo_id){combo_id}')
                            combo_done_list.append(combo_id)
                    else:
                        # self.logger.info(f'[timer_func] order infos {combo._order_infos} (len){len(combo._order_infos)}')
                        for order_id, order in combo._order_infos.items():
                            if order.traded_vol < combo.unit:
                                if not order._is_cancelled and now - order._start_time >= timedelta(seconds=combo.slice_time / combo.slice_number - self._CANCEL_ORDER_PREFIX_TIME):
                                    self.logger.warning(f'[timer_func] cancel order {order_id} (combo_id){combo_id} (symbol){combo.ticker}')
                                    self._cancel_order_with_lock(combo_id, order_id)
                                    order._is_cancelled = True

                if combo_done_list:
                    for combo_id in combo_done_list:
                        for order_id in self._combo_id_to_combo_info[combo_id]._order_infos.keys():
                            # delete all order ids belonged to the combo
                            del self._order_id_to_combo[order_id]
                        # delete the combo info
                        del self._combo_id_to_combo_info[combo_id]
                    continue

            time.sleep(0.05)

    def _insert_combo_order(self, order_info: ComboOrderInfo, combo_id: int):
        self._combo_id_to_combo_info[combo_id] = order_info
        self._fsm_q.put(FSMData(combo_id=combo_id, state=ComboOrderState.WAIT_INSERT))
        self.logger.info(f'[insert_combo_order] (id){combo_id} (info){order_info}')

    def insert_combo_order(self, order_info: ComboOrderInfo) -> int:
        assert order_info.slice_time / order_info.slice_number >= self._CANCEL_ORDER_PREFIX_TIME, f'time slice must be ge than {self._CANCEL_ORDER_PREFIX_TIME}!! now is {order_info.slice_time / order_info.slice_number}'

        with self._gl:
            combo_id = self._combo_id_base
            # subscribe these tickers' market snap
            if order_info.ticker not in self._mkt_snap:
                self.subscribe_snap(order_info.exchange, order_info.ticker)
            self._insert_combo_order(order_info, combo_id)
            self._combo_id_base += 1
        return combo_id

    def cancel_all_combo_orders(self, groups=None):
        '''取消所有combo orders, 注意该过程是异步的，需要在 combo_rtn_order_cb 中进行确认'''
        self.logger.info(f'[cancel_all_combo_orders] prepare to cancel all combo orders of groups {groups}')
        with self._gl:
            for combo_id, combo_order in self._combo_id_to_combo_info.items():
                if groups is None or combo_order.code in groups:
                    self._cancel_a_combo_order_with_lock(combo_id)

    def is_combo_order(self, order_id: int) -> bool:
        with self._gl:
            return order_id in self._order_id_to_combo

    def _on_rsp_order_insert(self, obj, frame_nano):
        if not self.is_combo_order(obj.order_id):
            self.on_rsp_order_insert(obj, frame_nano)
            return
        if obj.err_id == frame_pb.ErrType.ERR_TYPE_NO_ERR:
            return

        with self._gl:
            try:
                combo_id = self._order_id_to_combo[obj.order_id]
                combo_order_info = self._combo_id_to_combo_info[combo_id]
                order_info = combo_order_info._order_infos[obj.order_id]
            except KeyError:
                self.logger.warn(f'[on_rsp_order_insert] can not retrieve enough info by order id {obj.order_id}')
                return
            symbol = combo_order_info.ticker

        self.combo_error_cb(combo_id, symbol, obj.err_msg)

        self.logger.warning(f'[on_rsp_order_insert] to cancel combo order {combo_id} (order_id){obj.order_id}')
        with self._gl:
            if order_info.slice_index+1 < combo_order_info.slice_number:
                # 下单失败，直接开始下一次下单尝试
                self._fsm_q.put(FSMData(combo_id=combo_id, state=ComboOrderState.WAIT_ADDITIONAL_INSERT, order_id=obj.order_id))
            else:
                # do not call combo_rtn_order_cb here
                self._cancel_a_combo_order_with_lock(combo_id=combo_id, order_id_to_skip=obj.order_id)

    def _on_rsp_order_cancel(self, obj, frame_nano):
        if not self.is_combo_order(obj.order_id):
            self.on_rsp_order_cancel(obj, frame_nano)
            return
        if obj.err_id == frame_pb.ErrType.ERR_TYPE_NO_ERR:
            return

        with self._gl:
            try:
                combo_id = self._order_id_to_combo[obj.order_id]
                combo_order_info = self._combo_id_to_combo_info[combo_id]
                order_info = combo_order_info._order_infos[obj.order_id]
            except KeyError:
                self.logger.warn(f'[on_rsp_order_cancel] can not retrieve enough info by order id {obj.order_id}')
                return
            symbol = combo_order_info.ticker

        self.combo_error_cb(combo_id, symbol, obj.err_msg)

        self.logger.warning(f'[on_rsp_order_cancel] to cancel combo order {combo_id} (order_id){obj.order_id}')
        with self._gl:
            if order_info.slice_index+1 < combo_order_info.slice_number:
                # 撤单失败，直接开始下一次下单尝试
                self._fsm_q.put(FSMData(combo_id=combo_id, state=ComboOrderState.WAIT_ADDITIONAL_INSERT, order_id=obj.order_id))
            else:
                # do not call combo_rtn_order_cb here
                self._cancel_a_combo_order_with_lock(combo_id=combo_id, order_id_to_skip=obj.order_id)

    def _on_mkt_snap(self, obj, msg_type, frame_nano):
        with self._gl:
            self._mkt_snap[obj.ticker.decode()] = copy.deepcopy(obj)
        self.on_mkt_snap(obj, msg_type, frame_nano)

    def _on_rtn_order(self, obj: message_pb.RtnOrder, frame_nano):
        if not self.is_combo_order(obj.order_id):
            self.on_rtn_order(obj, frame_nano)
            return

        if obj.order_status == OrderStatus.UNKNOWN or obj.order_status == OrderStatus.NO_TRADE_QUEUEING:
            return

        with self._gl:
            try:
                combo_id = self._order_id_to_combo[obj.order_id]
                combo_order_info = self._combo_id_to_combo_info[combo_id]
                order_info = combo_order_info._order_infos[obj.order_id]
            except KeyError:
                return

            order_status = obj.order_status
            self.logger.info(f'[on_rtn_order] (order_id){obj.order_id} (combo_id){combo_id} (order_status){order_status}')

            if order_status == OrderStatus.ALL_TRADED:
                # should notify the combo_rtn_order_cb in on_rtn_trade (because it's no traded_price here)
                combo_order_info._combo_order_status = OrderStatus.ALL_TRADED
            elif order_status in (OrderStatus.CANCELED, OrderStatus.NO_TRADE_CANCELED, OrderStatus.PART_TRADE_CANCELED):
                do_notify = False
                if order_info.slice_index + 1 < combo_order_info.slice_number:
                    if combo_order_info._done_time is None:
                        # 尝试下一次下单
                        self._fsm_q.put(FSMData(combo_id=combo_id, state=ComboOrderState.WAIT_ADDITIONAL_INSERT, order_id=obj.order_id))
                    else:
                        # done time不为None表示主动cancel
                        do_notify = True
                else:
                    # 最后一次尝试下单也失败了，发一个通知
                    self._fsm_q.put(FSMData(combo_id=combo_id, state=ComboOrderState.DONE))
                    do_notify = True

                if do_notify:
                    result = copy.copy(combo_order_info._result[obj.symbol])
                    self._gl.release()
                    self.combo_rtn_order_cb(combo_id=combo_id, result=result, status=order_status)
                    self._gl.acquire()

    def _on_rtn_trade(self, obj: message_pb.RtnTrade, frame_nano):
        if not self.is_combo_order(obj.order_id):
            self.on_rtn_trade(obj, frame_nano)
            return

        with self._gl:
            try:
                combo_id = self._order_id_to_combo[obj.order_id]
                combo_order_info = self._combo_id_to_combo_info[combo_id]
                order_info = combo_order_info._order_infos[obj.order_id]
            except KeyError:
                return

            self.logger.info(f'[on_rtn_trade] (order_id){obj.order_id} (combo_id){combo_id} (traded_price){obj.traded_price} (traded_vol){obj.traded_vol} (vol){combo_order_info.unit}')
            result = combo_order_info._result[obj.symbol]
            total_traded = result.traded_vol + obj.traded_vol
            if total_traded != 0:
                result.traded_price = (result.traded_vol * result.traded_price + obj.traded_vol * obj.traded_price) / total_traded
                result.traded_vol = total_traded

            order_info.traded_vol += obj.traded_vol
            order_info.trading_vol -= obj.traded_vol
            if combo_order_info._combo_order_status == OrderStatus.ALL_TRADED or order_info.traded_vol == combo_order_info.unit:
                result = copy.copy(result)
                self._gl.release()
                self.combo_rtn_order_cb(combo_id=combo_id, result=result, status=OrderStatus.ALL_TRADED)
                self._gl.acquire()
                # if order status marked ALL_TRADED, put a done event into fsm
                self._fsm_q.put(FSMData(combo_id=combo_id, state=ComboOrderState.DONE))

    # should be overrided
    def combo_rtn_order_cb(self, combo_id: int, result: ComboResult, status: OrderStatus):
        pass

    def combo_error_cb(self, combo_id: int, contract: str, err_msg: str):
        pass

    # could override these two callback func
    def on_req_order_insert(self, order_id: int, exchange: ExchangeID, symbol: str, price: float, vol: float, order_type: OrderType, side: Side, offset_flag: OffsetFlag):
        pass

    def on_req_order_cancel(self, order_id: int):
        pass
