/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as api from 'aws-cdk-lib/aws-apigateway';
import * as kinesis from 'aws-cdk-lib/aws-kinesis';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import * as logs from 'aws-cdk-lib/aws-logs';
import * as cloudwatch from 'aws-cdk-lib/aws-cloudwatch';
/**
 * @summary The properties for the ApiGatewayToKinesisStreamsProps class.
 */
export interface ApiGatewayToKinesisStreamsProps {
    /**
     * Optional user-provided props to override the default props for the API Gateway.
     *
     * @default - Default properties are used.
     */
    readonly apiGatewayProps?: api.RestApiProps;
    /**
     * API Gateway request template for the PutRecord action.
     * If not provided, a default one will be used.
     *
     * @default - { "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))",
     *  "PartitionKey": "$input.path('$.partitionKey')" }
     */
    readonly putRecordRequestTemplate?: string;
    /**
     * API Gateway request model for the PutRecord action.
     * If not provided, a default one will be created.
     *
     * @default - {"$schema":"http://json-schema.org/draft-04/schema#","title":"PutRecord proxy single-record payload","type":"object",
     * "required":["data","partitionKey"],"properties":{"data":{"type":"string"},"partitionKey":{"type":"string"}}}
     */
    readonly putRecordRequestModel?: api.ModelOptions;
    /**
     * API Gateway request template for the PutRecords action.
     * If not provided, a default one will be used.
     *
     * @default - { "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records'))
     *  { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }
     */
    readonly putRecordsRequestTemplate?: string;
    /**
     * API Gateway request model for the PutRecords action.
     * If not provided, a default one will be created.
     *
     * @default - {"$schema":"http://json-schema.org/draft-04/schema#","title":"PutRecords proxy payload data","type":"object","required":["records"],
     * "properties":{"records":{"type":"array","items":{"type":"object",
     * "required":["data","partitionKey"],"properties":{"data":{"type":"string"},"partitionKey":{"type":"string"}}}}}}
     */
    readonly putRecordsRequestModel?: api.ModelOptions;
    /**
     * Existing instance of Kinesis Stream, providing both this and `kinesisStreamProps` will cause an error.
     *
     * @default - None
     */
    readonly existingStreamObj?: kinesis.Stream;
    /**
     * Optional user-provided props to override the default props for the Kinesis Data Stream.
     *
     * @default - Default properties are used.
     */
    readonly kinesisStreamProps?: kinesis.StreamProps;
    /**
     * User provided props to override the default props for the CloudWatchLogs LogGroup.
     *
     * @default - Default props are used
     */
    readonly logGroupProps?: logs.LogGroupProps;
    /**
     * Whether to create recommended CloudWatch alarms
     *
     * @default - Alarms are created
     */
    readonly createCloudWatchAlarms?: boolean;
}
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
export declare class ApiGatewayToKinesisStreams extends Construct {
    readonly apiGateway: api.RestApi;
    readonly apiGatewayRole: iam.Role;
    readonly apiGatewayCloudWatchRole?: iam.Role;
    readonly apiGatewayLogGroup: logs.LogGroup;
    readonly kinesisStream: kinesis.Stream;
    readonly cloudwatchAlarms?: cloudwatch.Alarm[];
    /**
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToKinesisStreamsProps} props - user provided props for the construct.
     * @since 1.62.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: ApiGatewayToKinesisStreamsProps);
    private getPutRecordTemplate;
    private getPutRecordModel;
    private getPutRecordsTemplate;
    private getPutRecordsModel;
}
