"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * Contains static factory methods for creating health checks for different protocols.
 *
 * @stability stable
 */
class HealthCheck {
    /**
     * Construct a HTTP health check.
     *
     * @stability stable
     */
    static http(options = {}) {
        return new HealthCheckImpl(shared_interfaces_1.Protocol.HTTP, options.healthyThreshold, options.unhealthyThreshold, options.interval, options.timeout, options.path);
    }
    /**
     * Construct a HTTP2 health check.
     *
     * @stability stable
     */
    static http2(options = {}) {
        return new HealthCheckImpl(shared_interfaces_1.Protocol.HTTP2, options.healthyThreshold, options.unhealthyThreshold, options.interval, options.timeout, options.path);
    }
    /**
     * Construct a GRPC health check.
     *
     * @stability stable
     */
    static grpc(options = {}) {
        return new HealthCheckImpl(shared_interfaces_1.Protocol.GRPC, options.healthyThreshold, options.unhealthyThreshold, options.interval, options.timeout);
    }
    /**
     * Construct a TCP health check.
     *
     * @stability stable
     */
    static tcp(options = {}) {
        return new HealthCheckImpl(shared_interfaces_1.Protocol.TCP, options.healthyThreshold, options.unhealthyThreshold, options.interval, options.timeout);
    }
}
exports.HealthCheck = HealthCheck;
_a = JSII_RTTI_SYMBOL_1;
HealthCheck[_a] = { fqn: "@aws-cdk/aws-appmesh.HealthCheck", version: "1.124.0" };
class HealthCheckImpl extends HealthCheck {
    constructor(protocol, healthyThreshold = 2, unhealthyThreshold = 2, interval = cdk.Duration.seconds(5), timeout = cdk.Duration.seconds(2), path) {
        super();
        this.protocol = protocol;
        this.healthyThreshold = healthyThreshold;
        this.unhealthyThreshold = unhealthyThreshold;
        this.interval = interval;
        this.timeout = timeout;
        this.path = path;
        if (healthyThreshold < 2 || healthyThreshold > 10) {
            throw new Error('healthyThreshold must be between 2 and 10');
        }
        if (unhealthyThreshold < 2 || unhealthyThreshold > 10) {
            throw new Error('unhealthyThreshold must be between 2 and 10');
        }
        if (interval.toMilliseconds() < 5000 || interval.toMilliseconds() > 300000) {
            throw new Error('interval must be between 5 seconds and 300 seconds');
        }
        if (timeout.toMilliseconds() < 2000 || timeout.toMilliseconds() > 60000) {
            throw new Error('timeout must be between 2 seconds and 60 seconds');
        }
        // Default to / for HTTP Health Checks
        if (path === undefined && (protocol === shared_interfaces_1.Protocol.HTTP || protocol === shared_interfaces_1.Protocol.HTTP2)) {
            this.path = '/';
        }
    }
    bind(_scope, options) {
        return {
            virtualNodeHealthCheck: {
                protocol: this.protocol,
                healthyThreshold: this.healthyThreshold,
                unhealthyThreshold: this.unhealthyThreshold,
                intervalMillis: this.interval.toMilliseconds(),
                timeoutMillis: this.timeout.toMilliseconds(),
                path: this.path,
                port: options.defaultPort,
            },
            virtualGatewayHealthCheck: {
                protocol: this.protocol,
                healthyThreshold: this.healthyThreshold,
                unhealthyThreshold: this.unhealthyThreshold,
                intervalMillis: this.interval.toMilliseconds(),
                timeoutMillis: this.timeout.toMilliseconds(),
                path: this.path,
                port: options.defaultPort,
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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