# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fallback_property']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'fallback-property',
    'version': '0.2.0',
    'description': 'A decorator which prefers a precalculated attribute over calling the decorated method.',
    'long_description': "===================\nfallback-property\n===================\n\n.. image:: https://img.shields.io/pypi/v/fallback-property.svg\n    :target: https://pypi.python.org/pypi/fallback-property\n\n.. image:: https://travis-ci.org/jonasundderwolf/fallback-property.png?branch=master\n    :target: http://travis-ci.org/jonasundderwolf/fallback-property\n    :alt: Build Status\n\n.. image:: https://coveralls.io/repos/github/jonasundderwolf/fallback-property/badge.svg?branch=master\n   :target: https://coveralls.io/github/jonasundderwolf/fallback-property?branch=master\n\n.. image:: https://img.shields.io/pypi/pyversions/fallback-property.svg\n    :target: https://pypi.python.org/pypi/fallback-property\n\n\nRequirements\n============\n\n- Python 3.6+\n\n\nWhat is it?\n===========\n\n``fallback_property`` transforms a function into a property and uses the\ndecorated function as fallback if no value was assigned to the property itself.\nA special descriptor (``fallback_property.FallbackDescriptor``)\nis used internally.\n\n\nDjango (or similar frameworks)\n------------------------------\n\n``fallback_property`` is useful if you have a function that aggregates\nvalues from related objects, which could already be fetched using an annotated\nqueryset.\nThe decorator will favor the precalculated value over calling the actual method.\n\nIt is especially helpful, if you optimize your application and want to\nreplace legacy or performance critical properties with precalulated values\nusing ``.annotate()``.\n\n\nHow to use it?\n==============\n\nSimply define a function and use the decorator ``fallback_property`` ::\n\n    from fallback_property import fallback_property\n\n    class Foo:\n\n        @fallback_property()\n        def fallback_func(self):\n            return 7\n\n\nArguments\n---------\n\nThe ``fallback_property()`` has two optional arguments.\n\n``cached: bool = True``\n    If the property is accessed multiple times, call the fallback function only once.\n\n``logging: bool = False``\n    Log a warning if there was a fallback to the decorated, original method.\n\n\nUsage Example (Django)\n======================\n\nSuppose we have the following models ::\n\n    from django.db import models\n\n\n    class Pipeline(model.Model):\n        @property\n        def total_length(self):\n            return sum(self.parts.values_list('length', flat=True))\n\n\n    class Parts(models.Model):\n        length = models.PositiveIntegerField()\n        pipeline = models.ForeignKey(Pipeline, related_name='parts')\n\n\nCalling ``pipline.total_length`` will always trigger another query and is\neven more expensive when dealing with multiple objects. This can be\noptimized by using ``.annotate()`` and ``fallback_property()`` ::\n\n    from django.db import models, QuerySet\n    from django.db.functions import Coalesce\n    from django.db.models import Sum\n    from fallback_property import fallback_property\n\n\n    class PipelineQuerySet(QuerySet):\n\n        def with_total_length(self):\n            return self.annotate(\n                total_length=Coalesce(\n                    Sum('parts__length', output_field=models.IntegerField()),\n                    0,\n                )\n            )\n\n\n    class Pipeline(model.Model):\n\n        @fallback_property(logging=True)\n        def total_length(self):\n            return sum(self.parts.values_list('length', flat=True))\n\n\nYou can now access the ``total_length`` without triggering another query or\nget a warning, when the fallback function is used ::\n\n    pipeline = Pipeline.objects.with_total_length().first()\n    print(pipeline.total_length)\n\n\n**Important: The annotated value and the property must have the same name.**\n\n\nRelated objects\n---------------\n\nWhen dealing with related objects in Django be aware that the ORM imposes certain limitations:\n\nIn the following example one might expect to get an instance of ``User``, but instead the\nvalue of the primary key is returned::\n\n    from django.db import models, QuerySet\n    from django.db.functions import Coalesce\n    from django.db.models import F\n    from fallback_property import fallback_property\n\n\n    class PartQuerySet(QuerySet):\n\n        def with_owner(self):\n            return self.annotate(\n                owner=Coalesce(\n                    F('_owner'),\n                    F('pipeline__owner'),\n                    None,\n                )\n            )\n\n\n    class Pipeline(model.Model):\n        owner = models.ForeignKey(User)\n\n\n    class Parts(models.Model):\n        _owner = models.ForeignKey(User, blank=True, null=True, on_delete=models.SET_NULL)\n        length = models.PositiveIntegerField()\n        pipeline = models.ForeignKey(Pipeline, related_name='parts')\n\n        objects = PartQuerySet()\n\n        @fallback_property()\n        def owner(self):\n            return self._owner or self.pipline.owner\n\n\n    >>> print(Part.objects.with_owner().first().owner)\n    >>> 1\n\n\nDevelopment\n===========\n\nThis project is using `poetry <https://poetry.eustace.io/>`_ to manage all\ndev dependencies.\n\nClone this repository and run ::\n\n   poetry install\n   poetry run pip install django\n\nto create a virtual environment with all dependencies.\n\nYou can now run the test suite using ::\n\n  poetry run pytest\n\n\nThis repository follows the `angular commit conventions <https://github.com/marionebl/commitlint/tree/master/@commitlint/config-angular>`_.\nYou can register a pre-commit hook to validate your commit messages by using\n`husky <https://github.com/typicode/husky>`_. The configurations are already in place if\nyou have nodejs installed. Just run ::\n\n   npm install\n\n\nand the pre-commit hook will be registered.\n",
    'author': 'Jonas und der Wolf GmbH',
    'author_email': 'opensource@jonasundderwolf.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jonasunnderwolf/fallback-property',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6',
}


setup(**setup_kwargs)
