#!/usr/bin/env python3

_pitch = """
CREATE TABLE IF NOT EXISTS pitch (
  pitch_id INT AUTO_INCREMENT,
  pitch_type CHAR(2) COMMENT \'The type of pitch derived from Statcast.\',
  game_date DATE COMMENT \'Date of the Game.\',
  release_speed DECIMAL(4,1) COMMENT \'Pitch velocities from 2008-16 are via Pitch F/X, and adjusted to roughly out-of-hand release point. All velocities from 2017 and beyond are Statcast, which are reported out-of-hand.\',
  release_pos_x DECIMAL(5,1) COMMENT \'Horizontal Release Position of the ball measured in feet from the catcher\\\'s perspective.\',
  release_pos_z DECIMAL(5,1) COMMENT \'Vertical Release Position of the ball measured in feet from the catcher\\\'s perspective.\',
  player_name VARCHAR(100) COMMENT \'Player\\\'s name tied to the event of the search.\',
  batter MEDIUMINT COMMENT \'MLB Player Id tied to the play event.\',
  pitcher MEDIUMINT COMMENT \'MLB Player Id tied to the play event.\',
  events VARCHAR(30) COMMENT \'Event of the resulting Plate Appearance.\',
  description VARCHAR(50) COMMENT \'Description of the resulting pitch.\',
  spin_dir CHAR(0) COMMENT \'* Deprecated field from the old tracking system.\',
  spin_rate_deprecated CHAR(0) COMMENT \'* Deprecated field from the old tracking system. Replaced by release_spin\',
  break_angle_deprecated CHAR(0) COMMENT \'* Deprecated field from the old tracking system.\',
  break_length_deprecated CHAR(0) COMMENT \'* Deprecated field from the old tracking system.\',
  zone TINYINT COMMENT \'Zone location of the ball when it crosses the plate from the catcher\\\'s perspective. https://baseballsavant.mlb.com/sections/statcast_search_v2/images/zones.png\',
  des TEXT COMMENT \'Plate appearance description from game day.\',
  game_type CHAR(1) COMMENT \'Type of Game. E = Exhibition, S = Spring Training, R = Regular Season, F = Wild Card, D = Divisional Series, L = League Championship Series, W = World Series\',
  stand CHAR(1) COMMENT \'Side of the plate batter is standing.\',
  p_throws CHAR(1) COMMENT \'Hand pitcher throws with.\',
  home_team VARCHAR(3) COMMENT \'Abbreviation of home team.\',
  away_team VARCHAR(3) COMMENT \'Abbreviation of away team.\',
  type CHAR(1) COMMENT \'Short hand of pitch result. B = ball, S = strike, X = in play.\',
  hit_location TINYINT COMMENT \'Position of first fielder to touch the ball.\',
  bb_type VARCHAR(15) COMMENT \'Batted ball type, ground_ball, line_drive, fly_ball, popup.\',
  balls TINYINT COMMENT \'Pre-pitch number of balls in count.\',
  strikes TINYINT COMMENT \'Pre-pitch number of strikes in count.\',
  game_year SMALLINT COMMENT \'Year game took place.\',
  pfx_x DECIMAL(5,4) COMMENT \'Horizontal movement in feet from the catcher\\\'s perspective.\',
  pfx_z DECIMAL(5,4) COMMENT \'Vertical movement in feet from the catcher\\\'s perpsective.\',
  plate_x DECIMAL(6,4) COMMENT \'Horizontal position of the ball when it crosses home plate from the catcher\\\'s perspective.\',
  plate_z DECIMAL(6,4) COMMENT \'Vertical position of the ball when it crosses home plate from the catcher\\\'s perspective.\',
  on_3b MEDIUMINT COMMENT \'Pre-pitch MLB Player Id of Runner on 3B.\',
  on_2b MEDIUMINT COMMENT \'Pre-pitch MLB Player Id of Runner on 2B.\',
  on_1b MEDIUMINT COMMENT \'Pre-pitch MLB Player Id of Runner on 1B.\',
  outs_when_up TINYINT COMMENT \'Pre-pitch number of outs.\',
  inning TINYINT COMMENT \'Pre-pitch inning number.\',
  inning_topbot CHAR(3) COMMENT \'Pre-pitch top or bottom of inning.\',
  hc_x DECIMAL(5,2) COMMENT \'Hit coordinate X of batted ball.\',
  hc_y DECIMAL(5,2) COMMENT \'Hit coordinate Y of batted ball.\',
  tfs_deprecated CHAR(0) COMMENT \'* Deprecated field from old tracking system.\',
  tfs_zulu_deprecated CHAR(0) COMMENT \'* Deprecated field from old tracking system.\',
  umpire CHAR(0) COMMENT \'* Deprecated field from old tracking system.\',
  sv_id VARCHAR(15) COMMENT \'Non-unique Id of play event per game.\',
  vx0 DECIMAL(5,3) COMMENT \'The velocity of the pitch, in feet per second, in x-dimension, determined at y=50 feet.\',
  vy0 DECIMAL(7,4) COMMENT \'The velocity of the pitch, in feet per second, in y-dimension, determined at y=50 feet.\',
  vz0 DECIMAL(6,4) COMMENT \'The velocity of the pitch, in feet per second, in z-dimension, determined at y=50 feet.\',
  ax DECIMAL(6,4) COMMENT \'The acceleration of the pitch, in feet per second per second, in x-dimension, determined at y=50 feet.\',
  ay DECIMAL(6,4) COMMENT \'The acceleration of the pitch, in feet per second per second, in y-dimension, determined at y=50 feet.\',
  az DECIMAL(6,4) COMMENT \'The acceleration of the pitch, in feet per second per second, in z-dimension, determined at y=50 feet.\',
  sz_top DECIMAL(5,4) COMMENT \'Top of the batter\\\'s strike zone set by the operator when the ball is halfway to the plate.\',
  sz_bot DECIMAL(5,4) COMMENT \'Bottom of the batter\\\'s strike zone set by the operator when the ball is halfway to the plate.\',
  hit_distance_sc SMALLINT COMMENT \'Projected hit distance of the batted ball.\',
  launch_speed DECIMAL(4,1) COMMENT \'Exit velocity of the batted ball as tracked by Statcast. For the limited subset of batted balls not tracked directly, estimates are included based on the process described here.\',
  launch_angle TINYINT COMMENT \'Launch angle of the batted ball as tracked by Statcast. For the limited subset of batted balls not tracked directly, estimates are included based on the process described here.\',
  effective_speed DECIMAL(6,3) COMMENT \'Derived speed based on the the extension of the pitcher\\\'s release.\',
  release_spin_rate SMALLINT COMMENT \'Spin rate of pitch tracked by Statcast.\',
  release_extension DECIMAL(5,3) COMMENT \'Release extension of pitch in feet as tracked by Statcast.\',
  game_pk MEDIUMINT COMMENT \'Unique Id for Game.\',
  fielder_2 MEDIUMINT COMMENT \'MLB Player Id for catcher.\',
  fielder_3 MEDIUMINT COMMENT \'MLB Player Id for 1B.\',
  fielder_4 MEDIUMINT COMMENT \'MLB Player Id for 2B.\',
  fielder_5 MEDIUMINT COMMENT \'MLB Player Id for 3B.\',
  fielder_6 MEDIUMINT COMMENT \'MLB Player Id for SS.\',
  fielder_7 MEDIUMINT COMMENT \'MLB Player Id for LF.\',
  fielder_8 MEDIUMINT COMMENT \'MLB Player Id for CF.\',
  fielder_9 MEDIUMINT COMMENT \'MLB Player Id for RF.\',
  release_pos_y DECIMAL(6,4) COMMENT \'Release position of pitch measured in feet from the catcher\\\'s perspective.\',
  estimated_ba_using_speedangle DECIMAL(4,3) COMMENT \'Estimated Batting Avg based on launch angle and exit velocity.\',
  estimated_woba_using_speedangle DECIMAL(4,3) COMMENT \'Estimated wOBA based on launch angle and exit velocity.\',
  woba_value DECIMAL(3,2) COMMENT \'wOBA value based on result of play.\',
  woba_denom DECIMAL(3,2) COMMENT \'wOBA denominator based on result of play.\',
  babip_value DECIMAL(3,2) COMMENT \'BABIP value based on result of play.\',
  iso_value DECIMAL(3,2) COMMENT \'ISO value based on result of play.\',
  launch_speed_angle DECIMAL(3,2) COMMENT \'Launch speed/angle zone based on launch angle and exit velocity.\',
  at_bat_number SMALLINT COMMENT \'Plate appearance number of the game.\',
  pitch_number SMALLINT COMMENT \'Total pitch number of the plate appearance.\',
  pitch_name VARCHAR(30) COMMENT \'The name of the pitch derived from the Statcast Data.\',
  home_score TINYINT COMMENT \'Pre-pitch home score\',
  away_score TINYINT COMMENT \'Pre-pitch away score\',
  bat_score TINYINT COMMENT \'Pre-pitch bat team score\',
  fld_score TINYINT COMMENT \'Pre-pitch field team score\',
  post_home_score TINYINT COMMENT \'Post-pitch home score\',
  post_away_score TINYINT COMMENT \'Post-pitch away score\',
  post_bat_score TINYINT COMMENT \'Post-pitch bat team score\',
  post_fld_score TINYINT COMMENT \'Post-pitch field team score\',
  if_fielding_alignment VARCHAR(30) COMMENT \'Infield fielding alignment at the time of the pitch.\',
  of_fielding_alignment VARCHAR(30) COMMENT \'Outfield fielding alignment at the time of the pitch.\',
  PRIMARY KEY (pitch_id)
) COMMENT \'Represents a pitch from Statcast data\';
"""

_person = """
CREATE TABLE IF NOT EXISTS  person (
  person_id VARCHAR(10) NOT NULL,
  mlbam INT NULL,
  bbref VARCHAR(20) NULL,
  fangraphs VARCHAR(10) NULL,
  retro VARCHAR(10) NULL,
  lastname TEXT NULL,
  firstname TEXT NULL,
  givenname TEXT NULL,
  nickname TEXT NULL,
  matrilinealname TEXT NULL,
  name_suffix TEXT NULL,
  birth_month INT NULL,
  birth_day INT NULL,
  birth_year INT NULL,
  death_month INT NULL,
  death_day INT NULL,
  death_year INT NULL,
  pro_played_first INT NULL,
  pro_played_last INT NULL,
  mlb_played_first INT NULL,
  mlb_played_last INT NULL,
  col_played_first INT NULL,
  col_played_last INT NULL,
  pro_managed_first INT NULL,
  pro_managed_last INT NULL,
  mlb_managed_first INT NULL,
  mlb_managed_last INT NULL,
  col_managed_first INT NULL,
  col_managed_last INT NULL,
  pro_umpired_first INT NULL,
  pro_umpired_last INT NULL,
  mlb_umpired_first INT NULL,
  mlb_umpired_last INT NULL,
  PRIMARY KEY (person_id),
  UNIQUE INDEX person_id_UNIQUE (person_id ASC),
  UNIQUE INDEX mlbam_UNIQUE (mlbam ASC),
  UNIQUE INDEX bbref_UNIQUE (bbref ASC),
  UNIQUE INDEX fangraphs_UNIQUE (fangraphs ASC),
  UNIQUE INDEX retro_UNIQUE (retro ASC)
) COMMENT \'Represents a player\';
"""

_event = """
CREATE TABLE IF NOT EXISTS event (
  event_key INT AUTO_INCREMENT,
  GAME_ID CHAR(12) COMMENT 'Game ID',
  AWAY_TEAM_ID CHAR(3) COMMENT 'Visiting team',
  INN_CT TINYINT COMMENT 'Inning',
  BAT_HOME_ID TINYINT COMMENT 'Batting team',
  OUTS_CT TINYINT COMMENT 'Outs',
  BALLS_CT TINYINT COMMENT 'Balls',
  STRIKES_CT TINYINT COMMENT 'Strikes',
  PITCH_SEQ_TX TEXT COMMENT 'Pitch sequence',
  AWAY_SCORE_CT TINYINT COMMENT 'Visitor score',
  HOME_SCORE_CT TINYINT COMMENT 'Home score',
  BAT_ID CHAR(8) COMMENT 'Batter',
  BAT_HAND_CD CHAR(1) COMMENT 'Batter hand',
  RESP_BAT_ID CHAR(8) COMMENT 'Result batter',
  RESP_BAT_HAND_CD CHAR(1) COMMENT 'Result batter hand',
  PIT_ID CHAR(8) COMMENT 'Pitcher',
  PIT_HAND_CD CHAR(1) COMMENT 'Pitcher hand',
  RESP_PIT_ID CHAR(8) COMMENT 'Result pitcher',
  RESP_PIT_HAND_CD CHAR(1) COMMENT 'Result pitcher hand',
  POS2_FLD_ID CHAR(8) COMMENT 'Catcher',
  POS3_FLD_ID CHAR(8) COMMENT 'First baseman',
  POS4_FLD_ID CHAR(8) COMMENT 'Second baseman',
  POS5_FLD_ID CHAR(8) COMMENT 'Third baseman',
  POS6_FLD_ID CHAR(8) COMMENT 'Shortstop',
  POS7_FLD_ID CHAR(8) COMMENT 'Left fielder',
  POS8_FLD_ID CHAR(8) COMMENT 'Center fielder',
  POS9_FLD_ID CHAR(8) COMMENT 'Right fielder',
  BASE1_RUN_ID CHAR(8) COMMENT 'Runner on first',
  BASE2_RUN_ID CHAR(8) COMMENT 'Runner on second',
  BASE3_RUN_ID CHAR(8) COMMENT 'Runner on third',
  EVENT_TX TEXT COMMENT 'Event text',
  LEADOFF_FL CHAR(1) COMMENT 'Leadoff flag',
  PH_FL CHAR(1) COMMENT 'Pinch-hit flag',
  BAT_FLD_CD TINYINT COMMENT 'Defensive position',
  BAT_LINEUP_ID TINYINT COMMENT 'Lineup position',
  EVENT_CD TINYINT COMMENT 'Event type',
  BAT_EVENT_FL CHAR(1) COMMENT 'Batter event flag',
  AB_FL CHAR(1) COMMENT 'Official time at bat flag',
  H_CD TINYINT COMMENT 'Hit value',
  SH_FL CHAR(1) COMMENT 'Sacrifice hit flag',
  SF_FL CHAR(1) COMMENT 'Sacrifice fly flag',
  EVENT_OUTS_CT TINYINT COMMENT 'Outs on play',
  DP_FL CHAR(1) COMMENT 'Double play flag',
  TP_FL CHAR(1) COMMENT 'Triple play flag',
  RBI_CT TINYINT COMMENT 'RBI on play',
  WP_FL CHAR(1) COMMENT 'Wild pitch flag',
  PB_FL CHAR(1) COMMENT 'Passed ball flag',
  FLD_CD TINYINT COMMENT 'Fielded by',
  BATTEDBALL_CD CHAR(1) COMMENT 'Batted ball type',
  BUNT_FL CHAR(1) COMMENT 'Bunt flag',
  FOUL_FL CHAR(1) COMMENT 'Foul flag',
  BATTEDBALL_LOC_TX TEXT COMMENT 'Hit location',
  ERR_CT TINYINT COMMENT 'Number of errors',
  ERR1_FLD_CD TINYINT COMMENT '1st error player',
  ERR1_CD CHAR(1) COMMENT '1st error type',
  ERR2_FLD_CD TINYINT COMMENT '2nd error player',
  ERR2_CD CHAR(1) COMMENT '2nd error type',
  ERR3_FLD_CD TINYINT COMMENT '3rd error player',
  ERR3_CD CHAR(1) COMMENT '3rd error type',
  BAT_DEST_ID TINYINT COMMENT 'Batter destination',
  RUN1_DEST_ID TINYINT COMMENT 'Runner on first destination',
  RUN2_DEST_ID TINYINT COMMENT 'Runner on second destination',
  RUN3_DEST_ID TINYINT COMMENT 'Runner on third destination',
  BAT_PLAY_TX TEXT COMMENT 'Play on batter',
  RUN1_PLAY_TX TEXT COMMENT 'Play on runner on first',
  RUN2_PLAY_TX TEXT COMMENT 'Play on runner on second',
  RUN3_PLAY_TX TEXT COMMENT 'Play on runner on third',
  RUN1_SB_FL CHAR(1) COMMENT 'Stolen base for runner on first',
  RUN2_SB_FL CHAR(1) COMMENT 'Stolen base for runner on second',
  RUN3_SB_FL CHAR(1) COMMENT 'Stolen base for runner on third',
  RUN1_CS_FL CHAR(1) COMMENT 'Caught stealing for runner on first',
  RUN2_CS_FL CHAR(1) COMMENT 'Caught stealing for runner on second',
  RUN3_CS_FL CHAR(1) COMMENT 'Caught stealing for runner on third',
  RUN1_PK_FL CHAR(1) COMMENT 'Pickoff of runner on first',
  RUN2_PK_FL CHAR(1) COMMENT 'Pickoff of runner on second',
  RUN3_PK_FL CHAR(1) COMMENT 'Pickoff of runner on third',
  RUN1_RESP_PIT_ID CHAR(8) COMMENT 'Pitcher charged with runner on first',
  RUN2_RESP_PIT_ID CHAR(8) COMMENT 'Pitcher charged with runner on second',
  RUN3_RESP_PIT_ID CHAR(8) COMMENT 'Pitcher charged with runner on third',
  GAME_NEW_FL CHAR(1) COMMENT 'New game flag',
  GAME_END_FL CHAR(1) COMMENT 'End game flag',
  PR_RUN1_FL CHAR(1) COMMENT 'Pinch-runner on first',
  PR_RUN2_FL CHAR(1) COMMENT 'Pinch-runner on second',
  PR_RUN3_FL CHAR(1) COMMENT 'Pinch-runner on third',
  REMOVED_FOR_PR_RUN1_ID CHAR(8) COMMENT 'Runner removed for pinch-runner on first',
  REMOVED_FOR_PR_RUN2_ID CHAR(8) COMMENT 'Runner removed for pinch-runner on second',
  REMOVED_FOR_PR_RUN3_ID CHAR(8) COMMENT 'Runner removed for pinch-runner on third',
  REMOVED_FOR_PH_BAT_ID CHAR(8) COMMENT 'Batter removed for pinch-hitter',
  REMOVED_FOR_PH_BAT_FLD_CD TINYINT COMMENT 'Position of batter removed for pinch-hitter',
  PO1_FLD_CD TINYINT COMMENT 'Fielder with first putout',
  PO2_FLD_CD TINYINT COMMENT 'Fielder with second putout',
  PO3_FLD_CD TINYINT COMMENT 'Fielder with third putout',
  ASS1_FLD_CD TINYINT COMMENT 'Fielder with first assist',
  ASS2_FLD_CD TINYINT COMMENT 'Fielder with second assist',
  ASS3_FLD_CD TINYINT COMMENT 'Fielder with third assist',
  ASS4_FLD_CD TINYINT COMMENT 'Fielder with fourth assist',
  ASS5_FLD_CD TINYINT COMMENT 'Fielder with fifth assist',
  EVENT_ID SMALLINT COMMENT 'Event number',
  HOME_TEAM_ID CHAR(3) COMMENT 'home team id',
  BAT_TEAM_ID CHAR(3) COMMENT 'batting team id',
  FLD_TEAM_ID CHAR(3) COMMENT 'fielding team id',
  BAT_LAST_ID TINYINT COMMENT 'half inning (differs from batting team if home team bats first)',
  INN_NEW_FL CHAR(1) COMMENT 'start of half inning flag',
  INN_END_FL CHAR(1) COMMENT 'end of half inning flag',
  START_BAT_SCORE_CT SMALLINT COMMENT 'score for team on offense',
  START_FLD_SCORE_CT SMALLINT COMMENT 'score for team on defense',
  INN_RUNS_CT SMALLINT COMMENT 'runs scored in this half inning',
  GAME_PA_CT SMALLINT COMMENT 'number of plate appearances in game for team on offense',
  INN_PA_CT SMALLINT COMMENT 'number of plate appearances in inning for team on offense',
  PA_NEW_FL CHAR(1) COMMENT 'start of plate appearance flag',
  PA_TRUNC_FL CHAR(1) COMMENT 'truncated plate appearance flag',
  START_BASES_CD TINYINT COMMENT 'base state at start of play',
  END_BASES_CD TINYINT COMMENT 'base state at end of play',
  BAT_START_FL CHAR(1) COMMENT 'batter is starter flag',
  RESP_BAT_START_FL CHAR(1) COMMENT 'result batter is starter flag',
  BAT_ON_DECK_ID CHAR(8) COMMENT 'ID of the batter on deck',
  BAT_IN_HOLD_ID CHAR(8) COMMENT 'ID of the batter in the hold',
  PIT_START_FL CHAR(1) COMMENT 'pitcher is starter flag',
  RESP_PIT_START_FL CHAR(1) COMMENT 'result pitcher is starter flag',
  RUN1_FLD_CD TINYINT COMMENT 'defensive position of runner on first',
  RUN1_LINEUP_CD TINYINT COMMENT 'lineup position of runner on first',
  RUN1_ORIGIN_EVENT_ID SMALLINT COMMENT 'event number on which runner on first reached base',
  RUN2_FLD_CD TINYINT COMMENT 'defensive position of runner on second',
  RUN2_LINEUP_CD TINYINT COMMENT 'lineup position of runner on second',
  RUN2_ORIGIN_EVENT_ID SMALLINT COMMENT 'event number on which runner on second reached base',
  RUN3_FLD_CD TINYINT COMMENT 'defensive position of runner on third',
  RUN3_LINEUP_CD TINYINT COMMENT 'lineup position of runner on third',
  RUN3_ORIGIN_EVENT_ID SMALLINT COMMENT 'event number on which runner on third reached base',
  RUN1_RESP_CAT_ID CHAR(8) COMMENT 'responsible catcher for runner on first',
  RUN2_RESP_CAT_ID CHAR(8) COMMENT 'responsible catcher for runner on second',
  RUN3_RESP_CAT_ID CHAR(8) COMMENT 'responsible catcher for runner on third',
  PA_BALL_CT TINYINT COMMENT 'number of balls in plate appearance',
  PA_CALLED_BALL_CT TINYINT COMMENT 'number of called balls in plate appearance',
  PA_INTENT_BALL_CT TINYINT COMMENT 'number of intentional balls in plate appearance',
  PA_PITCHOUT_BALL_CT TINYINT COMMENT 'number of pitchouts in plate appearance',
  PA_HITBATTER_BALL_CT TINYINT COMMENT 'number of pitches hitting batter in plate appearance',
  PA_OTHER_BALL_CT TINYINT COMMENT 'number of other balls in plate appearance',
  PA_STRIKE_CT TINYINT COMMENT 'number of strikes in plate appearance',
  PA_CALLED_STRIKE_CT TINYINT COMMENT 'number of called strikes in plate appearance',
  PA_SWINGMISS_STRIKE_CT TINYINT COMMENT 'number of swinging strikes in plate appearance',
  PA_FOUL_STRIKE_CT TINYINT COMMENT 'number of foul balls in plate appearance',
  PA_INPLAY_STRIKE_CT TINYINT COMMENT 'number of balls in play in plate appearance',
  PA_OTHER_STRIKE_CT TINYINT COMMENT 'number of other strikes in plate appearance',
  EVENT_RUNS_CT TINYINT COMMENT 'number of runs on play',
  FLD_ID CHAR(8) COMMENT 'id of player fielding batted ball',
  BASE2_FORCE_FL CHAR(1) COMMENT 'force play at second flag',
  BASE3_FORCE_FL CHAR(1) COMMENT 'force play at third flag',
  BASE4_FORCE_FL CHAR(1) COMMENT 'force play at home flag',
  BAT_SAFE_ERR_FL CHAR(1) COMMENT 'batter safe on error flag',
  BAT_FATE_ID TINYINT COMMENT 'fate of batter (base ultimately advanced to)',
  RUN1_FATE_ID TINYINT COMMENT 'fate of runner on first',
  RUN2_FATE_ID TINYINT COMMENT 'fate of runner on second',
  RUN3_FATE_ID TINYINT COMMENT 'fate of runner on third',
  FATE_RUNS_CT SMALLINT COMMENT 'runs scored in half inning after this event',
  ASS6_FLD_CD TINYINT COMMENT 'fielder with sixth assist',
  ASS7_FLD_CD TINYINT COMMENT 'fielder with seventh assist',
  ASS8_FLD_CD TINYINT COMMENT 'fielder with eighth assist',
  ASS9_FLD_CD TINYINT COMMENT 'fielder with ninth assist',
  ASS10_FLD_CD TINYINT COMMENT 'fielder with tenth assist',
  UNKNOWN_OUT_EXC_FL CHAR(1) COMMENT 'unknown fielding credit flag',
  UNCERTAIN_PLAY_EXC_FL CHAR(1) COMMENT 'uncertain play flag',

  PRIMARY KEY (event_key)
) COMMENT \'Represents an event in a game, as generated by Chadwick from Retrosheet data\';
"""

_game = """
CREATE TABLE IF NOT EXISTS game (
  GAME_ID CHAR(12) COMMENT 'Game ID',
  GAME_DT DATE COMMENT 'Date',
  GAME_CT TINYINT COMMENT 'Game number',
  GAME_DY TEXT COMMENT 'Day of week',
  START_GAME_TM SMALLINT COMMENT 'Start time',
  DH_FL CHAR(1) COMMENT 'DH used flag',
  DAYNIGHT_PARK_CD CHAR(1) COMMENT 'Day/night flag',
  AWAY_TEAM_ID CHAR(3) COMMENT 'Visiting team',
  HOME_TEAM_ID CHAR(3) COMMENT 'Home team',
  PARK_ID CHAR(5) COMMENT 'Game site',
  AWAY_START_PIT_ID CHAR(8) COMMENT 'Visitors starting pitcher',
  HOME_START_PIT_ID CHAR(8) COMMENT 'Home starting pitcher',
  BASE4_UMP_ID CHAR(8) COMMENT 'Home plate umpire',
  BASE1_UMP_ID CHAR(8) COMMENT 'First base umpire',
  BASE2_UMP_ID CHAR(8) COMMENT 'Second base umpire',
  BASE3_UMP_ID CHAR(8) COMMENT 'Third base umpire',
  LF_UMP_ID CHAR(8) COMMENT 'Left field umpire',
  RF_UMP_ID CHAR(8) COMMENT 'Right field umpire',
  ATTEND_PARK_CT MEDIUMINT COMMENT 'Attendance',
  SCORER_RECORD_ID TEXT COMMENT 'PS scorer',
  TRANSLATOR_RECORD_ID TEXT COMMENT 'Translator',
  INPUTTER_RECORD_ID TEXT COMMENT 'Inputter',
  INPUT_RECORD_TS DATETIME COMMENT 'Input time',
  EDIT_RECORD_TS DATETIME COMMENT 'Edit time',
  METHOD_RECORD_CD TINYINT COMMENT 'How scored',
  PITCHES_RECORD_CD TINYINT COMMENT 'Pitches entered',
  TEMP_PARK_CT SMALLINT COMMENT 'Temperature',
  WIND_DIRECTION_PARK_CD TINYINT COMMENT 'Wind direction',
  WIND_SPEED_PARK_CT TINYINT COMMENT 'Wind speed',
  FIELD_PARK_CD TINYINT COMMENT 'Field condition',
  PRECIP_PARK_CD TINYINT COMMENT 'Precipitation',
  SKY_PARK_CD TINYINT COMMENT 'Sky',
  MINUTES_GAME_CT SMALLINT COMMENT 'Time of game',
  INN_CT SMALLINT COMMENT 'Number of innings',
  AWAY_SCORE_CT SMALLINT COMMENT 'Visitor final score',
  HOME_SCORE_CT SMALLINT COMMENT 'Home final score',
  AWAY_HITS_CT SMALLINT COMMENT 'Visitor hits',
  HOME_HITS_CT SMALLINT COMMENT 'Home hits',
  AWAY_ERR_CT SMALLINT COMMENT 'Visitor errors',
  HOME_ERR_CT SMALLINT COMMENT 'Home errors',
  AWAY_LOB_CT SMALLINT COMMENT 'Visitor left on base',
  HOME_LOB_CT SMALLINT COMMENT 'Home left on base',
  WIN_PIT_ID CHAR(8) COMMENT 'Winning pitcher',
  LOSE_PIT_ID CHAR(8) COMMENT 'Losing pitcher',
  SAVE_PIT_ID CHAR(8) COMMENT 'Save',
  GWRBI_BAT_ID CHAR(8) COMMENT 'Game-winning RBI',
  AWAY_LINEUP1_BAT_ID CHAR(8) COMMENT 'Visitor batter 1',
  AWAY_LINEUP1_FLD_CD TINYINT COMMENT 'Visitor batter 1 position',
  AWAY_LINEUP2_BAT_ID CHAR(8) COMMENT 'Visitor batter 2',
  AWAY_LINEUP2_FLD_CD TINYINT COMMENT 'Visitor batter 2 position',
  AWAY_LINEUP3_BAT_ID CHAR(8) COMMENT 'Visitor batter 3',
  AWAY_LINEUP3_FLD_CD TINYINT COMMENT 'Visitor batter 3 position',
  AWAY_LINEUP4_BAT_ID CHAR(8) COMMENT 'Visitor batter 4',
  AWAY_LINEUP4_FLD_CD TINYINT COMMENT 'Visitor batter 4 position',
  AWAY_LINEUP5_BAT_ID CHAR(8) COMMENT 'Visitor batter 5',
  AWAY_LINEUP5_FLD_CD TINYINT COMMENT 'Visitor batter 5 position',
  AWAY_LINEUP6_BAT_ID CHAR(8) COMMENT 'Visitor batter 6',
  AWAY_LINEUP6_FLD_CD TINYINT COMMENT 'Visitor batter 6 position',
  AWAY_LINEUP7_BAT_ID CHAR(8) COMMENT 'Visitor batter 7',
  AWAY_LINEUP7_FLD_CD TINYINT COMMENT 'Visitor batter 7 position',
  AWAY_LINEUP8_BAT_ID CHAR(8) COMMENT 'Visitor batter 8',
  AWAY_LINEUP8_FLD_CD TINYINT COMMENT 'Visitor batter 8 position',
  AWAY_LINEUP9_BAT_ID CHAR(8) COMMENT 'Visitor batter 9',
  AWAY_LINEUP9_FLD_CD TINYINT COMMENT 'Visitor batter 9 position',
  HOME_LINEUP1_BAT_ID CHAR(8) COMMENT 'Home batter 1',
  HOME_LINEUP1_FLD_CD TINYINT COMMENT 'Home batter 1 position',
  HOME_LINEUP2_BAT_ID CHAR(8) COMMENT 'Home batter 2',
  HOME_LINEUP2_FLD_CD TINYINT COMMENT 'Home batter 2 position',
  HOME_LINEUP3_BAT_ID CHAR(8) COMMENT 'Home batter 3',
  HOME_LINEUP3_FLD_CD TINYINT COMMENT 'Home batter 3 position',
  HOME_LINEUP4_BAT_ID CHAR(8) COMMENT 'Home batter 4',
  HOME_LINEUP4_FLD_CD TINYINT COMMENT 'Home batter 4 position',
  HOME_LINEUP5_BAT_ID CHAR(8) COMMENT 'Home batter 5',
  HOME_LINEUP5_FLD_CD TINYINT COMMENT 'Home batter 5 position',
  HOME_LINEUP6_BAT_ID CHAR(8) COMMENT 'Home batter 6',
  HOME_LINEUP6_FLD_CD TINYINT COMMENT 'Home batter 6 position',
  HOME_LINEUP7_BAT_ID CHAR(8) COMMENT 'Home batter 7',
  HOME_LINEUP7_FLD_CD TINYINT COMMENT 'Home batter 7 position',
  HOME_LINEUP8_BAT_ID CHAR(8) COMMENT 'Home batter 8',
  HOME_LINEUP8_FLD_CD TINYINT COMMENT 'Home batter 8 position',
  HOME_LINEUP9_BAT_ID CHAR(8) COMMENT 'Home batter 9',
  HOME_LINEUP9_FLD_CD TINYINT COMMENT 'Home batter 9 position',
  AWAY_FINISH_PIT_ID CHAR(8) COMMENT 'Visitor finishing pitcher',
  HOME_FINISH_PIT_ID CHAR(8) COMMENT 'Home finishing pitcher',
  AWAY_TEAM_LEAGUE_ID CHAR(2) COMMENT 'visiting team league',
  HOME_TEAM_LEAGUE_ID CHAR(2) COMMENT 'home team league',
  AWAY_TEAM_GAME_CT SMALLINT COMMENT 'visiting team game number',
  HOME_TEAM_GAME_CT SMALLINT COMMENT 'home team game number',
  OUTS_CT SMALLINT COMMENT 'length of game in outs',
  COMPLETION_TX TEXT COMMENT 'information on completion of game',
  FORFEIT_TX TEXT COMMENT 'information on forfeit of game',
  PROTEST_TX TEXT COMMENT 'information on protest of game',
  AWAY_LINE_TX TEXT COMMENT 'visiting team linescore',
  HOME_LINE_TX TEXT COMMENT 'home team linescore',
  AWAY_AB_CT SMALLINT COMMENT 'visiting team AB',
  AWAY_2B_CT SMALLINT COMMENT 'visiting team 2B',
  AWAY_3B_CT SMALLINT COMMENT 'visiting team 3B',
  AWAY_HR_CT SMALLINT COMMENT 'visiting team HR',
  AWAY_RBI_CT SMALLINT COMMENT 'visiting team RBI',
  AWAY_SH_CT SMALLINT COMMENT 'visiting team SH',
  AWAY_SF_CT SMALLINT COMMENT 'visiting team SF',
  AWAY_HP_CT SMALLINT COMMENT 'visiting team HP',
  AWAY_BB_CT SMALLINT COMMENT 'visiting team BB',
  AWAY_IBB_CT SMALLINT COMMENT 'visiting team IBB',
  AWAY_SO_CT SMALLINT COMMENT 'visiting team SO',
  AWAY_SB_CT SMALLINT COMMENT 'visiting team SB',
  AWAY_CS_CT SMALLINT COMMENT 'visiting team CS',
  AWAY_GDP_CT SMALLINT COMMENT 'visiting team GDP',
  AWAY_XI_CT SMALLINT COMMENT 'visiting team reach on interference',
  AWAY_PITCHER_CT SMALLINT COMMENT 'number of pitchers used by visiting team',
  AWAY_ER_CT SMALLINT COMMENT 'visiting team individual ER allowed',
  AWAY_TER_CT SMALLINT COMMENT 'visiting team team ER allowed',
  AWAY_WP_CT SMALLINT COMMENT 'visiting team WP',
  AWAY_BK_CT SMALLINT COMMENT 'visiting team BK',
  AWAY_PO_CT SMALLINT COMMENT 'visiting team PO',
  AWAY_A_CT SMALLINT COMMENT 'visiting team A',
  AWAY_PB_CT SMALLINT COMMENT 'visiting team PB',
  AWAY_DP_CT SMALLINT COMMENT 'visiting team DP',
  AWAY_TP_CT SMALLINT COMMENT 'visiting team TP',
  HOME_AB_CT SMALLINT COMMENT 'home team AB',
  HOME_2B_CT SMALLINT COMMENT 'home team 2B',
  HOME_3B_CT SMALLINT COMMENT 'home team 3B',
  HOME_HR_CT SMALLINT COMMENT 'home team HR',
  HOME_RBI_CT SMALLINT COMMENT 'home team RBI',
  HOME_SH_CT SMALLINT COMMENT 'home team SH',
  HOME_SF_CT SMALLINT COMMENT 'home team SF',
  HOME_HP_CT SMALLINT COMMENT 'home team HP',
  HOME_BB_CT SMALLINT COMMENT 'home team BB',
  HOME_IBB_CT SMALLINT COMMENT 'home team IBB',
  HOME_SO_CT SMALLINT COMMENT 'home team SO',
  HOME_SB_CT SMALLINT COMMENT 'home team SB',
  HOME_CS_CT SMALLINT COMMENT 'home team CS',
  HOME_GDP_CT SMALLINT COMMENT 'home team GDP',
  HOME_XI_CT SMALLINT COMMENT 'home team reach on interference',
  HOME_PITCHER_CT SMALLINT COMMENT 'number of pitchers used by home team',
  HOME_ER_CT SMALLINT COMMENT 'home team individual ER allowed',
  HOME_TER_CT SMALLINT COMMENT 'home team team ER allowed',
  HOME_WP_CT SMALLINT COMMENT 'home team WP',
  HOME_BK_CT SMALLINT COMMENT 'home team BK',
  HOME_PO_CT SMALLINT COMMENT 'home team PO',
  HOME_A_CT SMALLINT COMMENT 'home team A',
  HOME_PB_CT SMALLINT COMMENT 'home team PB',
  HOME_DP_CT SMALLINT COMMENT 'home team DP',
  HOME_TP_CT SMALLINT COMMENT 'home team TP',
  UMP_HOME_NAME_TX TEXT COMMENT 'home plate umpire name',
  UMP_1B_NAME_TX TEXT COMMENT 'first base umpire name',
  UMP_2B_NAME_TX TEXT COMMENT 'second base umpire name',
  UMP_3B_NAME_TX TEXT COMMENT 'third base umpire name',
  UMP_LF_NAME_TX TEXT COMMENT 'left field umpire name',
  UMP_RF_NAME_TX TEXT COMMENT 'right field umpire name',
  AWAY_MANAGER_ID CHAR(8) COMMENT 'visitors manager ID',
  AWAY_MANAGER_NAME_TX TEXT COMMENT 'visitors manager name',
  HOME_MANAGER_ID CHAR(8) COMMENT 'home manager ID',
  HOME_MANAGER_NAME_TX TEXT COMMENT 'home manager name',
  WIN_PIT_NAME_TX TEXT COMMENT 'winning pitcher name',
  LOSE_PIT_NAME_TX TEXT COMMENT 'losing pitcher name',
  SAVE_PIT_NAME_TX TEXT COMMENT 'save pitcher name',
  GOAHEAD_RBI_ID CHAR(8) COMMENT 'batter with goahead RBI ID',
  GOAHEAD_RBI_NAME_TX TEXT COMMENT 'batter with goahead RBI',
  AWAY_LINEUP1_BAT_NAME_TX TEXT COMMENT 'visitor batter 1 name',
  AWAY_LINEUP2_BAT_NAME_TX TEXT COMMENT 'visitor batter 2 name',
  AWAY_LINEUP3_BAT_NAME_TX TEXT COMMENT 'visitor batter 3 name',
  AWAY_LINEUP4_BAT_NAME_TX TEXT COMMENT 'visitor batter 4 name',
  AWAY_LINEUP5_BAT_NAME_TX TEXT COMMENT 'visitor batter 5 name',
  AWAY_LINEUP6_BAT_NAME_TX TEXT COMMENT 'visitor batter 6 name',
  AWAY_LINEUP7_BAT_NAME_TX TEXT COMMENT 'visitor batter 7 name',
  AWAY_LINEUP8_BAT_NAME_TX TEXT COMMENT 'visitor batter 8 name',
  AWAY_LINEUP9_BAT_NAME_TX TEXT COMMENT 'visitor batter 9 name',
  HOME_LINEUP1_BAT_NAME_TX TEXT COMMENT 'home batter 1 name',
  HOME_LINEUP2_BAT_NAME_TX TEXT COMMENT 'home batter 2 name',
  HOME_LINEUP3_BAT_NAME_TX TEXT COMMENT 'home batter 3 name',
  HOME_LINEUP4_BAT_NAME_TX TEXT COMMENT 'home batter 4 name',
  HOME_LINEUP5_BAT_NAME_TX TEXT COMMENT 'home batter 5 name',
  HOME_LINEUP6_BAT_NAME_TX TEXT COMMENT 'home batter 6 name',
  HOME_LINEUP7_BAT_NAME_TX TEXT COMMENT 'home batter 7 name',
  HOME_LINEUP8_BAT_NAME_TX TEXT COMMENT 'home batter 8 name',
  HOME_LINEUP9_BAT_NAME_TX TEXT COMMENT 'home batter 9 name',
  ADD_INFO_TX TEXT COMMENT 'additional information',
  ACQ_INFO_TX TEXT COMMENT 'acquisition information',
  TYPE TEXT COMMENT 'REG if regular season, AS if all star game, otherwise playoff series code',
  PRIMARY KEY (GAME_ID)
) COMMENT \'Represents a game, as generated by Chadwick from Retrosheet data\';
"""

_sub = """
CREATE TABLE IF NOT EXISTS sub (
  sub_key INT AUTO_INCREMENT,
  GAME_ID CHAR(12) COMMENT 'Game ID',
  INN_CT TINYINT COMMENT 'Inning',
  BAT_HOME_ID TINYINT COMMENT 'Batting team',
  SUB_ID CHAR(8) COMMENT 'Substitute',
  SUB_HOME_ID TINYINT COMMENT 'Team',
  SUB_LINEUP_ID TINYINT COMMENT 'Lineup position',
  SUB_FLD_CD TINYINT COMMENT 'Fielding position',
  REMOVED_ID CHAR(8) COMMENT 'Removed player',
  REMOVED_FLD_CD TINYINT COMMENT 'Position of removed player',
  EVENT_ID SMALLINT COMMENT 'Event number',
  PRIMARY KEY (sub_key)
) COMMENT \'Represents a substitution in a game, as generated by Chadwick from Retrosheet data\';
"""

schemas = [_pitch, _person, _event, _game, _sub]
