"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const k8s_manifest_1 = require("./k8s-manifest");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_AwsAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AwsAuth);
            }
            throw error;
        }
        this.stack = core_1.Stack.of(this);
        new k8s_manifest_1.KubernetesManifest(this, 'manifest', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_AwsAuthMapping(mapping);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRoleMapping);
            }
            throw error;
        }
        this.assertSameStack(role);
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_AwsAuthMapping(mapping);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addUserMapping);
            }
            throw error;
        }
        this.assertSameStack(user);
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    assertSameStack(construct) {
        const thisStack = core_1.Stack.of(this);
        if (core_1.Stack.of(construct) !== thisStack) {
            // aws-auth is always part of the cluster stack, and since resources commonly take
            // a dependency on the cluster, allowing those resources to be in a different stack,
            // will create a circular dependency. granted, it won't always be the case,
            // but we opted for the more causious and restrictive approach for now.
            throw new Error(`${construct.node.path} should be defined in the scope of the ${thisStack.stackName} stack to prevent circular dependencies`);
        }
    }
    synthesizeMapRoles() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => ({
                rolearn: m.role.roleArn,
                username: m.mapping.username ?? m.role.roleArn,
                groups: m.mapping.groups,
            }))),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => ({
                userarn: m.user.userArn,
                username: m.mapping.username ?? m.user.userArn,
                groups: m.mapping.groups,
            }))),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
_a = JSII_RTTI_SYMBOL_1;
AwsAuth[_a] = { fqn: "@aws-cdk/aws-eks.AwsAuth", version: "1.169.0" };
//# sourceMappingURL=data:application/json;base64,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