"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const assert_1 = require("@aws-cdk/assert");
const mediastore = require("@aws-cdk/aws-mediastore");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const lib_1 = require("../lib");
// --------------------------------------------------------------
// Test the default deployment snapshot
// --------------------------------------------------------------
test('Test the default deployment snapshot', () => {
    // Initial setup
    const stack = new core_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {});
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test the default deployment pattern variables
// --------------------------------------------------------------
test('Test the default deployment pattern variables', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {});
    // Assertion
    expect(cloudFrontToMediaStore.cloudFrontWebDistribution).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.mediaStoreContainer).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontLoggingBucket).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginRequestPolicy).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.edgeLambdaFunctionVersion).not.toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment without HTTP security headers
// --------------------------------------------------------------
test('Test the deployment without HTTP security headers', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        insertHttpSecurityHeaders: false
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(cloudFrontToMediaStore.edgeLambdaFunctionVersion).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with existing MediaStore container
// --------------------------------------------------------------
test('Test the deployment with existing MediaStore container', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MyMediaStoreContainer', {
        containerName: 'MyMediaStoreContainer'
    });
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        existingMediaStoreContainerObj: mediaStoreContainer
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        LambdaFunctionARN: {
                            Ref: 'testcloudfrontmediastoreSetHttpSecurityHeadersVersionE87B65C3'
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MyMediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer'
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided MediaStore properties
// --------------------------------------------------------------
test('Test the deployment with the user provided MediaStore properties', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        mediaStoreContainerProps: {
            containerName: 'MyMediaStoreContainer',
            policy: '{}',
            lifecyclePolicy: '{}',
            corsPolicy: [],
            metricPolicy: {
                containerLevelMetrics: 'DISABLED'
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        LambdaFunctionARN: {
                            Ref: 'testcloudfrontmediastoreSetHttpSecurityHeadersVersionE87B65C3'
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer',
        Policy: '{}',
        LifecyclePolicy: '{}',
        CorsPolicy: [],
        MetricPolicy: {
            ContainerLevelMetrics: 'DISABLED'
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided CloudFront properties
// --------------------------------------------------------------
test('Test the deployment with the user provided CloudFront properties', () => {
    // Initial setup
    const stack = new core_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        cloudFrontDistributionProps: {
            defaultBehavior: {
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.HTTPS_ONLY,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD'
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        LambdaFunctionARN: {
                            Ref: 'testcloudfrontmediastoreSetHttpSecurityHeadersVersionE87B65C3'
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'https-only'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        AccessLoggingEnabled: true,
        ContainerName: {
            Ref: 'AWS::StackName'
        },
        CorsPolicy: [
            {
                AllowedHeaders: ['*'],
                AllowedMethods: ['GET'],
                AllowedOrigins: ['*'],
                ExposeHeaders: ["*"],
                MaxAgeSeconds: 3000
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
        CloudFrontOriginAccessIdentityConfig: {
            Comment: {
                'Fn::Join': [
                    '',
                    [
                        'access-identity-',
                        {
                            Ref: 'AWS::Region',
                        },
                        '-',
                        {
                            Ref: 'AWS::StackName',
                        }
                    ]
                ]
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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