"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Utils = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const assertions_1 = require("aws-cdk-lib/assertions");
const cdk_nag_1 = require("cdk-nag");
const _ = require("lodash");
/**
 * A utility class that have common functions.
 */
class Utils {
    /**
     * Will append the suffix to the given name if the name do not contain the suffix
     * @param name - a string
     * @param suffixes - the string to append
     * @returns the name with the suffix appended if necessary delimited by a hyphen
     */
    static appendIfNecessary(name, ...suffixes) {
        let newName = name;
        suffixes.forEach(suffix => {
            newName = _.includes(newName, suffix) ? newName : `${newName}-${suffix}`;
        });
        return newName;
    }
    /**
     * Will wrap the given string using the given delimiter.
     * @param str - the string to wrap
     * @param delimiter - the delimiter to use
     * @returns the wrapped string
     */
    static wrap(str, delimiter) {
        let newStr = str;
        if (!Utils.startsWith(str, delimiter))
            newStr = `${delimiter}${newStr}`;
        if (!Utils.endsWith(str, delimiter))
            newStr = `${newStr}${delimiter}`;
        return newStr;
    }
    /**
     * Will check if the given string starts with the given prefix.
     * @param str - a string
     * @param s - the prefix to check
     */
    static startsWith(str, s) {
        return _.startsWith(str, s);
    }
    /**
     * Will check if the given string ends with the given suffix.
     * @param str - a string
     * @param s - suffix to check
     */
    static endsWith(str, s) {
        return _.endsWith(str, s);
    }
    /**
     * Will check if the given object is empty
     * @param value
     */
    static isEmpty(value) {
        return _.isEmpty(value);
    }
    /**
     * Will convert the given string to lower case and transform any spaces to hyphens
     * @param str - a string
     */
    static kebabCase(str) {
        return _.kebabCase(_.toLower(str));
    }
    /**
     * Splits a given Github URL and extracts the owner and repo name
     * @param url
     */
    static parseGithubUrl(url) {
        if (Utils.isEmpty(url))
            throw new TypeError('Invalid URL');
        if (!(Utils.startsWith(url, 'https://github.cms.gov/') || Utils.startsWith(url, 'https://github.com')))
            throw new TypeError('Invalid URL');
        if (!Utils.endsWith(url, '.git'))
            throw new TypeError('Invalid URL');
        // find the details from url
        let u = new URL(url);
        let owner = u.pathname.split('/')[1];
        let repo = _.replace(u.pathname.split('/')[2], '.git', '');
        let enterprise = u.hostname == 'github.cms.gov';
        let github = u.hostname == 'github.com';
        if (_.isEmpty(owner) || _.isEmpty(repo))
            throw new TypeError('Invalid URL');
        return {
            owner,
            repo,
            github,
            enterprise,
        };
    }
    /**
     * A utility function that will print the content of a CDK stack.
     * @warning This function is only used for debugging purpose.
     * @param stack - a valid stack
     */
    static prettyPrintStack(stack, persist = true, path = `tmp/${stack.stackName}.json`) {
        let t = assertions_1.Template.fromStack(stack);
        console.log(JSON.stringify(t.toJSON(), null, 2));
        if (persist) {
            fs.writeFileSync(path, JSON.stringify(t.toJSON(), null, 2));
        }
    }
    /**
     * Will disable the CDK NAG rule for the given construct and its children.
     * @param scope - the scope to disable the rule for
     * @param ruleId - the rule id to disable
     * @param reason - reason for disabling the rule.
     */
    static suppressNagRule(scope, ruleId, reason) {
        cdk_nag_1.NagSuppressions.addResourceSuppressions(scope, [{
                id: ruleId,
                reason: reason || `${ruleId} is not needed in this context (${scope.node.id}).`,
            }], true);
    }
}
exports.Utils = Utils;
_a = JSII_RTTI_SYMBOL_1;
Utils[_a] = { fqn: "ez-constructs.Utils", version: "0.0.22" };
//# sourceMappingURL=data:application/json;base64,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