"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleCodebuildProject = exports.GitEvent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const ez_construct_1 = require("../ez-construct");
const utils_1 = require("../lib/utils");
const secure_bucket_1 = require("../secure-bucket");
/**
 * The Github events which should trigger this build.
 */
var GitEvent;
(function (GitEvent) {
    GitEvent["PULL_REQUEST"] = "pull_request";
    GitEvent["PUSH"] = "push";
    GitEvent["ALL"] = "all";
})(GitEvent = exports.GitEvent || (exports.GitEvent = {}));
/**
 * Most of the cases,a developer will use CodeBuild setup to perform simple CI tasks such as:
 * - Build and test your code on a PR
 * - Run a specific script based on a cron schedule.
 * Also, they might want:
 * - artifacts like testcase reports to be available via Reports UI and/or S3.
 * - logs to be available via CloudWatch Logs.
 *
 * However, there can be additional organizational retention policies, for example retaining logs for a particular period of time.
 * With this construct, you can easily create a basic CodeBuild project with many opinated defaults that are compliant with FISMA and NIST.
 *
 * Example, creates a project named `my-project`, with artifacts going to my-project-artifacts-<accountId>-<region>
 *  and logs going to `/aws/codebuild/my-project` log group with a retention period of 90 days and 14 months respectively.
 *
 * ```ts
 *    new SimpleCodebuildProject(stack, 'MyProject')
 *      .projectName('myproject')
 *      .gitRepoUrl('https://github.com/bijujoseph/cloudbiolinux.git')
 *      .gitBaseBranch('main')
 *      .triggerEvent(GitEvent.PULL_REQUEST)
 *      .buildSpecPath('buildspecs/my-pr-checker.yml')
 *      .assemble();
 * ```
 *
 */
class SimpleCodebuildProject extends ez_construct_1.EzConstruct {
    constructor(scope, id) {
        super(scope, id);
        this._gitBaseBranch = 'develop';
        this._grantReportGroupPermissions = true;
        this._privileged = false;
        this._computType = aws_codebuild_1.ComputeType.MEDIUM;
        this._buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_5_0;
        this._envVariables = {};
        this.scope = scope;
        this.id = id;
    }
    /**
     * The underlying codebuild project that is created by this construct.
     */
    get project() {
        return this._project;
    }
    /**
     * A convenient way to set the project environment variables.
     * The values set here will be presnted on the UI when build with overriding is used.
     * @param name - The environment variable name
     * @param envVar - The environment variable value
     *Example:
     *
     * ```ts
     *  project
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'abcd})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: '/dev/thatkey, type: BuildEnvironmentVariableType.PARAMETER_STORE})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'arn:of:secret, type: BuildEnvironmentVariableType.SECRETS_MANAGER});
     * ```
     */
    addEnv(name, envVar) {
        this._envVariables[name] = envVar;
        return this;
    }
    /**
     * The name of the codebuild project
     * @param projectName - a valid name string
     */
    projectName(projectName) {
        this._projectName = projectName;
        return this;
    }
    /**
     * The description of the codebuild project
     * @param projectDescription - a valid description string
     */
    projectDescription(projectDescription) {
        this._projectDescription = projectDescription;
        return this;
    }
    /**
     * The compute type to use
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html
     * @param computeType
     */
    computeType(computeType) {
        this._computType = computeType;
        return this;
    }
    /**
     * The build image to use
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @see https://docs.aws.amazon.com/cdk/api/v1/docs/@aws-cdk_aws-codebuild.IBuildImage.html
     * @param buildImage
     */
    buildImage(buildImage) {
        this._buildImage = buildImage;
        return this;
    }
    /**
     * The build image to use
     * @param ecrRepoName - the ecr repository name
     * @param imageTag - the image tag
     */
    ecrBuildImage(ecrRepoName, imageTag) {
        let repo = aws_ecr_1.Repository.fromRepositoryName(this.scope, 'ClaimsCodeBuildBaseImageRepository', ecrRepoName);
        return this.buildImage(aws_codebuild_1.LinuxBuildImage.fromEcrRepository(repo, imageTag));
    }
    /**
     * The vpc network interfaces to add to the codebuild
     * @see https://docs.aws.amazon.com/cdk/api/v1/docs/aws-codebuild-readme.html#definition-of-vpc-configuration-in-codebuild-project
     * @param vpcId
     */
    inVpc(vpcId) {
        this._vpcId = vpcId;
        return this;
    }
    /**
     * Set privileged mode of execution. Usually needed if this project builds Docker images,
     * and the build environment image you chose is not provided by CodeBuild with Docker support.
     * By default, Docker containers do not allow access to any devices.
     * Privileged mode grants a build project's Docker container access to all devices
     * https://docs.aws.amazon.com/codebuild/latest/userguide/change-project-console.html#change-project-console-environment
     * @param p - true/false
     */
    privileged(p) {
        this._privileged = p;
        return this;
    }
    /**
     * The build spec file path
     * @param buildSpecPath - relative location of the build spec file
     */
    buildSpecPath(buildSpecPath) {
        this._buildSpecPath = buildSpecPath;
        return this;
    }
    /**
     * The github or enterprise github repository url
     * @param gitRepoUrl
     */
    gitRepoUrl(gitRepoUrl) {
        this._gitRepoUrl = gitRepoUrl;
        return this;
    }
    /**
     * The main branch of the github project.
     * @param branch
     */
    gitBaseBranch(branch) {
        this._gitBaseBranch = branch;
        return this;
    }
    /**
     * The Github events that can trigger this build.
     * @param event
     */
    triggerBuildOnGitEvent(event) {
        this._triggerOnGitEvent = event;
        return this;
    }
    /**
     * The cron schedule on which this build gets triggerd.
     * @param schedule
     */
    triggerBuildOnSchedule(schedule) {
        this._triggerOnSchedule = schedule;
        return this;
    }
    /**
     * The name of the bucket to store the artifacts.
     * By default the buckets will get stored in `<project-name>-artifacts` bucket.
     * This function can be used to ovrride the default behavior.
     * @param artifactBucket - a valid existing Bucket reference or bucket name to use.
     */
    artifactBucket(artifactBucket) {
        this._artifactBucket = artifactBucket;
        return this;
    }
    overrideProjectProps(props) {
        var _b;
        let projectName = this._projectName ? this._projectName : utils_1.Utils.kebabCase(this.id);
        let description = this._projectDescription ? this._projectDescription : `Codebuild description for ${projectName}`;
        let defaults = {
            projectName,
            description,
            grantReportGroupPermissions: this._grantReportGroupPermissions,
        };
        // set the default environment if not provided.
        if (utils_1.Utils.isEmpty(props.environment)) {
            // @ts-ignore
            defaults.environment = {
                buildImage: this._buildImage ? this._buildImage : aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
                privileged: this._privileged,
                computeType: this._computType,
                environmentVariables: this._envVariables,
            };
        }
        // add default logging if not provided
        if (utils_1.Utils.isEmpty(props.logging)) {
            // @ts-ignore
            defaults.logging = {
                cloudWatch: {
                    logGroup: new aws_logs_1.LogGroup(this, 'ProjectLogGroup', {
                        logGroupName: `/aws/codebuild/${utils_1.Utils.kebabCase(projectName)}`,
                        retention: aws_logs_1.RetentionDays.THIRTEEN_MONTHS,
                    }),
                },
            };
        }
        // create source if not provided in props
        if (utils_1.Utils.isEmpty(props.source)) {
            // @ts-ignore
            defaults.source = this.createSource(this._gitRepoUrl, this._gitBaseBranch, this._triggerOnGitEvent);
        }
        // create artifact bucket if one was not provided
        if (utils_1.Utils.isEmpty(props.artifacts)) {
            // @ts-ignore
            defaults.artifacts = this.createArtifacts((_b = this._artifactBucket) !== null && _b !== void 0 ? _b : `${this._projectName}-artifacts`);
        }
        // create the build spec if one was not provided
        if (utils_1.Utils.isEmpty(props.buildSpec) && !utils_1.Utils.isEmpty(this._buildSpecPath)) {
            // @ts-ignore
            defaults.buildSpec = this.createBuildSpec(this._buildSpecPath);
        }
        // create vpc interface if needed
        if (!utils_1.Utils.isEmpty(this._vpcId)) {
            // @ts-ignore
            defaults.vpc = aws_ec2_1.Vpc.fromLookup(this, 'VPC', { vpcId: this._vpcId });
        }
        this._props = Object.assign({}, defaults, props);
        return this;
    }
    assemble(defaultProps) {
        // create the default project properties
        this.overrideProjectProps(defaultProps !== null && defaultProps !== void 0 ? defaultProps : {});
        // create a codebuild project
        let project = new aws_codebuild_1.Project(this.scope, 'Project', this._props);
        // run above project on a schedule ?
        if (this._triggerOnSchedule) {
            new aws_events_1.Rule(this.scope, 'ScheduleRule', {
                schedule: this._triggerOnSchedule,
                targets: [new aws_events_targets_1.CodeBuildProject(project)],
            });
        }
        this._project = project;
        utils_1.Utils.suppressNagRule(this._project, 'AwsSolutions-CB4', 'Artifacts produced by this project are encrypted using `aws/s3` key, and also at rest by S3.');
        utils_1.Utils.suppressNagRule(this._project, 'AwsSolutions-IAM5', 'There is an artifact bucket per project. The wild card IAM policies are scoped to this project and only used to manage artifacts produced by the project.');
        return this;
    }
    /**
     * Create an S3 bucket for storing artifacts produced by the codebuild project
     * @param bucketName - the s3 bucket
     * @private
     */
    createBucket(bucketName) {
        return new secure_bucket_1.SecureBucket(this, 'ProjectArtifactBucket')
            .bucketName(bucketName)
            .objectsExpireInDays(90)
            .assemble()
            .bucket;
    }
    /**
     * Creates an S3 artifact store for storing the objects produced by the codebuild project
     * @param artifactBucket - a bucket object or bucket name
     * @private
     */
    createArtifacts(artifactBucket) {
        let theBucket = (typeof artifactBucket === 'string') ? this.createBucket(artifactBucket) : artifactBucket;
        return aws_codebuild_1.Artifacts.s3({
            bucket: theBucket,
            includeBuildId: true,
            packageZip: true,
        });
    }
    /**
     * Creates a Github or Enterprise Githb repo source object
     * @param repoUrl - the url of the repo
     * @param base - the main or base branch used by the repo
     * @param gitEvent - the github events that can trigger builds
     * @private
     */
    createSource(repoUrl, base, gitEvent) {
        let webhook = gitEvent && true;
        let repoDetails = utils_1.Utils.parseGithubUrl(repoUrl);
        let webhookFilter = this.createWebHookFilters(base, gitEvent);
        if (repoDetails.enterprise == true) {
            return aws_codebuild_1.Source.gitHubEnterprise({
                httpsCloneUrl: repoUrl,
                webhook,
                webhookFilters: webhookFilter,
            });
        }
        return aws_codebuild_1.Source.gitHub({
            owner: repoDetails.owner,
            repo: repoDetails.repo,
            webhook,
            webhookFilters: webhookFilter,
        });
    }
    /**
     * Creates a webhook filter object
     * @param base - the base branch
     * @param gitEvent - the github event
     * @private
     */
    createWebHookFilters(base, gitEvent) {
        // @ts-ignore
        let fg = null;
        if (!gitEvent)
            return undefined;
        if (gitEvent == GitEvent.PULL_REQUEST) {
            fg = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PULL_REQUEST_CREATED, aws_codebuild_1.EventAction.PULL_REQUEST_UPDATED, aws_codebuild_1.EventAction.PULL_REQUEST_REOPENED);
            if (base) {
                fg = fg.andBaseBranchIs(base);
            }
        }
        if (gitEvent == GitEvent.PUSH) {
            fg = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PUSH);
        }
        if (gitEvent == GitEvent.ALL) {
            fg = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PUSH, aws_codebuild_1.EventAction.PULL_REQUEST_CREATED, aws_codebuild_1.EventAction.PULL_REQUEST_UPDATED, aws_codebuild_1.EventAction.PULL_REQUEST_REOPENED, aws_codebuild_1.EventAction.PULL_REQUEST_MERGED);
        }
        return [fg];
    }
    /**
     * Loads the build spec associated with the given codebuild project
     * @param buildSpecPath - location of the build spec file.
     * @private
     */
    createBuildSpec(buildSpecPath) {
        return aws_codebuild_1.BuildSpec.fromSourceFilename(buildSpecPath);
    }
}
exports.SimpleCodebuildProject = SimpleCodebuildProject;
_a = JSII_RTTI_SYMBOL_1;
SimpleCodebuildProject[_a] = { fqn: "ez-constructs.SimpleCodebuildProject", version: "0.0.22" };
//# sourceMappingURL=data:application/json;base64,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