"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionsBoundaryAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * As a best practice organizations enforce policies which require all custom IAM Roles created to be defined under
 * a specific path and permission boundary. Well, this allows better governance and also prevents unintended privilege escalation.
 * AWS CDK high level constructs and patterns encapsulates the role creation from end users.
 * So it is a laborious and at times impossible to get a handle of newly created roles within a stack.
 * This aspect will scan all roles within the given scope and will attach the right permission boundary and path to them.
 * Example:
 * ```ts
 *    const app = new App();
 *    const mystack = new MyStack(app, 'MyConstruct'); // assuming this will create a role by name `myCodeBuildRole` with admin access.
 *    Aspects.of(app).add(new PermissionsBoundaryAspect('/my/devroles/', 'boundary/dev-max'));
 * ```
 */
class PermissionsBoundaryAspect {
    /**
     * Constructs a new PermissionsBoundaryAspect.
     * @param rolePath - the role path to attach to newly created roles.
     * @param rolePermissionBoundary - the permission boundary to attach to newly created roles.
     */
    constructor(rolePath, rolePermissionBoundary) {
        this.rolePath = rolePath;
        this.rolePermissionBoundary = rolePermissionBoundary;
    }
    visit(node) {
        if (node instanceof aws_iam_1.Role) {
            const stack = aws_cdk_lib_1.Stack.of(node);
            const roleResource = node.node.findChild('Resource');
            // set the path if it is provided
            if (this.rolePath) {
                roleResource.addPropertyOverride('Path', this.rolePath);
            }
            // set the permission boundary if it is provided
            if (this.rolePermissionBoundary && !this.rolePermissionBoundary.startsWith('arn:aws:iam')) {
                roleResource.addPropertyOverride('PermissionsBoundary', `arn:aws:iam::${stack.account}:policy/${this.rolePermissionBoundary}`);
            }
        }
    }
}
exports.PermissionsBoundaryAspect = PermissionsBoundaryAspect;
_a = JSII_RTTI_SYMBOL_1;
PermissionsBoundaryAspect[_a] = { fqn: "ez-constructs.PermissionsBoundaryAspect", version: "0.0.22" };
//# sourceMappingURL=data:application/json;base64,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