# Copyright 2019 DeepMind Technologies Limited.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ============================================================================

"""Text mazes generated by DeepMind Lab's maze generator."""

import abc
from labmaze import defaults


class BaseMaze(metaclass=abc.ABCMeta):
  """A abstract base class for DeepMind Lab-style text mazes."""

  def regenerate(self):
    """Regenerates the maze if required."""
    pass

  @abc.abstractproperty
  def entity_layer(self):
    """The entity layer of the current maze.

    The entity layer defines the actual structure of the maze, i.e. which cells
    are vacant and which are occupied by walls. For further explanation, see:
    https://github.com/deepmind/lab/blob/master/docs/developers/creating_levels/text_level.md#text-level-format

    Returns:
      A 2-dimensional NumPy array of printable characters.
    """

  @abc.abstractproperty
  def variations_layer(self):
    """The variations layer of the current maze.

    The variations layer defines the variations in decorative aspects of each
    cell in the maze. Each variation is assigned a different character (DeepMind
    Lab uses '.' for default styling, and A-Z for variations) For further
    explanation, see:
    https://github.com/deepmind/lab/blob/master/docs/developers/creating_levels/text_level.md.md#text-level-format

    Returns:
      A 2-dimensional NumPy array of printable characters.
    """

  @abc.abstractproperty
  def height(self):
    """The number of cells along the y-direction of this maze."""

  @abc.abstractproperty
  def width(self):
    """The number of cells along the x-direction of this maze."""

  @property
  def max_variations(self):
    """The maximum number of variations in the variations layer of this maze."""

  @property
  def spawn_token(self):
    return defaults.SPAWN_TOKEN

  @property
  def object_token(self):
    return defaults.OBJECT_TOKEN
