"""

"""

#  Copyright (c) 2022-2022 Theodor Möser
#  .
#  Licensed under the EUPL-1.2-or-later (the "Licence");
#  .
#  You may not use this work except in compliance with the Licence.
#  You may obtain a copy of the Licence at:
#  .
#  https://joinup.ec.europa.eu/software/page/eupl
#  .
#  Unless required by applicable law or agreed to in writing,
#  software distributed under the Licence is distributed on an "AS IS" basis,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  .
#  See the Licence for the specific language governing
#  permissions and limitations under the Licence.

from formgram.formgram_machines.turing_machines import subroutines
from formgram.formgram_machines.turing_machines.transformations import to_grammar_transformable_form
from formgram.formgram_procedures.utility.unrestricted.helper import find_new_nonterminal


def from_grammar(grammar: dict) -> dict:
    """Create a nondeterministic turing machine accepting the words generated by given grammar

    The basic idea for the created turing machine is to apply the grammar productions backwards.
    If after enough backwards applications the tape is empty save for a lone starting symbol,
    the machine accepts.

    The main loop of the machine is:
     #. check if machine is done
     #. move to a random place on the tape
     #. apply random production

     The randomness as well as the check rely heavily on the nondeterminism of the machine.
     The check especially is implemented by allowing the machine to either skip the check,
     or be forced to confirm that the tape is in the correct state for halting in accepting state.

    :param grammar:
    :return:
    """
    blank_symbol = find_new_nonterminal(grammar, base_symbol=".")

    move_to_left_blank = find_new_nonterminal(grammar, base_symbol="check_if_done_move_left")
    move_to_first_symbol = find_new_nonterminal(grammar, base_symbol="ckeck_if_done_move_right")
    check_if_done = find_new_nonterminal(grammar, base_symbol="check_if_done")

    select_random_position = find_new_nonterminal(grammar, base_symbol="select_random_position")
    apply_random_production = find_new_nonterminal(grammar, base_symbol="apply_random_production")
    accepting = find_new_nonterminal(grammar, base_symbol="accepting")

    machine = {
        "alphabet": grammar["terminals"],
        "control_symbols": grammar["nonterminals"] | {blank_symbol},
        "states": {move_to_left_blank, move_to_first_symbol, check_if_done,
                   select_random_position, apply_random_production},
        "initial_state": move_to_left_blank,
        "accepting_states": {accepting},
        "blank_symbol": blank_symbol,
        "transitions": {}  # will be added with subroutines
    }

    machine = subroutines.add_move_to_matching_symbol_subroutine(machine=machine,
                                                                 start_state=move_to_left_blank,
                                                                 stop_state=move_to_first_symbol,
                                                                 matching_symbols={blank_symbol},
                                                                 direction="L")

    machine = subroutines.add_move_n_steps_subroutine(machine=machine,
                                                      start_state=move_to_first_symbol,
                                                      stop_state=check_if_done,
                                                      steps=1,
                                                      direction="R"
                                                      )

    machine = subroutines.add_confirm_read_string(machine=machine,
                                                  start_state=check_if_done,
                                                  stop_state=accepting,
                                                  to_confirm=(grammar["starting_symbol"], blank_symbol))

    machine = subroutines.add_move_to_random_place_subroutine(machine=machine,
                                                              start_state=check_if_done,
                                                              stop_state=apply_random_production
                                                              )

    for production in grammar["productions"]:
        left_hand_side, right_hand_side = production
        machine = subroutines.add_replace_string_subroutine(machine=machine,
                                                            start_state=apply_random_production,
                                                            stop_state=move_to_left_blank,
                                                            to_replace=right_hand_side,
                                                            to_write=left_hand_side)

    return machine


def to_grammar(machine: dict) -> dict:
    """Create a grammar from turing machine which generates the accepted words

    The basic idea of this algorithm is to simulate the machine in reverse.
    For this a notation of instantaneous descriptions of the machine is used,
    where the tape is recorded with the current state inserted right after the
    current head position.

    E.g. with a tape `abcdefg` and state `Q` at position `3` (starting with 0) would
    result in the instantaneous description
    `abcdQefg`

    First the machine is transformed such that:

    #. The finishing tape is has a single "1" written on it
    #. The starting state is never used besides starting the machine
    #. No blank symbols are written except for the right to left wiping of the tape at the end


    This gives the grammar the guarantee that

    #. If a starting state is encountered, the tape must contain exactly the accepted word
    #. The accepting instantaneous description of any word is well known
    #. The ends of the tape are easily detectable as the only places using blanks
    #. The head never leaves the written portion of the tape for more then one cell

    The grammar can thus use limited information in the form of a partial word
    `rQx` to simulate the machine.
    The partial word `rQx` states that the machine is in state `Q` reads symbol
    `r` and the following cell of the read symbol is `x`.

    To allow the grammar to detect the ends of tapes, additional blanks at the
    end of the tapes are also saved.
    E.g above `abcdQefg` would be saved as the word `.abcdefQefg..` with `.` as
    the blank symbol.
    This allows `rQx` to "read" past the right end of the tape via `r=.` while
    still having a symbol to "read" for `x`.
    Note that above rules prohibit states like `.a..Q` but allows `.a.Q.`.

    :param machine:
    :return:
    """
    machine = to_grammar_transformable_form(machine)
    for key, value in machine.items():
        if key == "transitions":
            print(f"{key}:")
            for line in sorted(value):
                print(line)
        else:
            print(f"{key}: {value}")
    # Set up the grammar, the productions will be filled below
    productions = set()
    grammar = {
        "nonterminals": machine["states"] | machine["control_symbols"],
        "terminals": machine["alphabet"],
        "productions": productions
    }
    starting_symbol = find_new_nonterminal(grammar, "start")
    grammar["nonterminals"].add(starting_symbol)
    grammar["starting_symbol"] = starting_symbol

    # Name some of the more frequently used sets
    blank_symbol = machine["blank_symbol"]
    tape_symbols = machine["control_symbols"] | machine["alphabet"]

    # Add the easy productions
    sole_accepting_state = list(machine["accepting_states"])[0]
    productions.update({
        # starting configuration of the grammar is the accepting configuration of the machine
        ((grammar["starting_symbol"],), (blank_symbol, "1", blank_symbol, sole_accepting_state, blank_symbol)),

        # following is the "empty word" accepted special case
        ((blank_symbol, machine["initial_state"], blank_symbol, blank_symbol), ()),
    })
    # These productions mean, that the machine is reverted to a configuration where
    # the head is at the start of the tape in the initial state
    productions.update({((blank_symbol, symbol, machine["initial_state"]), (symbol,)) for symbol in grammar["terminals"]})
    productions.update({((blank_symbol, symbol), (symbol, )) for symbol in grammar["terminals"]})
    productions.update({((symbol, blank_symbol, blank_symbol), (symbol, )) for symbol in grammar["terminals"]})

    # Now for the actual transitions
    # Line comments assume unary alphabet: 1 for alphabet symbol, . for blank
    for transition in machine["transitions"]:
        (read_state, read_symbol), (write_state, write_symbol, head_movement_direction) = transition
        for edge_detection_symbol in tape_symbols:
            if head_movement_direction == "L":
                # Qwx ::= rQx
                left_hand_side = (write_state, write_symbol, edge_detection_symbol)
                if (read_symbol != blank_symbol) == (blank_symbol != write_symbol):
                    # Q.. ::= .Q. or Q11 ::= 1Q1 or Q1. ::= 1Q.
                    # theoretically Q.1 ::= .Q1 but that can't happen thus dead production
                    right_hand_side = (read_symbol, read_state, edge_detection_symbol)
                    if write_symbol == blank_symbol and edge_detection_symbol != blank_symbol:
                        continue  # dead production skip

                elif read_symbol == blank_symbol:
                    # Q1. ::= Q. or Q11 ::= Q1
                    if edge_detection_symbol == blank_symbol:
                        # right edge of tape, need to delete blank symbol on right hand side
                        right_hand_side = (read_symbol, read_state)
                    else:
                        # left edge of tape need to delete blank symbol on left hand side
                        left_hand_side = (blank_symbol, ) + left_hand_side
                        right_hand_side = (read_symbol, read_state, edge_detection_symbol)

                elif edge_detection_symbol == blank_symbol:
                    # Q.. := 1Q..
                    # reading a 1 and writing a . would result in loss of a blank on the "reading" side
                    # thus pad right hand side with a blank
                    right_hand_side = (read_symbol, read_state, edge_detection_symbol, blank_symbol)

                else:
                    # Q.1 ::= ???
                    # Head is beyond the tape and a new blank needs to be added to make tape legal again
                    # THis may never happen skip this edge_detection symbol
                    continue

            elif head_movement_direction == "R":
                # wxQ ::= rQx
                left_hand_side = (write_symbol, edge_detection_symbol, write_state)
                if (read_symbol != blank_symbol) == (blank_symbol != write_symbol):
                    # 11Q ::= 1Q1, 1.Q ::= 1Q. and .1Q ::= .Q1
                    # theoretically ..Q ::= .Q. which can't happen
                    right_hand_side = (read_symbol, read_state, edge_detection_symbol)

                elif read_symbol == blank_symbol:
                    # 1.Q ::= Q. and 11Q ::= Q1
                    # each omitting written r which would double the left blank
                    left_hand_side = (blank_symbol, ) + left_hand_side
                    right_hand_side = (read_symbol, read_state, edge_detection_symbol)

                else:  # write_symbol == blank_symbol:
                    # only "L" transitions may write the blank symbol
                    raise RuntimeError(
                        f"Can't transform this transition {transition}:"
                        f" writes blank symbol with direction {head_movement_direction}")

            else:  # head_movement_direction == "S":
                # wQx ::= rQx
                if (read_symbol != blank_symbol) == (blank_symbol != write_symbol):
                    # 1Q ::= 1Q or .Q ::= .Q
                    # leaving the edge detection as it is irrelevant here
                    left_hand_side = (write_symbol, write_state)
                    right_hand_side = (read_symbol, read_state)

                elif read_symbol == blank_symbol and edge_detection_symbol == blank_symbol:
                    # 1Q. ::= .Q
                    left_hand_side = (write_symbol, write_state, edge_detection_symbol)
                    right_hand_side = (read_symbol, read_state)

                elif read_symbol == blank_symbol:
                    # 1Q1 ::= Q1
                    left_hand_side = (blank_symbol, write_symbol, write_state, edge_detection_symbol)
                    right_hand_side = (read_symbol, read_state, edge_detection_symbol)

                elif edge_detection_symbol != blank_symbol:
                    # .Q1 ::= .1Q1
                    left_hand_side = (write_symbol, write_state, edge_detection_symbol)
                    right_hand_side = (blank_symbol, read_symbol, read_state, edge_detection_symbol)
                    productions.add((left_hand_side, right_hand_side))

                    # x.Q. ::= x1Q..
                    # abusing the edge_detection_symbol on the left hand side
                    left_hand_side = (edge_detection_symbol, write_symbol, write_state, blank_symbol)
                    right_hand_side = (edge_detection_symbol, read_symbol, read_state, blank_symbol, blank_symbol)

                else:
                    # .Q.. := .1Q..
                    left_hand_side = (write_symbol, write_state, edge_detection_symbol, blank_symbol)
                    right_hand_side = (blank_symbol, read_symbol, read_state, edge_detection_symbol, blank_symbol)
                    productions.add((left_hand_side, right_hand_side))

                    # ..Q. := ..1Q.
                    left_hand_side = (blank_symbol, write_symbol, write_state, edge_detection_symbol)
                    right_hand_side = (blank_symbol, blank_symbol, read_symbol, read_state, edge_detection_symbol)

            productions.add((left_hand_side, right_hand_side))

    return grammar
