# (c) Kevin Dunn, 2010-2021. MIT License. Based on own private work over the years.

# Built-in libraries
from typing import Dict
import plotly.graph_objects as go
from pydantic import BaseModel


def score_plot(
    model, pc_horiz=1, pc_vert=2, settings: Dict = None, fig=None
) -> go.Figure:
    """Generates a 2-dimensional score plot for the given latent variable model.

    Parameters
    ----------
    model : MVmodel object (PCA, or PLS)
        A latent variable model generated by this library.
    pc_horiz : int, optional
        Which component to plot on the horizontal axis, by default 1 (the first component)
    pc_vert : int, optional
        Which component to plot on the vertical axis, by default 2 (the second component)
    settings : dict
        Default settings are = {
            "show_ellipse": True [bool],
                Should the Hotelling's T2 ellipse be added

            "ellipse_conf_level": 0.95 [float]
                If the ellipse is added, which confidence level is used. A number < 1.00.

            "title": f"Score plot of component {pc_horiz} vs component {pc_vert}"
                Overall plot title

            "show_observation_labels": False,
                Adds a label for each observation. Labels are always available in the hover.

            "show_legend": True,
                Shows a clickable legend (allows to turn the ellipse(s) on/off)

            "html_image_height": 500,
                in pixels

            "html_aspect_ratio_w_over_h": 16/9,
                sets the image width, as a ratio of the height

        }
    """
    margin_dict: Dict = dict(l=10, r=10, b=5, t=80)  # Defaults: l=80, r=80, t=100, b=80

    class Settings(BaseModel):
        show_ellipse: bool = True
        ellipse_conf_level: float = 0.95  # TODO: check constraint
        title: str = f"Score plot of component {pc_horiz} vs component {pc_vert}"
        show_observation_labels: bool = False  # TODO
        show_legend: bool = True
        html_image_height: float = 500.0
        html_aspect_ratio_w_over_h: float = 16 / 9.0

    if settings:
        setdict = Settings(settings).dict()
    else:
        setdict = Settings().dict()
    if fig is None:
        fig = go.Figure()

    fig = model.x_scores.plot.scatter(x=pc_horiz, y=pc_vert, text=model.x_scores.index)
    fig.update_traces(textposition="top center")
    ellipse = model.ellipse_coordinates(
        score_horiz=pc_horiz,
        score_vert=pc_vert,
        T2_limit_conf_level=setdict["ellipse_conf_level"],
    )
    fig.add_trace(go.Scatter(x=ellipse[0], y=ellipse[1], name="Hotelling's T^2 [95%]"))
    fig.update_layout(
        xaxis_title_text=f"PC {pc_horiz}", yaxis_title_text=f"PC {pc_vert}"
    )
    fig.add_hline(y=0)
    fig.add_vline(x=0)
    fig.update_layout(
        title_text=setdict["title"],
        margin=margin_dict,
        hovermode="closest",
        showlegend=setdict["show_legend"],
        legend=dict(
            orientation="h",
            traceorder="normal",
            font=dict(family="sans-serif", size=12, color="#000"),
            bordercolor="#DDDDDD",
            borderwidth=1,
        ),
        autosize=False,
        xaxis=dict(
            gridwidth=1,
            mirror=True,  # ticks are mirror at the top of the frame also
            showspikes=True,
            visible=True,
        ),
        yaxis=dict(
            gridwidth=2,
            type="linear",
            autorange=True,
            showspikes=True,
            visible=True,
            showline=True,  # show a separating line
            side="left",  # show on the RHS
        ),
        width=setdict["html_aspect_ratio_w_over_h"] * setdict["html_image_height"],
        height=setdict["html_image_height"],
    )
    return fig


def spe_plot(model, with_a=-1, settings: Dict = None, fig=None) -> go.Figure:
    """Generates a squared-prediction error (SPE) plot for the given latent variable model using
    `with_a` number of latent variables. The default will use the total number of latent variables
    which have already been fitted.

    Parameters
    ----------
    model : MVmodel object (PCA, or PLS)
        A latent variable model generated by this library.
    with_a : int, optional
        Uses this many number of latent variables, and therefore shows the SPE after this number of
        model components. By default the total number of components fitted will be used.
    settings : dict
        Default settings are = {
            "show_limit": True [bool],
                Should the SPE limit be plotted.

            "conf_level": 0.95 [float]
                If the limit line is added, which confidence level is used. Number < 1.00.

            "title": f"Squared prediction error plot after fitting {with_a} components"
                Overall plot title

            "show_observation_labels": False,
                Adds a label for each observation. Labels are always available in the hover.

            "show_legend": True,
                Shows a clickable legend (allows to turn the limit on/off)

            "html_image_height": 500,
                Image height, in pixels.

            "html_aspect_ratio_w_over_h": 16/9,
                Sets the image width, as a ratio of the height.

        }
    """
    margin_dict: Dict = dict(l=10, r=10, b=5, t=80)  # Defaults: l=80, r=80, t=100, b=80

    if with_a < 0:
        with_a = model.squared_prediction_error.columns[with_a]

    # TODO: check `with_a`: what should it plot if `with_a` is zero, or > A?
    # TODO: show a proper y-axis label

    class Settings(BaseModel):
        show_limit: bool = True
        conf_level: float = 0.95  # TODO: check constraint < 1
        title: str = f"Squared prediction error plot after fitting {with_a} components"
        show_observation_labels: bool = False  # TODO
        show_legend: bool = True
        html_image_height: float = 500.0
        html_aspect_ratio_w_over_h: float = 16 / 9.0

    if settings:
        setdict = Settings(settings).dict()
    else:
        setdict = Settings().dict()
    if fig is None:
        fig = go.Figure()

    fig = model.squared_prediction_error.loc[:, [with_a]].plot.scatter(
        x=model.squared_prediction_error.index,
        y=model.squared_prediction_error.columns[with_a - 1],
    )
    limit_SPE_conf_level = model.SPE_limit(conf_level=setdict["conf_level"])

    fig.add_hline(
        y=limit_SPE_conf_level,
        line_color="red",
        annotation_text=f'{setdict["conf_level"]*100:.3g}% limit',
        annotation_position="bottom right",
    )
    fig.update_layout(
        title_text=setdict["title"],
        margin=margin_dict,
        hovermode="closest",
        showlegend=setdict["show_legend"],
        legend=dict(
            orientation="h",
            traceorder="normal",
            font=dict(family="sans-serif", size=12, color="#000"),
            bordercolor="#DDDDDD",
            borderwidth=1,
        ),
        autosize=False,
        xaxis=dict(
            gridwidth=1,
            mirror=True,  # ticks are mirror at the top of the frame also
            showspikes=True,
            visible=True,
        ),
        yaxis=dict(
            title=setdict["title"],
            gridwidth=2,
            type="linear",
            autorange=True,
            showspikes=True,
            visible=True,
            showline=True,  # show a separating line
            side="left",  # show on the RHS
        ),
        width=setdict["html_aspect_ratio_w_over_h"] * setdict["html_image_height"],
        height=setdict["html_image_height"],
    )
    return fig


def t2_plot(model, with_a=-1, settings: Dict = None, fig=None) -> go.Figure:
    """Generates a Hotelling's T2 (T^2) plot for the given latent variable model using
    `with_a` number of latent variables. The default will use the total number of latent variables
    which have already been fitted.

    Parameters
    ----------
    model : MVmodel object (PCA, or PLS)
        A latent variable model generated by this library.
    with_a : int, optional
        Uses this many number of latent variables, and therefore shows the SPE after this number of
        model components. By default the total number of components fitted will be used.
    settings : dict
        Default settings are = {
            "show_limit": True [bool],
                Should the T2 limit be plotted.

            "conf_level": 0.95 [float]
                If the limit line is added, which confidence level is used. Number < 1.00.

            "title": f"Hotelling's T2 plot after fitting {with_a} components"
                Overall plot title

            "show_observation_labels": False,
                Adds a label for each observation. Labels are always available in the hover.

            "show_legend": True,
                Shows a clickable legend (allows to turn the limit on/off)

            "html_image_height": 500,
                Image height, in pixels.

            "html_aspect_ratio_w_over_h": 16/9,
                Sets the image width, as a ratio of the height.

        }
    """
    margin_dict: Dict = dict(l=10, r=10, b=5, t=80)  # Defaults: l=80, r=80, t=100, b=80

    if with_a < 0:
        with_a = model.Hotellings_T2.columns[with_a]

    # TODO: check `with_a`: what should it plot if `with_a` is zero, or > A?
    # TODO: show a proper y-axis label

    class Settings(BaseModel):
        show_limit: bool = True
        conf_level: float = 0.95  # TODO: check constraint < 1
        title: str = f"Hotelling's T2 plot after fitting {with_a} components"
        show_observation_labels: bool = False  # TODO
        show_legend: bool = True
        html_image_height: float = 500.0
        html_aspect_ratio_w_over_h: float = 16 / 9.0

    if settings:
        setdict = Settings(settings).dict()
    else:
        setdict = Settings().dict()
    if fig is None:
        fig = go.Figure()

    fig = model.Hotellings_T2.loc[:, [with_a]].plot.scatter(
        x=model.Hotellings_T2.index,
        y=model.Hotellings_T2.columns[with_a - 1],
    )
    limit_HT2_conf_level = model.T2_limit(conf_level=setdict["conf_level"])

    fig.add_hline(
        y=limit_HT2_conf_level,
        line_color="red",
        annotation_text=f'{setdict["conf_level"]*100:.3g}% limit',
        annotation_position="bottom right",
    )
    fig.update_layout(
        title_text=setdict["title"],
        margin=margin_dict,
        hovermode="closest",
        showlegend=setdict["show_legend"],
        legend=dict(
            orientation="h",
            traceorder="normal",
            font=dict(family="sans-serif", size=12, color="#000"),
            bordercolor="#DDDDDD",
            borderwidth=1,
        ),
        autosize=False,
        xaxis=dict(
            gridwidth=1,
            mirror=True,  # ticks are mirror at the top of the frame also
            showspikes=True,
            visible=True,
        ),
        yaxis=dict(
            title_text=setdict["title"],
            gridwidth=2,
            type="linear",
            autorange=True,
            showspikes=True,
            visible=True,
            showline=True,  # show a separating line
            side="left",  # show on the RHS
        ),
        width=setdict["html_aspect_ratio_w_over_h"] * setdict["html_image_height"],
        height=setdict["html_image_height"],
    )
    return fig
