# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/08_external_data.ipynb.

# %% auto 0
__all__ = ['MURLs', 'download_ixi_data']

# %% ../nbs/08_external_data.ipynb 2
from pathlib import Path
from glob import glob
import pandas as pd
from monai.apps import download_url, download_and_extract

# %% ../nbs/08_external_data.ipynb 4
class MURLs():
    '''A class with external medical dataset URLs.'''

    IXI_DATA = 'http://biomedic.doc.ic.ac.uk/brain-development/downloads/IXI/IXI-T1.tar'
    IXI_DEMOGRAPHIC_INFORMATION = 'http://biomedic.doc.ic.ac.uk/brain-development/downloads/IXI/IXI.xls'

# %% ../nbs/08_external_data.ipynb 5
def _process_ixi_xls(xls_path:(str, Path), img_path: Path):
    '''Private method to process the demographic information for the IXI dataset.

    Args:
        xls_path: File path to the xls file with the demographic information.
        img_path: Folder path to the images

    Returns:
        DataFrame: A processed dataframe with image path and demographic information.
    '''

    print('Preprocessing ' + str(xls_path))

    df = pd.read_excel(xls_path)

    duplicate_subject_ids = df[df.duplicated(['IXI_ID'], keep=False)].IXI_ID.unique()

    for subject_id in duplicate_subject_ids:
        age = df.loc[df.IXI_ID == subject_id].AGE.nunique()
        if age != 1: df = df.loc[df.IXI_ID != subject_id] #Remove duplicates with two different age values

    df = df.drop_duplicates(subset='IXI_ID', keep='first').reset_index(drop=True)

    df['subject_id'] = ['IXI' + str(subject_id).zfill(3) for subject_id in df.IXI_ID.values]
    df = df.rename(columns={'SEX_ID (1=m, 2=f)': 'gender'})
    df['age_at_scan'] = df.AGE.round(2)
    df = df.replace({'gender': {1:'M', 2:'F'}})

    img_list = list(img_path.glob('*.nii.gz'))
    for path in img_list:
        subject_id = path.parts[-1].split('-')[0]
        df.loc[df.subject_id == subject_id, 't1_path'] = path

    df = df.dropna()
    df = df[['t1_path', 'subject_id', 'gender', 'age_at_scan']]
    return df

# %% ../nbs/08_external_data.ipynb 6
def download_ixi_data(path:(str, Path)='../data' # Path to the directory where the data will be stored
                     ):
    '''Download T1 scans and demographic information from the IXI dataset, then process the demographic 
        information for each subject and save the information as a CSV file.
    Returns path to the stored CSV file.
    '''
    path = Path(path)/'IXI'
    img_path = path/'T1_images' 

    # Check whether image data already present in img_path:
    is_extracted=False
    try:
        if len(list(img_path.iterdir())) >= 581: # 581 imgs in the IXI dataset
            is_extracted=True
            print(f"Images already downloaded and extracted to {img_path}")
    except:
        is_extracted=False

    # Download and extract images
    if not is_extracted: 
        download_and_extract(url=MURLs.IXI_DATA, filepath=path/'IXI-T1.tar', output_dir=img_path)
        (path/'IXI-T1.tar').unlink()


    # Download demographic info
    download_url(url=MURLs.IXI_DEMOGRAPHIC_INFORMATION, filepath=path/'IXI.xls')

    processed_df = _process_ixi_xls(xls_path=path/'IXI.xls', img_path=img_path)
    processed_df.to_csv(path/'dataset.csv',index=False)

    return path
