# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['comma', 'comma.classes']

package_data = \
{'': ['*']}

extras_require = \
{'autodetect': ['chardet>=3.0.4,<4.0.0',
                'binaryornot>=0.4.4,<0.5.0',
                'clevercsv>=0.6.3,<0.7.0'],
 'net': ['requests>=2.23.0,<3.0.0']}

setup_kwargs = {
    'name': 'comma',
    'version': '0.5.1',
    'description': 'Python CSV, and delimiter-spaced files, for humans!',
    'long_description': '# Comma: A Python CSV Library for Humans\n\n![pytest](https://github.com/jlumbroso/comma/workflows/pytest/badge.svg)\n [![codecov](https://codecov.io/gh/jlumbroso/comma/branch/master/graph/badge.svg)](https://codecov.io/gh/jlumbroso/comma)\n [![Documentation Status](https://readthedocs.org/projects/comma/badge/?version=latest)](https://comma.readthedocs.io/en/latest/?badge=latest)\n [![Run on Repl.it](https://repl.it/badge/github/jlumbroso/comma)](https://repl.it/github/jlumbroso/comma)\n [![Stargazers](https://img.shields.io/github/stars/jlumbroso/comma?style=social)](https://github.com/jlumbroso/comma)\n\nThis library tries to make manipulating CSV files a great experience.\n\n![Example session](https://raw.githubusercontent.com/jlumbroso/comma/master/example-screenshot.png)\n\n## Installation\n\nIf you use pip:\n```shell script\npip install \'comma[autodetect,net]\'\n```\nor if you use pipenv:\n```shell script\npipenv install \'comma[autodetect,net]\'\n```\n\n## Why?\n\nAlthough Python, fortuitously, is\n"[batteries included](https://www.python.org/dev/peps/pep-0206/#batteries-included-philosophy)",\non occasion, some of the libraries end up being designed with APIs\nthat don\'t map well to what turns out to be the most common usage\npatterns. This is what happened with the various `urllib` libraries,\nincredibly powerful, but limiting users by its complexity---it was\nnot straightforward, for instance, to use cookies: One of several\nproblems that [`requests`](https://github.com/psf/requests) by\n[@ken-reitz](https://github.com/ken-reitz) addressed. Indeed,\n`requests` abstracts power beneath simplicity, smart defaults, and\ndiscoverability.\n\nFor the CSV format, we are confronted with a similar situation. While\nboth the JSON and YAML formats have packages that provide, one-command\nmeans to load content from files in those respective formats to a\nnested Python object, for the CSV format,\n[the standard library](https://www.python.org/dev/peps/pep-0305/) has\nyou use an iterator to access the data. Many details require significant\nsyntax change (for instance the difference between have lists or\ndictionaries depends on the class that is used to read the file).\n\nSince then, we also have several excellent libraries that, by providing\ngreat auto-detection (of dialect, file format, encoding, etc.) allow\nfor hiding many details from the end user.\n\nAll this to say, `comma` will try to do exactly what you want\nwhen you do:\n\n```python\nimport comma\ndata = comma.load("file.csv")\ndata[0]["field"] = "changed value"\ncomma.dump(data, filename="file_modified.csv")\n``` \n\n## Alternatives\n\nPython is fortunate to have a lot of very good libraries to read/write\nCSV and tabular files in general. (Some of these were discovered through\nthe excellent [Awesome Python](https://awesome-python.com/) list.)\n\n- [`clevercsv`](https://github.com/alan-turing-institute/CleverCSV): An\n  exceptional library by [@GjjvdBurg](https://github.com/GjjvdBurg),\n  builds on statistical and empirical\n  to provide powerful and reliable CSV dialect detection. However, it\n  strives to be a drop-in replacement for the original Python `csv`\n  module, and as such does not improve on the complex syntax. This\n  library is [the culmination of serious peer-reviewed\n  research](https://arxiv.org/abs/1811.11242), and `comma` uses it\n  internally to improve auto-detection.\n  \n- [`csvkit`](https://github.com/wireservice/csvkit): This is a set of\n  command-line tools (rather than a module/package) written in Python,\n  to make it easier to manipulate CSV files. One of the highlights is\n  a tool called `csvpy <file.csv>` to open a Python shell with the CSV\n  data loaded into a Python object called `reader`, to quickly run\n  some Python logic on the data. While it is technically possible to\n  use `csvkit`\'s internals in a project, this is not documented.\n  \n- [`pandas`](https://github.com/pandas-dev/pandas): An advanced data\n  science package for Python, this certainly provides a powerful CSV\n  (and more generally, table file) reader and parser. The API of the\n  table object is very powerful, but you need to take the time to learn\n  how to use it. This library is perhaps not ideal for file manipulations.\n\n- [`pyexcel`](https://github.com/pyexcel/pyexcel): This library provides\n  access to Excel and other tabular formats, including CSV, and various\n  data sources (stream, database, file, ...). It emphasizes one common\n  format-agnostic API, that instead has the user choose the data format\n  (list, matrix, dictionary, ...).\n\n- [`tablib`](https://github.com/jazzband/tablib/): This library was\n  originally written by [Kenneth Reitz](https://github.com/ken-reitz),\n  the creator who brought [`requests`](https://github.com/psf/requests),\n  [`pipenv`](https://github.com/pypa/pipenv) and many other goodies to\n  Python---and then included in the [Jazzband](https://jazzband.co/)\n  collective. The focus of this library is on interoperating between many\n  different file formats (such as XLS, CSV, JSON, YAML, DF, etc., ...,\n  even LaTeX `booktabs`!). It seems to have a very high adoption rate\n  because it is a dependency for many Jazzband libraries. The API is\n  class-based rather than method-based. A companion library,\n  [`prettytable`](https://github.com/jazzband/prettytable) focuses on\n  pretty printing tabular data (including from a CSV file).\n\n- [`tabulator`](https://github.com/frictionlessdata/tabulator-py): This\n  library provides a single interface to manipulate extremely large\n  tabular data---and useful for files so large that they need to be\n  streamed line-by-line; the library supports a broad array of formats\n  including reading data directly from Google Spreadsheets. However\n  this power means that reading a CSV file requires several operations.\n\nAlthough not specifically restricted to Python, the\n[AwesomeCSV](https://github.com/secretGeek/AwesomeCSV) resource is also\ninteresting.\n\n### Miscellaneous\n\nAlthough not specifically a Python library, nor designed to read/write CSV\nfiles (but instead to compare them), [`daff`](https://github.com/paulfitz/daff)\nis a really cool project: It provides a `diff` of tabular data [with cell-level\nawareness](http://specs.frictionlessdata.io/tabular-diff/).\n\nAnother unrelated project is [Grist](https://www.getgrist.com/), a spreadsheet\nPaaS, which among other useful features, allows [the use of Python within\nformulas](https://support.getgrist.com/formulas/). \n\n## Acknowledgements\n\nThanks to [@zbanks](https://github.com/zbanks) for the name of the package!\nThanks to [@rfreling](https://github.com/rfreling),\n[@adamfinkelstein](https://github.com/adamfinkelstein) for discussing ideas\nbefore I got started on this. Thanks to [@GjjvdBurg](https://github.com/GjjvdBurg)\nand collaborators for awesome, awesome contribution to text processing science\nand our Python community with `clevercsv`. \n\n## License\n\nThis project is licensed under the LGPLv3 license, with the understanding\nthat importing a Python modular is similar in spirit to dynamically linking\nagainst it.\n\n- You can use the library `comma` in any project, for any purpose, as long\n  as you provide some acknowledgement to this original project for use of\n  the library.\n\n- If you make improvements to `comma`, you are required to make those\n  changes publicly available.\n  \n',
    'author': 'Jérémie Lumbroso',
    'author_email': 'lumbroso@cs.princeton.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jlumbroso/comma',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
