import * as ecs from 'aws-cdk-lib/aws-ecs';
import * as inst from './instrumentation-versions';
/**
 * Interface for environment extensions.
 */
export interface EnvironmentExtension {
    /**
     * The name of the environment variable.
     */
    readonly name: string;
    /**
     * The value of the environment variable.
     */
    readonly value: string;
}
/**
 * Injector is a base class for all SDK injects to mutate the task definition
 * to inject the ADOT init container and configure the application container with
 * the necessary environment variables.
 */
export declare abstract class Injector {
    protected static readonly DEFAULT_ENVS: EnvironmentExtension[];
    protected sharedVolumeName: string;
    protected instrumentationVersion: inst.InstrumentationVersion;
    private overrideEnvironments?;
    constructor(sharedVolumeName: string, instrumentationVersion: inst.InstrumentationVersion, overrideEnvironments?: EnvironmentExtension[]);
    /**
     * The command to run the init container.
     */
    abstract get command(): string[];
    /**
     * The path to ADOT SDK agent in the init container.
     */
    abstract get containerPath(): string;
    /**
     * Inject additional environment variables to the application container other than the DEFAULT_ENVS.
     */
    protected abstract injectAdditionalEnvironments(envsToInject: {
        [key: string]: string;
    }, envsFromTaskDef: {
        [key: string]: string;
    }): void;
    /**
     * Override environment variables in the application container.
     */
    protected abstract overrideAdditionalEnvironments(envsToOverride: {
        [key: string]: string;
    }, envsFromTaskDef: {
        [key: string]: string;
    }): void;
    /**
     * Inject ADOT SDK agent init container.
     * @param taskDefinition The TaskDefinition to render
     * @returns The created ContainerDefinition
     */
    injectInitContainer(taskDefinition: ecs.TaskDefinition): ecs.ContainerDefinition;
    /**
     * Render the application container for SDK instrumentation.
     * @param taskDefinition The TaskDefinition to render
     */
    renderDefaultContainer(taskDefinition: ecs.TaskDefinition): void;
}
/**
 * Java-specific implementation of the SDK injector.
 * Handles Java agent configuration and environment setup.
 */
export declare class JavaInjector extends Injector {
    get command(): string[];
    get containerPath(): string;
    protected injectAdditionalEnvironments(envsToInject: {
        [key: string]: string;
    }, _envsFromTaskDef: {
        [key: string]: string;
    }): void;
    protected overrideAdditionalEnvironments(_envsToOverride: {
        [key: string]: string;
    }, _overrideEnvironments: {
        [key: string]: string;
    }): void;
}
/**
 * Python-specific implementation of the SDK injector.
 * Handles Python auto-instrumentation setup and PYTHONPATH configuration.
 */
export declare class PythonInjector extends Injector {
    protected static readonly PYTHON_ENVS: EnvironmentExtension[];
    get command(): string[];
    protected injectAdditionalEnvironments(envsToInject: {
        [key: string]: string;
    }, _envsFromTaskDef: {
        [key: string]: string;
    }): void;
    get containerPath(): string;
    protected overrideAdditionalEnvironments(envsToOverride: {
        [key: string]: string;
    }, envsFromTaskDef: {
        [key: string]: string;
    }): void;
}
/**
 * Base class for .NET SDK injectors.
 * Contains common .NET configuration settings used by both Windows and Linux implementations.
 */
export declare abstract class DotNetInjector extends Injector {
    protected static readonly DOTNET_COMMON_ENVS: EnvironmentExtension[];
}
/**
 * Linux-specific implementation of the .NET SDK injector.
 * Handles CoreCLR profiler setup and paths for Linux environments.
 */
export declare class DotNetLinuxInjector extends DotNetInjector {
    protected static readonly DOTNET_LINUX_ENVS: EnvironmentExtension[];
    private cpuArch;
    constructor(sharedVolumeName: string, instrumentationVersion: inst.InstrumentationVersion, cpuArch: ecs.CpuArchitecture, overrideEnvironments?: EnvironmentExtension[]);
    get command(): string[];
    protected injectAdditionalEnvironments(envsToInject: {
        [key: string]: string;
    }, envsFromTaskDef: {
        [key: string]: string;
    }): void;
    get containerPath(): string;
    protected overrideAdditionalEnvironments(_envsToOverride: {
        [key: string]: string;
    }, _envsFromTaskDef: {
        [key: string]: string;
    }): void;
    private getCoreCLRProfilerPath;
}
/**
 * Windows-specific implementation of the .NET SDK injector.
 * Handles CoreCLR profiler setup and paths for Windows environments.
 */
export declare class DotNetWindowsInjector extends DotNetInjector {
    protected static readonly DOTNET_WINDOWS_ENVS: EnvironmentExtension[];
    get command(): string[];
    protected injectAdditionalEnvironments(envsToInject: {
        [key: string]: string;
    }, envsFromTaskDef: {
        [key: string]: string;
    }): void;
    get containerPath(): string;
    protected overrideAdditionalEnvironments(_envsToOverride: {
        [key: string]: string;
    }, _envsFromTaskDef: {
        [key: string]: string;
    }): void;
}
/**
 * Node.js-specific implementation of the SDK injector.
 * Handles Node.js auto-instrumentation setup and NODE_OPTIONS configuration.
 */
export declare class NodeInjector extends Injector {
    get command(): string[];
    protected injectAdditionalEnvironments(envsToInject: {
        [key: string]: string;
    }, _envsFromTaskDef: {
        [key: string]: string;
    }): void;
    get containerPath(): string;
    protected overrideAdditionalEnvironments(envsToOverride: {
        [key: string]: string;
    }, envsFromTaskDef: {
        [key: string]: string;
    }): void;
}
