"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationOU = exports.OrganizationOUProvider = exports.processOrgObj = exports.OrgObjectTypes = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The supported OrgObject types.
 *
 * @stability stable
 */
var OrgObjectTypes;
(function (OrgObjectTypes) {
    OrgObjectTypes["OU"] = "ou";
    OrgObjectTypes["ACCOUNT"] = "account";
})(OrgObjectTypes = exports.OrgObjectTypes || (exports.OrgObjectTypes = {}));
/**
 * @function processOrgObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} provider The custom resource provider the custom resources will utilized to create the resources.
 * @param {OrgObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
function processOrgObj(provider, obj, parent) {
    if (obj.type === OrgObjectTypes.OU) {
        const parentStr = parent instanceof OrganizationOU ? parent.resource.ref : parent;
        const ou = new OrganizationOU(this, obj.name, {
            provider,
            parent: parentStr,
            name: obj.name,
        });
        obj.children.forEach(child => {
            processOrgObj.call(this, provider, child, ou);
        });
    }
}
exports.processOrgObj = processOrgObj;
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
class OrganizationOUProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:UpdateOrganizationalUnit',
                            'organizations:DeleteOrganizationalUnit',
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:CreateOrganizationalUnit',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/ou'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(10),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationOUProvider = OrganizationOUProvider;
_a = JSII_RTTI_SYMBOL_1;
OrganizationOUProvider[_a] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOUProvider", version: "0.1.3" };
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
class OrganizationOU extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e;
        super(scope, id);
        const importOnDuplicate = (_c = props.importOnDuplicate) !== null && _c !== void 0 ? _c : false;
        const allowMergeOnMove = (_d = props.allowMergeOnMove) !== null && _d !== void 0 ? _d : false;
        const allowRecreateOnUpdate = (_e = props.allowRecreateOnUpdate) !== null && _e !== void 0 ? _e : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                ImportOnDuplicate: importOnDuplicate,
                AllowMergeOnMove: allowMergeOnMove,
                AllowRecreateOnUpdate: allowRecreateOnUpdate,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationOU = OrganizationOU;
_b = JSII_RTTI_SYMBOL_1;
OrganizationOU[_b] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOU", version: "0.1.3" };
//# sourceMappingURL=data:application/json;base64,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