// *****************************************************************************
// General attribute
// AttributeManager, platform-independent
//
// Namespaces:					Contact person:
//	GeneralAttribute				  JG
//	GeneralAttributeField			  JG
//	GeneralAttributeFieldMask		  JG
//	GeneralAttributeObserver		  JG
//
// [SG compatible]
// *****************************************************************************

#if !defined (GENERALATTRIBUTE_HPP)
#define GENERALATTRIBUTE_HPP

#pragma once

//============================== Includes ==================================

#include	"GSRoot.hpp"
#include	"ODBObject.hpp"
#include	"Array.hpp"
#include	"AttributeManagerEnv.hpp"
#include	"AttributeManager.hpp"
#include	"EventSource.hpp"
#include	"EventObserver.hpp"
#include	"FieldMask.hpp"
#include	"RegTblData.hpp"
#include    "String.hpp"
#include	"UniString.hpp"
#include	"GSUnID.hpp"

#include	"NamedEnum.hpp"
#include	"TWObject.hpp"
#include	"VBOwner.hpp"
#include	"ADBGeneralDefinitions.hpp"

#include	"ITWSpecificDataContainer.hpp"
#include	"IClientCreatableObject.hpp"


namespace GS {
	class XMLOChannel;
	class XMLIChannel;
}

class ATTRIBUTE_DLL_EXPORT IUsedAttrCollection
{
public:
	virtual void MarkAttributeAsUsed (ADB::IDAttribute, short attributeIndex) = 0;
	virtual bool IsAttributeUsed (ADB::IDAttribute, short attributeIndex) const = 0;
	virtual bool SkipAttributeType (ADB::IDAttribute idAttribute) const = 0;

    virtual ~IUsedAttrCollection ();
};

class ATTRIBUTE_DLL_EXPORT IUsedLibPartCollection
{
public:
	virtual void ByUniString (const GS::UniString& name, ADB::GeneralAttributeConstRef attrRef) = 0;
	virtual void ByUnIDAndUniString (const GS::UnID& id, const GS::UniString& name, ADB::GeneralAttributeConstRef attrRef) = 0;

	virtual ~IUsedLibPartCollection ();
};

//======================= Begin of the ADB namespace =======================

namespace ADB {

//============================= Definitions ================================

enum AttributeFlagBits {
	FromGDLBit			= 6,	// attribute is generated by GDL
	ClientCreatedBit	= 7,	// new attr. created by client
	FromMasterGDLBit	= 8		// attribute is defined in Master_GDL
};

//--------------------------------------------------------------------------

#define	FromGDL				(1<<ADB::FromGDLBit)
#define	FromMasterGDL		(1<<ADB::FromMasterGDLBit)
#define	ClientCreated		(1<<ADB::ClientCreatedBit)

//======================= Class GeneralAttribute ============================
//
// GeneralAttribute is the base class of the attributes.
//
//===========================================================================

class AttributeSet;
class AttributeHandle;
class AttributeTypeSet;
class GeneralAttributeObserver;
class IAttributeNewIndexTable;

class ATTRIBUTE_DLL_EXPORT GeneralAttribute : public TW::Object,
											  public GS::EventSource,
											  public TW::ITWSpecificDataContainer,
											  public TW::IClientCreatableObject
{
	DECLARE_CLASS_INFO

public:
		// constants

	enum Fields {
		FieldError			= 0,
		First				= 1,
		Name				= 1,	// First field in GeneralAttribute
		Index				= 2,
		Flags				= 3,
		ModificationTime	= 4,
		UserData			= 5,	// Last field in GeneralAttribute
		Last				= 5
	};

		// internal classes

	class ATTRIBUTE_DLL_EXPORT UserDataConverter {
	private:
		VBUtil::Owner	owner;
		GSType			attributeType;		// in case of handling a certain attribute type only

	public:
		static short	outputPlanVersion;	// 0 means current version

	public:
		UserDataConverter (GSType owner1, GSType owner2, GSType attributeType = 0);
		virtual ~UserDataConverter ();

		bool				IsHandlingAttributeType (GSType typeId) const;

		virtual bool		ConvertAttributeUserData (GeneralAttributeConstRef	originalAttribute,
													  unsigned short&			platform,
													  short&					version,
													  GSFlags&					flags,
													  GSHandle&					userData) const = 0;

		static void			SetOutputPlanVersion (short outputPlanVersion);
		static short		GetOutputPlanVersion (void);
	};

	class ATTRIBUTE_DLL_EXPORT NotifyDisableGuard
	{
	private:
		static Int32		m_NotifyDisableNum;
		bool				m_Disable;

	public:
		explicit	NotifyDisableGuard (bool disable)
		{
			m_Disable = disable;
			if (m_Disable) {
				++m_NotifyDisableNum;
			}
		}

		~NotifyDisableGuard ()
		{
			if (m_Disable) {
				--m_NotifyDisableNum;
				DBASSERT (m_NotifyDisableNum >= 0);
			}
		}

		static bool	IsDisabledNotify (void)
		{
			return m_NotifyDisableNum > 0;
		}
	};


private:
		// friend classes

	friend class AttributeSet;
	friend class AttributeContainer;
	friend class AttributeTypeSet;

#define _GENERALATTRIBUTE_DATA_MEMBERS_																						\
private:																													\
	GS::UniString			name;						/* name of attribute */												\
	short					ind;						/* index of attribute */											\
	short					flags;						/* attribute flags */												\
	GSTime					modiTime;					/* time of the last modification */									\
	RegTblData				regMemo;					/* user data */

		// data members

	_GENERALATTRIBUTE_DATA_MEMBERS_

protected:
	mutable GSHandle		prevUserData;				/* (non-persistent) for notifying user data changes */
	mutable short			oldVersionNumber;			/* (non-persistent) for writing older version data */
	mutable	bool			isDisabled;					/* (non-persistent) for showing disabled in popup controls */

		// transaction management class

protected:
	class ATTRIBUTE_DLL_EXPORT IsolatedState : public TW::Object::IsolatedState {
		private:
			friend class GeneralAttribute;

			_GENERALATTRIBUTE_DATA_MEMBERS_

		public:
			virtual ~IsolatedState ();
	};

protected:
	GeneralAttribute ();
	GeneralAttribute (const GS::ClassInfo* finalClassInfo);

public:
		// destructor

	virtual				   ~GeneralAttribute ();

	virtual GeneralAttribute*		Clone (void) const override = 0;
	virtual	GSType					GetTypeId (void) const = 0;

		// get methods

	inline	GS::Guid				GetGuid (void) const { return GetRef ().GetId ().ToGuid (); }
	inline	void					GetName (char* retName, Int32 bufSize = AttributeNameLength) const { CHTruncate (name.ToCStr ().Get (), retName, bufSize); }
	inline	const GS::UniString&	GetName (void) const { return name; }
	inline	void					GetName (GS::uchar_t* retName, Int32 bufSizeOfBytes) const { GS::UTruncate (name.ToUStr ().Get (), retName, bufSizeOfBytes); }
	inline	short					GetIndex (void) const { return ind; }
	inline	short					GetFlags (void) const { return flags; }
	inline	GSTime					GetModificationTime (void) const { return modiTime; }

	GSErrCode						GetUserData (GSType				owner1,
												 GSType				owner2,
												 unsigned short*	platform,
												 short*				version,
												 GSFlags*			flags,
												 GSHandle*			userData) const;

	AttributeContainerRef			GetAttributeContainer (void) const;
	AttributeContainerConstRef		GetConstAttributeContainer (void) const;
	AttributeSetRef					GetAttributeSet (void) const;
	AttributeSetConstRef			GetConstAttributeSet (void) const;

	const GS::Array<VBUtil::Owner>	GetOwners (void) const;

	inline	bool			IsSameFlags (short otherFlags) const { return (flags & otherFlags) > 0; }
	inline	bool			IsFromGDL (void) const { return (flags & FromGDL) > 0; }
	inline	bool			IsFromMasterGDL (void) const { return (flags & FromMasterGDL) > 0; }

	inline	bool			IsDisabled () const { return isDisabled; }
	inline	void			MarkDisabled (bool disableState) const { isDisabled = disableState; }

	virtual	FieldMask		Compare (GeneralAttributeConstRef attribute) const;

		// set methods

	GSErrCode				SetName (const GS::UniString& newName);
	GSErrCode				SetIndex (short newIndex);
	void					SetFlags (short newFlags);
	void					SetModificationTime (const GSTime& newModificationTime);
	void					UpdateModificationTime ();

	GSErrCode				SetUserData (GSType			owner1,
										 GSType			owner2,
										 unsigned short	platform,
										 short			version,
										 GSFlags		flags,
										 GSHandle		userData);

	void					AddPropertyToFlags (short newFeature);
	void					RemovePropertyFromFlags (short oldFeature);

	virtual void			Synchronize (GeneralAttributeConstRef newAttribute, const FieldMask& changes);
	virtual void			LookupExternalDataHandlers (VBUtil::ICBOwnerFound* cbInterface) const;
	virtual void			DeleteExternalDataHandlers	(const VBUtil::OwnerList& owners);

		// observer methods

	void					Attach (GeneralAttributeObserver& observer);
	void					Detach (GeneralAttributeObserver& observer);

		// delete methods

	void					Clear (void);
	GSErrCode				DeleteUserData (GSType owner1, GSType owner2);

		// check methods

	virtual bool			CheckAttribute (void) const;
	virtual GSErrCode		CorrectAttribute (void);

		// memo methods

	virtual bool			HasMemoData () const;
	virtual bool			HasSameMemoData (GeneralAttributeConstRef attribute) const;

		// reindex methods

	virtual void			CollectUsedAttrs (IUsedAttrCollection&) const;
	virtual	void			ReindAttr (IAttributeNewIndexTable* newIndexTable);

	static	GS::Guid		GetFieldClassId	(void);

		// used libpart methods

	virtual void			CollectUsedLibPartNames (IUsedLibPartCollection&) const { /* NOP */ }

protected:
		// constructors
							GeneralAttribute (const GeneralAttribute& source);

		// operator overloading
	GeneralAttribute&		operator= (const GeneralAttribute& source);

protected:
		// set methods

	void					DirectAddFlags (short flags) { this->flags |= flags; }
	void					DirectRemoveFlags (short flags) { this->flags &= ~flags; }

		// read / write methods (Object interface)

public:
	virtual GSErrCode		Read (GS::IChannel& ic) override;
	virtual GSErrCode		Write (GS::OChannel& oc) const override;
	virtual GSErrCode		WriteXML (GS::XMLOChannel& oc) const override;
	virtual GSErrCode		ReadXML (GS::XMLIChannel& ic) override;

private:
	GSErrCode				ReadVersion1 (GS::IChannel& ic);
	GSErrCode				WriteVersion1 (GS::OChannel& oc) const;
	GSErrCode				ReadVersion2 (GS::IChannel& ic);
	GSErrCode				WriteVersion2 (GS::OChannel& oc) const;
	GSErrCode				ReadVersion3 (GS::IChannel& ic);
	GSErrCode				WriteVersion3 (GS::OChannel& oc) const;
	GSErrCode				ReadVersion4 (GS::IChannel& ic);
	GSErrCode				WriteVersion4 (GS::OChannel& oc) const;

	RegTblData				ConvertRegMemo (void) const;

protected:
	GSErrCode				WriteGenAttrXML (GS::XMLOChannel& oc, const GSXML::NamedEnumType<short>* flagsInfo = nullptr, IRegDataWriter* regMemoDataWriter = nullptr) const;
	GSErrCode				ReadGenAttrXML (GS::XMLIChannel& ic, const GSXML::NamedEnumType<short>* flagsInfo = nullptr, IRegDataReader* regMemoDataReader = nullptr);

		// observer methods

	virtual	void			Notify (GS::EventObserver* observer, const FieldMask* changes) const;
	void					NotifyObservers (const FieldID& changedField) const;
	void					NotifyObservers (const FieldMask* changes) const;

		// copy methods

	virtual void			Copy (GeneralAttributeConstRef newAttribute);

		// memo methods

public:
	virtual	GSErrCode		Destruct (void);


private:
	void					GenerateNewName (void);

public:
		// old version read / write
	static GSErrCode		ReadVersion1 (GS::IChannel& ic,
										  Int32* iLen, GS::UniString* iName, short* iInd, short* iFlags,
										  GSTime* iModiTime, short* iCheckSum, short* iVersion, RegTblData* iRegMemo);
	static GSErrCode		WriteVersion1 (GS::OChannel& oc,
										   Int32 oLen, const GS::UniString& oName, short oInd, short oFlags,
										   GSTime oModiTime, short oCheckSum, short oVersion, const RegTblData& oRegMemo);

public:
		// Transaction management methods

	virtual GSErrCode					StoreIsolatedState		(ODB::Object::IsolatedState* isolatedState) const override;
	virtual GSErrCode					RestoreIsolatedState	(const ODB::Object::IsolatedState* isolatedState) override;
	virtual void						Isolate					(void) override;
	virtual ODB::Object::IsolatedState*	CreateIsolatedState		(void) const override;
	virtual USize						GetIsolatedSize			(void) const override;
	virtual GSErrCode					WriteIsolatedState		(GS::OChannel& oc) const override;
	virtual GSErrCode					ReadIsolatedState 		(GS::IChannel& ic) override;

private:
	GSErrCode							ReadIsolatedStateVersion4 (GS::IChannel& ic, const GS::InputFrame& frame);
	GSErrCode							WriteIsolatedStateVersion4 (GS::OChannel& oc, const GS::OutputFrame& frame) const;

		// TWObject methods

public:
	virtual GSErrCode		ReadTWIsolatedState  (GS::IChannel& ic) override;
	virtual GSErrCode		WriteTWIsolatedState (GS::OChannel& oc) const override;

private:
	GSErrCode				ReadTWIsolatedStateVersion3 (GS::IChannel& ic, const GS::InputFrame& frame);
	GSErrCode				WriteTWIsolatedStateVersion3 (GS::OChannel& oc, const GS::OutputFrame& frame) const;
	GSErrCode				ReadTWIsolatedStateVersion4 (GS::IChannel& ic, const GS::InputFrame& frame);
	GSErrCode				WriteTWIsolatedStateVersion4 (GS::OChannel& oc, const GS::OutputFrame& frame) const;

public:
	virtual	bool			IsEqual (GeneralAttributeConstRef attrib) const;
	virtual	bool			IsCustom (void) const;

		// ITWSpecificDataContainer method

	virtual void			ClearTWSpecificData (void) override;

		// IClientCreatableObject method

	virtual bool			IsClientCreated (void) const override;
	virtual void			SetClientCreated (bool newClientCreated) override;
};

//--------------------------------------------------------------------------

#define GeneralAttributeField(idx) FieldIDT<ADB::GeneralAttribute>(ADB::GeneralAttribute::GetClassInfoStatic ()->GetId (), idx)
typedef FieldMaskT<ADB::GeneralAttribute>		GeneralAttributeFieldMask;

//================ Class GeneralAttributeObserver ==========================
//
// Observer class for GeneralAttribute.
//
//==========================================================================

class ATTRIBUTE_DLL_EXPORT GeneralAttributeObserver : public GS::EventObserver {
protected:
		// destructor

	virtual ~GeneralAttributeObserver () {}

public:
		// changed methods

	virtual void	NameChanged (GeneralAttributeConstRef);
	virtual void	IndexChanged (GeneralAttributeConstRef);
	virtual void	FlagsChanged (GeneralAttributeConstRef);
	virtual void	ModificationTimeChanged (GeneralAttributeConstRef);
	virtual void	UserDataChanged (GSType owner1, GSType owner2);

	virtual void	AttributeChanged (GeneralAttributeConstRef, const FieldMask&);

		// name changed method specifying the old name too
	virtual void	NameChanged (GeneralAttributeRef, const GS::UniString&);
};


//======================== End of the ADB namespace ========================

}

#endif
