"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateTagsSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
const aws_service_1 = require("../../domain/aws-service");
/**
 * AutomationStep implemenation for aws:createTags
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createtag.html
 */
class CreateTagsSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:createTags';
        this.createTagsStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        const resourceType = this.createTagsStep.resourceType?.resolveToString(inputs) ?? 'EC2';
        switch (resourceType) {
            case 'EC2':
                this.createEc2Tags(inputs);
                break;
            case 'ManagedInstance':
            case 'MaintenanceWindow':
            case 'Parameter':
                this.createSsmTags(resourceType, inputs);
                break;
            default:
                throw new Error(`${resourceType} is not a valid resource type for tagging`);
        }
        return {};
    }
    createEc2Tags(inputs) {
        const resourceIds = this.createTagsStep.resourceIds.resolveToStringList(inputs);
        const tags = this.createTagsStep.tags.resolveToMapList(inputs);
        console.log('CreateTags: Create EC2 tags for resources');
        this.callEc2CreateTags(resourceIds, tags);
        console.log('CreateTags: Wait for tags to be completed');
        this.waitForEc2Tags(resourceIds, tags);
        console.log('CreateTags: EC2 tags are associated to the resources');
    }
    callEc2CreateTags(resourceIds, tags) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ec2CreateTags', {
            service: aws_service_1.AwsService.EC2,
            pascalCaseApi: 'CreateTags',
            apiParams: {
                Resources: resourceIds,
                Tags: tags,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Create EC2 tags failed: ${result.stackTrace}`);
        }
    }
    waitForEc2Tags(resourceIds, tags) {
        while (!this.areEc2TagsAssociatedToTheResources(resourceIds, tags)) {
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
        }
    }
    areEc2TagsAssociatedToTheResources(resourceIds, tags) {
        const tagKeys = tags.map(tag => tag.Key);
        const tagValues = tags.map(tag => tag.Value);
        const filters = this.buildEc2TagFilter(resourceIds, tagKeys, tagValues);
        const tagDescriptions = this.ec2DescribeTags(filters);
        for (const tag of tags) {
            const descriptionsOfTag = tagDescriptions.filter(x => x.Key === tag.Key && x.Value === tag.Value);
            const describedResourceIds = new Set(descriptionsOfTag.map(x => x.ResourceId));
            if (!resourceIds.every(x => describedResourceIds.has(x))) {
                return false;
            }
        }
        return true;
    }
    buildEc2TagFilter(resourceIds, tagKeys, tagValues) {
        const potentialFilters = {
            'resource-id': resourceIds,
            'key': tagKeys,
            'value': tagValues,
        };
        const filters = [];
        for (const name of Object.keys(potentialFilters)) {
            if (potentialFilters[name].length !== 0) {
                filters.push({ Name: name, Values: potentialFilters[name] });
            }
        }
        return filters;
    }
    ec2DescribeTags(filters) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ec2DescribeTags', {
            service: aws_service_1.AwsService.EC2,
            pascalCaseApi: 'DescribeTags',
            apiParams: {
                Filters: filters,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    selector: '$.Tags',
                    name: 'Tags',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Describe EC2 tags failed: ${result.stackTrace}`);
        }
        return result.outputs?.['ec2DescribeTags.Tags'] ?? [];
    }
    createSsmTags(resourceType, inputs) {
        const resourceId = this.getSingleResourceId(resourceType, inputs);
        const tags = this.createTagsStep.tags.resolveToMapList(inputs);
        console.log('CreateTags: Create tags for resource');
        this.callSsmAddTagsToResource(resourceType, resourceId, tags);
        console.log('CreateTags: Wait for tags to be completed');
        this.waitForSsmTags(resourceType, resourceId, tags);
        console.log('CreateTags: Tags are associated to the resource');
    }
    callSsmAddTagsToResource(resourceType, resourceId, tags) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ssmAddTagsToResource', {
            service: aws_service_1.AwsService.SSM,
            pascalCaseApi: 'AddTagsToResource',
            apiParams: {
                ResourceType: resourceType,
                ResourceId: resourceId,
                Tags: tags,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Add tags failed: ${result.stackTrace}`);
        }
    }
    waitForSsmTags(resourceType, resourceId, tags) {
        while (!this.areAllSsmTagsListed(resourceType, resourceId, tags)) {
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
        }
    }
    areAllSsmTagsListed(resourceType, resourceId, tags) {
        const listedTags = this.ssmListTagsForResource(resourceType, resourceId);
        const listedTagMap = new Map(listedTags.map(tag => [tag.Key, tag.Value]));
        return tags.every(tag => listedTagMap.get(tag.Key) === tag.Value);
    }
    ssmListTagsForResource(resourceType, resourceId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ssmListTagsForResource', {
            service: aws_service_1.AwsService.SSM,
            pascalCaseApi: 'ListTagsForResource',
            apiParams: {
                ResourceType: resourceType,
                ResourceId: resourceId,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    selector: '$.TagList',
                    name: 'Tags',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`SSM List tags failed: ${result.stackTrace}`);
        }
        return result.outputs?.['ssmListTagsForResource.Tags'] ?? [];
    }
    getSingleResourceId(resourceType, inputs) {
        const resourceIds = this.createTagsStep.resourceIds.resolve(inputs);
        if (resourceIds.length !== 1) {
            throw new Error(`Only 1 resource allowed for type ${resourceType}`);
        }
        return resourceIds[0];
    }
}
exports.CreateTagsSimulation = CreateTagsSimulation;
_a = JSII_RTTI_SYMBOL_1;
CreateTagsSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.CreateTagsSimulation", version: "0.0.29" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLXRhZ3Mtc2ltdWxhdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9zaW11bGF0aW9uL2F1dG9tYXRpb24vY3JlYXRlLXRhZ3Mtc2ltdWxhdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQUFvQztBQUNwQyxzREFBc0Q7QUFDdEQsOERBQTBEO0FBRzFELDZFQUF3RTtBQUV4RSx5RUFBcUU7QUFDckUsOEVBQXlFO0FBQ3pFLDZFQUF3RTtBQUN4RSwwREFBb0Q7QUFvQ3BEOzs7R0FHRztBQUNILE1BQWEsb0JBQXFCLFNBQVEscURBQXdCO0lBS2hFLFlBQVksSUFBb0IsRUFBRSxLQUFnQztRQUNoRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFMTCxXQUFNLEdBQUcsZ0JBQWdCLENBQUM7UUFNakMsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUM7UUFDM0IsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDckIsQ0FBQztJQUVNLFdBQVcsQ0FBQyxNQUEyQjtRQUM1QyxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksRUFBRSxlQUFlLENBQUMsTUFBTSxDQUFDLElBQUksS0FBSyxDQUFDO1FBQ3hGLFFBQVEsWUFBWSxFQUFFO1lBQ3BCLEtBQUssS0FBSztnQkFDUixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUMzQixNQUFNO1lBQ1IsS0FBSyxpQkFBaUIsQ0FBQztZQUN2QixLQUFLLG1CQUFtQixDQUFDO1lBQ3pCLEtBQUssV0FBVztnQkFDZCxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxNQUFNLENBQUMsQ0FBQztnQkFDekMsTUFBTTtZQUNSO2dCQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxZQUFZLDJDQUEyQyxDQUFDLENBQUM7U0FDL0U7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFFTyxhQUFhLENBQUMsTUFBMkI7UUFDL0MsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsbUJBQW1CLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDaEYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFL0QsT0FBTyxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDMUMsT0FBTyxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sQ0FBQyxHQUFHLENBQUMsc0RBQXNELENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBRU8saUJBQWlCLENBQUMsV0FBcUIsRUFBRSxJQUEyQjtRQUMxRSxNQUFNLE1BQU0sR0FBRyxJQUFJLHFEQUF3QixDQUFDLElBQUkseUJBQVUsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxlQUFlLEVBQUU7WUFDdkYsT0FBTyxFQUFFLHdCQUFVLENBQUMsR0FBRztZQUN2QixhQUFhLEVBQUUsWUFBWTtZQUMzQixTQUFTLEVBQUU7Z0JBQ1QsU0FBUyxFQUFFLFdBQVc7Z0JBQ3RCLElBQUksRUFBRSxJQUFJO2FBQ1g7WUFDRCxPQUFPLEVBQUUsRUFBRTtTQUNaLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQzNCLElBQUksTUFBTSxDQUFDLFlBQVksS0FBSyw0QkFBWSxDQUFDLE9BQU8sRUFBRTtZQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixNQUFNLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUNqRTtJQUNILENBQUM7SUFFTyxjQUFjLENBQUMsV0FBcUIsRUFBRSxJQUEyQjtRQUN2RSxPQUFPLENBQUMsSUFBSSxDQUFDLGtDQUFrQyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsRUFBRTtZQUNsRSxJQUFJLHFEQUF3QixDQUFDLElBQUksc0JBQVMsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxPQUFPLEVBQUU7Z0JBQy9ELFlBQVksRUFBRSxDQUFDO2FBQ2hCLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQzVCO0lBQ0gsQ0FBQztJQUVPLGtDQUFrQyxDQUFDLFdBQXFCLEVBQUUsSUFBMkI7UUFDM0YsTUFBTSxPQUFPLEdBQWEsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNuRCxNQUFNLFNBQVMsR0FBYSxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXZELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXLEVBQUUsT0FBTyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hFLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFdEQsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLEVBQUU7WUFDdEIsTUFBTSxpQkFBaUIsR0FBRyxlQUFlLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQyxLQUFLLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ2xHLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFFL0UsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDeEQsT0FBTyxLQUFLLENBQUM7YUFDZDtTQUNGO1FBRUQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU8saUJBQWlCLENBQUMsV0FBcUIsRUFBRSxPQUFpQixFQUFFLFNBQW1CO1FBQ3JGLE1BQU0sZ0JBQWdCLEdBQTZCO1lBQ2pELGFBQWEsRUFBRSxXQUFXO1lBQzFCLEtBQUssRUFBRSxPQUFPO1lBQ2QsT0FBTyxFQUFFLFNBQVM7U0FDbkIsQ0FBQztRQUNGLE1BQU0sT0FBTyxHQUFtQixFQUFFLENBQUM7UUFDbkMsS0FBSyxNQUFNLElBQUksSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEVBQUU7WUFDaEQsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUN2QyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQzlEO1NBQ0Y7UUFDRCxPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRU8sZUFBZSxDQUFDLE9BQXVCO1FBQzdDLE1BQU0sTUFBTSxHQUFHLElBQUkscURBQXdCLENBQUMsSUFBSSx5QkFBVSxDQUFDLElBQUksbUJBQUssRUFBRSxFQUFFLGlCQUFpQixFQUFFO1lBQ3pGLE9BQU8sRUFBRSx3QkFBVSxDQUFDLEdBQUc7WUFDdkIsYUFBYSxFQUFFLGNBQWM7WUFDN0IsU0FBUyxFQUFFO2dCQUNULE9BQU8sRUFBRSxPQUFPO2FBQ2pCO1lBQ0QsT0FBTyxFQUFFLENBQUM7b0JBQ1IsVUFBVSxFQUFFLHdCQUFZLENBQUMsUUFBUTtvQkFDakMsUUFBUSxFQUFFLFFBQVE7b0JBQ2xCLElBQUksRUFBRSxNQUFNO2lCQUNiLENBQUM7U0FDSCxDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUMzQixJQUFJLE1BQU0sQ0FBQyxZQUFZLEtBQUssNEJBQVksQ0FBQyxPQUFPLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDbkU7UUFDRCxPQUFPLE1BQU0sQ0FBQyxPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUN4RCxDQUFDO0lBRU8sYUFBYSxDQUFDLFlBQW9CLEVBQUUsTUFBMkI7UUFDckUsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFlBQVksRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNsRSxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUvRCxPQUFPLENBQUMsR0FBRyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFlBQVksRUFBRSxVQUFVLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDOUQsT0FBTyxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxFQUFFLFVBQVUsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNwRCxPQUFPLENBQUMsR0FBRyxDQUFDLGlEQUFpRCxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVPLHdCQUF3QixDQUFDLFlBQW9CLEVBQUUsVUFBa0IsRUFBRSxJQUEyQjtRQUNwRyxNQUFNLE1BQU0sR0FBRyxJQUFJLHFEQUF3QixDQUFDLElBQUkseUJBQVUsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxzQkFBc0IsRUFBRTtZQUM5RixPQUFPLEVBQUUsd0JBQVUsQ0FBQyxHQUFHO1lBQ3ZCLGFBQWEsRUFBRSxtQkFBbUI7WUFDbEMsU0FBUyxFQUFFO2dCQUNULFlBQVksRUFBRSxZQUFZO2dCQUMxQixVQUFVLEVBQUUsVUFBVTtnQkFDdEIsSUFBSSxFQUFFLElBQUk7YUFDWDtZQUNELE9BQU8sRUFBRSxFQUFFO1NBQ1osQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0IsSUFBSSxNQUFNLENBQUMsWUFBWSxLQUFLLDRCQUFZLENBQUMsT0FBTyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQzFEO0lBQ0gsQ0FBQztJQUVPLGNBQWMsQ0FBQyxZQUFvQixFQUFFLFVBQWtCLEVBQUUsSUFBMkI7UUFDMUYsT0FBTyxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxZQUFZLEVBQUUsVUFBVSxFQUFFLElBQUksQ0FBQyxFQUFFO1lBQ2hFLElBQUkscURBQXdCLENBQUMsSUFBSSxzQkFBUyxDQUFDLElBQUksbUJBQUssRUFBRSxFQUFFLE9BQU8sRUFBRTtnQkFDL0QsWUFBWSxFQUFFLENBQUM7YUFDaEIsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDNUI7SUFDSCxDQUFDO0lBRU8sbUJBQW1CLENBQUMsWUFBb0IsRUFBRSxVQUFrQixFQUFFLElBQTJCO1FBQy9GLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxZQUFZLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDekUsTUFBTSxZQUFZLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzFFLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRU8sc0JBQXNCLENBQUMsWUFBb0IsRUFBRSxVQUFrQjtRQUNyRSxNQUFNLE1BQU0sR0FBRyxJQUFJLHFEQUF3QixDQUFDLElBQUkseUJBQVUsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSx3QkFBd0IsRUFBRTtZQUNoRyxPQUFPLEVBQUUsd0JBQVUsQ0FBQyxHQUFHO1lBQ3ZCLGFBQWEsRUFBRSxxQkFBcUI7WUFDcEMsU0FBUyxFQUFFO2dCQUNULFlBQVksRUFBRSxZQUFZO2dCQUMxQixVQUFVLEVBQUUsVUFBVTthQUN2QjtZQUNELE9BQU8sRUFBRSxDQUFDO29CQUNSLFVBQVUsRUFBRSx3QkFBWSxDQUFDLFFBQVE7b0JBQ2pDLFFBQVEsRUFBRSxXQUFXO29CQUNyQixJQUFJLEVBQUUsTUFBTTtpQkFDYixDQUFDO1NBQ0gsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0IsSUFBSSxNQUFNLENBQUMsWUFBWSxLQUFLLDRCQUFZLENBQUMsT0FBTyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQy9EO1FBQ0QsT0FBTyxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUMsNkJBQTZCLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDL0QsQ0FBQztJQUVPLG1CQUFtQixDQUFDLFlBQW9CLEVBQUUsTUFBMkI7UUFDM0UsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3BFLElBQUksV0FBVyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO1NBQUU7UUFDdEcsT0FBTyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDeEIsQ0FBQzs7QUFwTEgsb0RBc0xDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBEYXRhVHlwZUVudW0gfSBmcm9tICcuLi8uLi9kb21haW4vZGF0YS10eXBlJztcbmltcG9ydCB7IFJlc3BvbnNlQ29kZSB9IGZyb20gJy4uLy4uL2RvbWFpbi9yZXNwb25zZS1jb2RlJztcbmltcG9ydCB7IElBd3NJbnZva2VyIH0gZnJvbSAnLi4vLi4vaW50ZXJmYWNlL2F3cy1pbnZva2VyJztcbmltcG9ydCB7IElTbGVlcEhvb2sgfSBmcm9tICcuLi8uLi9pbnRlcmZhY2Uvc2xlZXAtaG9vayc7XG5pbXBvcnQgeyBBd3NBcGlTdGVwIH0gZnJvbSAnLi4vLi4vcGFyZW50LXN0ZXBzL2F1dG9tYXRpb24vYXdzLWFwaS1zdGVwJztcbmltcG9ydCB7IENyZWF0ZVRhZ3NTdGVwIH0gZnJvbSAnLi4vLi4vcGFyZW50LXN0ZXBzL2F1dG9tYXRpb24vY3JlYXRlLXRhZ3Mtc3RlcCc7XG5pbXBvcnQgeyBTbGVlcFN0ZXAgfSBmcm9tICcuLi8uLi9wYXJlbnQtc3RlcHMvYXV0b21hdGlvbi9zbGVlcC1zdGVwJztcbmltcG9ydCB7IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbiB9IGZyb20gJy4uL2F1dG9tYXRpb24tc3RlcC1zaW11bGF0aW9uJztcbmltcG9ydCB7IEF1dG9tYXRpb25TaW11bGF0aW9uQmFzZSB9IGZyb20gJy4vYXV0b21hdGlvbi1zaW11bGF0aW9uLWJhc2UnO1xuaW1wb3J0IHtBd3NTZXJ2aWNlfSBmcm9tIFwiLi4vLi4vZG9tYWluL2F3cy1zZXJ2aWNlXCI7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgQ3JlYXRlVGFnU3RlcFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENyZWF0ZVRhZ3NTaW11bGF0aW9uUHJvcHMge1xuICAvKipcbiAgICAgKiAoT3B0aW9uYWwpIFVzZSB0aGlzIGFzIGEgaG9vayB0byBpbmplY3QgYW4gYWx0ZXJuYXRlIElBd3NJbnZva2VyIChmb3IgbW9ja2luZyB0aGUgQVdTIEFQSSBjYWxsKS5cbiAgICAgKiBAZGVmYXVsdCAtIHdpbGwgcGVyZm9ybSBhIHJlYWwgaW52b2NhdGlvbiBvZiB0aGUgSmF2YVNjcmlwdCBBV1MgU0RLIHVzaW5nIFJlZmxlY3RpdmVBd3NJbnZva2VyIGNsYXNzLlxuICAgICAqL1xuICByZWFkb25seSBhd3NJbnZva2VyOiBJQXdzSW52b2tlcjtcblxuICAvKipcbiAgICAgKiAoT3B0aW9uYWwpIFdoZXRoZXIgdG8gcmVhbGx5IHBlcmZvcm0gYSBwYXVzZSBvZiB0aGUgcnVudGltZS5cbiAgICAgKiBUbyBvdmVycmlkZSBzbGVlcCBiZWhhdmlvciwgaW5qZWN0IGFuIElTbGVlcEhvb2sgaW1wbCBvciB1c2UgdGhlIHByb3ZpZGVkIE1vY2tTbGVlcCBjbGFzcy5cbiAgICAgKiBAZGVmYXVsdCBTbGVlcGVySW1wbFxuICAgICAqL1xuICByZWFkb25seSBzbGVlcEhvb2s6IElTbGVlcEhvb2s7XG59XG5cbmludGVyZmFjZSBFYzJUYWdGaWx0ZXIge1xuICBOYW1lOiBzdHJpbmc7XG4gIFZhbHVlczogc3RyaW5nW107XG59XG5cbmludGVyZmFjZSBFYzJUYWdEZXNjcmlwdGlvbiB7XG4gIEtleTogc3RyaW5nO1xuICBWYWx1ZTogc3RyaW5nO1xuICBSZXNvdXJjZUlkOiBzdHJpbmc7XG59XG5cbmludGVyZmFjZSBTc21UYWcge1xuICBLZXk6IHN0cmluZztcbiAgVmFsdWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBBdXRvbWF0aW9uU3RlcCBpbXBsZW1lbmF0aW9uIGZvciBhd3M6Y3JlYXRlVGFnc1xuICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N5c3RlbXMtbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL2F1dG9tYXRpb24tYWN0aW9uLWNyZWF0ZXRhZy5odG1sXG4gKi9cbmV4cG9ydCBjbGFzcyBDcmVhdGVUYWdzU2ltdWxhdGlvbiBleHRlbmRzIEF1dG9tYXRpb25TaW11bGF0aW9uQmFzZSB7XG4gIHJlYWRvbmx5IGFjdGlvbiA9ICdhd3M6Y3JlYXRlVGFncyc7XG4gIHByaXZhdGUgY3JlYXRlVGFnc1N0ZXA6IENyZWF0ZVRhZ3NTdGVwO1xuICBwcml2YXRlIHByb3BzOiBDcmVhdGVUYWdzU2ltdWxhdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHN0ZXA6IENyZWF0ZVRhZ3NTdGVwLCBwcm9wczogQ3JlYXRlVGFnc1NpbXVsYXRpb25Qcm9wcykge1xuICAgIHN1cGVyKHN0ZXApO1xuICAgIHRoaXMuY3JlYXRlVGFnc1N0ZXAgPSBzdGVwO1xuICAgIHRoaXMucHJvcHMgPSBwcm9wcztcbiAgfVxuXG4gIHB1YmxpYyBleGVjdXRlU3RlcChpbnB1dHM6IFJlY29yZDxzdHJpbmcsIGFueT4pOiBSZWNvcmQ8c3RyaW5nLCBhbnk+IHtcbiAgICBjb25zdCByZXNvdXJjZVR5cGUgPSB0aGlzLmNyZWF0ZVRhZ3NTdGVwLnJlc291cmNlVHlwZT8ucmVzb2x2ZVRvU3RyaW5nKGlucHV0cykgPz8gJ0VDMic7XG4gICAgc3dpdGNoIChyZXNvdXJjZVR5cGUpIHtcbiAgICAgIGNhc2UgJ0VDMic6XG4gICAgICAgIHRoaXMuY3JlYXRlRWMyVGFncyhpbnB1dHMpO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgJ01hbmFnZWRJbnN0YW5jZSc6XG4gICAgICBjYXNlICdNYWludGVuYW5jZVdpbmRvdyc6XG4gICAgICBjYXNlICdQYXJhbWV0ZXInOlxuICAgICAgICB0aGlzLmNyZWF0ZVNzbVRhZ3MocmVzb3VyY2VUeXBlLCBpbnB1dHMpO1xuICAgICAgICBicmVhaztcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgJHtyZXNvdXJjZVR5cGV9IGlzIG5vdCBhIHZhbGlkIHJlc291cmNlIHR5cGUgZm9yIHRhZ2dpbmdgKTtcbiAgICB9XG4gICAgcmV0dXJuIHt9O1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVFYzJUYWdzKGlucHV0czogUmVjb3JkPHN0cmluZywgYW55Pik6IHZvaWQge1xuICAgIGNvbnN0IHJlc291cmNlSWRzID0gdGhpcy5jcmVhdGVUYWdzU3RlcC5yZXNvdXJjZUlkcy5yZXNvbHZlVG9TdHJpbmdMaXN0KGlucHV0cyk7XG4gICAgY29uc3QgdGFncyA9IHRoaXMuY3JlYXRlVGFnc1N0ZXAudGFncy5yZXNvbHZlVG9NYXBMaXN0KGlucHV0cyk7XG5cbiAgICBjb25zb2xlLmxvZygnQ3JlYXRlVGFnczogQ3JlYXRlIEVDMiB0YWdzIGZvciByZXNvdXJjZXMnKTtcbiAgICB0aGlzLmNhbGxFYzJDcmVhdGVUYWdzKHJlc291cmNlSWRzLCB0YWdzKTtcbiAgICBjb25zb2xlLmxvZygnQ3JlYXRlVGFnczogV2FpdCBmb3IgdGFncyB0byBiZSBjb21wbGV0ZWQnKTtcbiAgICB0aGlzLndhaXRGb3JFYzJUYWdzKHJlc291cmNlSWRzLCB0YWdzKTtcbiAgICBjb25zb2xlLmxvZygnQ3JlYXRlVGFnczogRUMyIHRhZ3MgYXJlIGFzc29jaWF0ZWQgdG8gdGhlIHJlc291cmNlcycpO1xuICB9XG5cbiAgcHJpdmF0ZSBjYWxsRWMyQ3JlYXRlVGFncyhyZXNvdXJjZUlkczogc3RyaW5nW10sIHRhZ3M6IFJlY29yZDxzdHJpbmcsIGFueT5bXSk6IHZvaWQge1xuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBBdXRvbWF0aW9uU3RlcFNpbXVsYXRpb24obmV3IEF3c0FwaVN0ZXAobmV3IFN0YWNrKCksICdlYzJDcmVhdGVUYWdzJywge1xuICAgICAgc2VydmljZTogQXdzU2VydmljZS5FQzIsXG4gICAgICBwYXNjYWxDYXNlQXBpOiAnQ3JlYXRlVGFncycsXG4gICAgICBhcGlQYXJhbXM6IHtcbiAgICAgICAgUmVzb3VyY2VzOiByZXNvdXJjZUlkcyxcbiAgICAgICAgVGFnczogdGFncyxcbiAgICAgIH0sXG4gICAgICBvdXRwdXRzOiBbXSxcbiAgICB9KSwgdGhpcy5wcm9wcykuaW52b2tlKHt9KTtcbiAgICBpZiAocmVzdWx0LnJlc3BvbnNlQ29kZSAhPT0gUmVzcG9uc2VDb2RlLlNVQ0NFU1MpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ3JlYXRlIEVDMiB0YWdzIGZhaWxlZDogJHtyZXN1bHQuc3RhY2tUcmFjZX1gKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHdhaXRGb3JFYzJUYWdzKHJlc291cmNlSWRzOiBzdHJpbmdbXSwgdGFnczogUmVjb3JkPHN0cmluZywgYW55PltdKTogdm9pZCB7XG4gICAgd2hpbGUgKCF0aGlzLmFyZUVjMlRhZ3NBc3NvY2lhdGVkVG9UaGVSZXNvdXJjZXMocmVzb3VyY2VJZHMsIHRhZ3MpKSB7XG4gICAgICBuZXcgQXV0b21hdGlvblN0ZXBTaW11bGF0aW9uKG5ldyBTbGVlcFN0ZXAobmV3IFN0YWNrKCksICdzbGVlcCcsIHtcbiAgICAgICAgc2xlZXBTZWNvbmRzOiAyLFxuICAgICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBhcmVFYzJUYWdzQXNzb2NpYXRlZFRvVGhlUmVzb3VyY2VzKHJlc291cmNlSWRzOiBzdHJpbmdbXSwgdGFnczogUmVjb3JkPHN0cmluZywgYW55PltdKTogYm9vbGVhbiB7XG4gICAgY29uc3QgdGFnS2V5czogc3RyaW5nW10gPSB0YWdzLm1hcCh0YWcgPT4gdGFnLktleSk7XG4gICAgY29uc3QgdGFnVmFsdWVzOiBzdHJpbmdbXSA9IHRhZ3MubWFwKHRhZyA9PiB0YWcuVmFsdWUpO1xuXG4gICAgY29uc3QgZmlsdGVycyA9IHRoaXMuYnVpbGRFYzJUYWdGaWx0ZXIocmVzb3VyY2VJZHMsIHRhZ0tleXMsIHRhZ1ZhbHVlcyk7XG4gICAgY29uc3QgdGFnRGVzY3JpcHRpb25zID0gdGhpcy5lYzJEZXNjcmliZVRhZ3MoZmlsdGVycyk7XG5cbiAgICBmb3IgKGNvbnN0IHRhZyBvZiB0YWdzKSB7XG4gICAgICBjb25zdCBkZXNjcmlwdGlvbnNPZlRhZyA9IHRhZ0Rlc2NyaXB0aW9ucy5maWx0ZXIoeCA9PiB4LktleSA9PT0gdGFnLktleSAmJiB4LlZhbHVlID09PSB0YWcuVmFsdWUpO1xuICAgICAgY29uc3QgZGVzY3JpYmVkUmVzb3VyY2VJZHMgPSBuZXcgU2V0KGRlc2NyaXB0aW9uc09mVGFnLm1hcCh4ID0+IHguUmVzb3VyY2VJZCkpO1xuXG4gICAgICBpZiAoIXJlc291cmNlSWRzLmV2ZXJ5KHggPT4gZGVzY3JpYmVkUmVzb3VyY2VJZHMuaGFzKHgpKSkge1xuICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkRWMyVGFnRmlsdGVyKHJlc291cmNlSWRzOiBzdHJpbmdbXSwgdGFnS2V5czogc3RyaW5nW10sIHRhZ1ZhbHVlczogc3RyaW5nW10pOiBFYzJUYWdGaWx0ZXJbXSB7XG4gICAgY29uc3QgcG90ZW50aWFsRmlsdGVyczogUmVjb3JkPHN0cmluZywgc3RyaW5nW10+ID0ge1xuICAgICAgJ3Jlc291cmNlLWlkJzogcmVzb3VyY2VJZHMsXG4gICAgICAna2V5JzogdGFnS2V5cyxcbiAgICAgICd2YWx1ZSc6IHRhZ1ZhbHVlcyxcbiAgICB9O1xuICAgIGNvbnN0IGZpbHRlcnM6IEVjMlRhZ0ZpbHRlcltdID0gW107XG4gICAgZm9yIChjb25zdCBuYW1lIG9mIE9iamVjdC5rZXlzKHBvdGVudGlhbEZpbHRlcnMpKSB7XG4gICAgICBpZiAocG90ZW50aWFsRmlsdGVyc1tuYW1lXS5sZW5ndGggIT09IDApIHtcbiAgICAgICAgZmlsdGVycy5wdXNoKHsgTmFtZTogbmFtZSwgVmFsdWVzOiBwb3RlbnRpYWxGaWx0ZXJzW25hbWVdIH0pO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gZmlsdGVycztcbiAgfVxuXG4gIHByaXZhdGUgZWMyRGVzY3JpYmVUYWdzKGZpbHRlcnM6IEVjMlRhZ0ZpbHRlcltdKTogRWMyVGFnRGVzY3JpcHRpb25bXSB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbihuZXcgQXdzQXBpU3RlcChuZXcgU3RhY2soKSwgJ2VjMkRlc2NyaWJlVGFncycsIHtcbiAgICAgIHNlcnZpY2U6IEF3c1NlcnZpY2UuRUMyLFxuICAgICAgcGFzY2FsQ2FzZUFwaTogJ0Rlc2NyaWJlVGFncycsXG4gICAgICBhcGlQYXJhbXM6IHtcbiAgICAgICAgRmlsdGVyczogZmlsdGVycyxcbiAgICAgIH0sXG4gICAgICBvdXRwdXRzOiBbe1xuICAgICAgICBvdXRwdXRUeXBlOiBEYXRhVHlwZUVudW0uTUFQX0xJU1QsXG4gICAgICAgIHNlbGVjdG9yOiAnJC5UYWdzJyxcbiAgICAgICAgbmFtZTogJ1RhZ3MnLFxuICAgICAgfV0sXG4gICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgaWYgKHJlc3VsdC5yZXNwb25zZUNvZGUgIT09IFJlc3BvbnNlQ29kZS5TVUNDRVNTKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYERlc2NyaWJlIEVDMiB0YWdzIGZhaWxlZDogJHtyZXN1bHQuc3RhY2tUcmFjZX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIHJlc3VsdC5vdXRwdXRzPy5bJ2VjMkRlc2NyaWJlVGFncy5UYWdzJ10gPz8gW107XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVNzbVRhZ3MocmVzb3VyY2VUeXBlOiBzdHJpbmcsIGlucHV0czogUmVjb3JkPHN0cmluZywgYW55Pik6IHZvaWQge1xuICAgIGNvbnN0IHJlc291cmNlSWQgPSB0aGlzLmdldFNpbmdsZVJlc291cmNlSWQocmVzb3VyY2VUeXBlLCBpbnB1dHMpO1xuICAgIGNvbnN0IHRhZ3MgPSB0aGlzLmNyZWF0ZVRhZ3NTdGVwLnRhZ3MucmVzb2x2ZVRvTWFwTGlzdChpbnB1dHMpO1xuXG4gICAgY29uc29sZS5sb2coJ0NyZWF0ZVRhZ3M6IENyZWF0ZSB0YWdzIGZvciByZXNvdXJjZScpO1xuICAgIHRoaXMuY2FsbFNzbUFkZFRhZ3NUb1Jlc291cmNlKHJlc291cmNlVHlwZSwgcmVzb3VyY2VJZCwgdGFncyk7XG4gICAgY29uc29sZS5sb2coJ0NyZWF0ZVRhZ3M6IFdhaXQgZm9yIHRhZ3MgdG8gYmUgY29tcGxldGVkJyk7XG4gICAgdGhpcy53YWl0Rm9yU3NtVGFncyhyZXNvdXJjZVR5cGUsIHJlc291cmNlSWQsIHRhZ3MpO1xuICAgIGNvbnNvbGUubG9nKCdDcmVhdGVUYWdzOiBUYWdzIGFyZSBhc3NvY2lhdGVkIHRvIHRoZSByZXNvdXJjZScpO1xuICB9XG5cbiAgcHJpdmF0ZSBjYWxsU3NtQWRkVGFnc1RvUmVzb3VyY2UocmVzb3VyY2VUeXBlOiBzdHJpbmcsIHJlc291cmNlSWQ6IHN0cmluZywgdGFnczogUmVjb3JkPHN0cmluZywgYW55PltdKTogdm9pZCB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbihuZXcgQXdzQXBpU3RlcChuZXcgU3RhY2soKSwgJ3NzbUFkZFRhZ3NUb1Jlc291cmNlJywge1xuICAgICAgc2VydmljZTogQXdzU2VydmljZS5TU00sXG4gICAgICBwYXNjYWxDYXNlQXBpOiAnQWRkVGFnc1RvUmVzb3VyY2UnLFxuICAgICAgYXBpUGFyYW1zOiB7XG4gICAgICAgIFJlc291cmNlVHlwZTogcmVzb3VyY2VUeXBlLFxuICAgICAgICBSZXNvdXJjZUlkOiByZXNvdXJjZUlkLFxuICAgICAgICBUYWdzOiB0YWdzLFxuICAgICAgfSxcbiAgICAgIG91dHB1dHM6IFtdLFxuICAgIH0pLCB0aGlzLnByb3BzKS5pbnZva2Uoe30pO1xuICAgIGlmIChyZXN1bHQucmVzcG9uc2VDb2RlICE9PSBSZXNwb25zZUNvZGUuU1VDQ0VTUykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBZGQgdGFncyBmYWlsZWQ6ICR7cmVzdWx0LnN0YWNrVHJhY2V9YCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSB3YWl0Rm9yU3NtVGFncyhyZXNvdXJjZVR5cGU6IHN0cmluZywgcmVzb3VyY2VJZDogc3RyaW5nLCB0YWdzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+W10pOiB2b2lkIHtcbiAgICB3aGlsZSAoIXRoaXMuYXJlQWxsU3NtVGFnc0xpc3RlZChyZXNvdXJjZVR5cGUsIHJlc291cmNlSWQsIHRhZ3MpKSB7XG4gICAgICBuZXcgQXV0b21hdGlvblN0ZXBTaW11bGF0aW9uKG5ldyBTbGVlcFN0ZXAobmV3IFN0YWNrKCksICdzbGVlcCcsIHtcbiAgICAgICAgc2xlZXBTZWNvbmRzOiAyLFxuICAgICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBhcmVBbGxTc21UYWdzTGlzdGVkKHJlc291cmNlVHlwZTogc3RyaW5nLCByZXNvdXJjZUlkOiBzdHJpbmcsIHRhZ3M6IFJlY29yZDxzdHJpbmcsIGFueT5bXSk6IGJvb2xlYW4ge1xuICAgIGNvbnN0IGxpc3RlZFRhZ3MgPSB0aGlzLnNzbUxpc3RUYWdzRm9yUmVzb3VyY2UocmVzb3VyY2VUeXBlLCByZXNvdXJjZUlkKTtcbiAgICBjb25zdCBsaXN0ZWRUYWdNYXAgPSBuZXcgTWFwKGxpc3RlZFRhZ3MubWFwKHRhZyA9PiBbdGFnLktleSwgdGFnLlZhbHVlXSkpO1xuICAgIHJldHVybiB0YWdzLmV2ZXJ5KHRhZyA9PiBsaXN0ZWRUYWdNYXAuZ2V0KHRhZy5LZXkpID09PSB0YWcuVmFsdWUpO1xuICB9XG5cbiAgcHJpdmF0ZSBzc21MaXN0VGFnc0ZvclJlc291cmNlKHJlc291cmNlVHlwZTogc3RyaW5nLCByZXNvdXJjZUlkOiBzdHJpbmcpOiBTc21UYWdbXSB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbihuZXcgQXdzQXBpU3RlcChuZXcgU3RhY2soKSwgJ3NzbUxpc3RUYWdzRm9yUmVzb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlOiBBd3NTZXJ2aWNlLlNTTSxcbiAgICAgIHBhc2NhbENhc2VBcGk6ICdMaXN0VGFnc0ZvclJlc291cmNlJyxcbiAgICAgIGFwaVBhcmFtczoge1xuICAgICAgICBSZXNvdXJjZVR5cGU6IHJlc291cmNlVHlwZSxcbiAgICAgICAgUmVzb3VyY2VJZDogcmVzb3VyY2VJZCxcbiAgICAgIH0sXG4gICAgICBvdXRwdXRzOiBbe1xuICAgICAgICBvdXRwdXRUeXBlOiBEYXRhVHlwZUVudW0uTUFQX0xJU1QsXG4gICAgICAgIHNlbGVjdG9yOiAnJC5UYWdMaXN0JyxcbiAgICAgICAgbmFtZTogJ1RhZ3MnLFxuICAgICAgfV0sXG4gICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgaWYgKHJlc3VsdC5yZXNwb25zZUNvZGUgIT09IFJlc3BvbnNlQ29kZS5TVUNDRVNTKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFNTTSBMaXN0IHRhZ3MgZmFpbGVkOiAke3Jlc3VsdC5zdGFja1RyYWNlfWApO1xuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0Lm91dHB1dHM/Llsnc3NtTGlzdFRhZ3NGb3JSZXNvdXJjZS5UYWdzJ10gPz8gW107XG4gIH1cblxuICBwcml2YXRlIGdldFNpbmdsZVJlc291cmNlSWQocmVzb3VyY2VUeXBlOiBzdHJpbmcsIGlucHV0czogUmVjb3JkPHN0cmluZywgYW55Pik6IHN0cmluZyB7XG4gICAgY29uc3QgcmVzb3VyY2VJZHMgPSB0aGlzLmNyZWF0ZVRhZ3NTdGVwLnJlc291cmNlSWRzLnJlc29sdmUoaW5wdXRzKTtcbiAgICBpZiAocmVzb3VyY2VJZHMubGVuZ3RoICE9PSAxKSB7IHRocm93IG5ldyBFcnJvcihgT25seSAxIHJlc291cmNlIGFsbG93ZWQgZm9yIHR5cGUgJHtyZXNvdXJjZVR5cGV9YCk7IH1cbiAgICByZXR1cm4gcmVzb3VyY2VJZHNbMF07XG4gIH1cblxufSJdfQ==