"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmDocument = exports.DocumentFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const constructs_1 = require("constructs");
const string_variable_1 = require("../interface/variables/string-variable");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const yaml = require('js-yaml');
var DocumentFormat;
(function (DocumentFormat) {
    DocumentFormat[DocumentFormat["JSON"] = 0] = "JSON";
    DocumentFormat[DocumentFormat["YAML"] = 1] = "YAML";
})(DocumentFormat = exports.DocumentFormat || (exports.DocumentFormat = {}));
class SsmDocument extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.documentName = props.documentName ?? id;
        this.description = props.description ?? this.documentName;
        this.header = props.header;
        this.assumeRole = props.assumeRole;
        this.docOutputs = props.docOutputs ?? [];
        this.docInputs = props.docInputs ?? [];
        if (this.assumeRole && this.assumeRole instanceof string_variable_1.StringVariable &&
            !this.docInputs.map(i => i.name).includes(this.assumeRole.reference)) {
            throw new Error('Assume role specified but not provided in inputs: ' + this.assumeRole);
        }
        this.props = props;
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        this.cfnDocument = new aws_ssm_1.CfnDocument(this, 'Resource', {
            ...this.props,
            ...{
                content: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const doc = this.buildSsmDocument();
                        return JSON.parse(JSON.stringify(doc));
                    },
                }),
                name: this.documentName,
                documentFormat: isYaml ? 'YAML' : 'JSON',
                documentType: this.documentType(),
                tags: [{ key: 'CdkGenerated', value: 'true' }, ...(this.props.tags || [])],
            },
        });
    }
    /**
     * The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.
     * This function builds an object containing the relevant (declared) input parameters.
     * The return object will be used to build the yaml/json representation of the document.
     */
    formatInputs() {
        const ssmInputs = {};
        this.docInputs.forEach(inp => {
            ssmInputs[inp.name] = inp.toSsm();
        });
        return ssmInputs;
    }
    /**
       * Synthesize before calling this function!
       * You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);
       *
       * Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
       * The format is dependency on the documentFormat property provided to the class.
       * The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().
       * @returns a string representation of this document as an SSM formatted yaml/json.
       */
    print() {
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        // format as JSON first so that the JSON printing goes into effect
        const doc = JSON.parse(JSON.stringify(this.buildSsmDocument()));
        if (isYaml) {
            // Prepend the document with the header if defined
            return this.headerWithComments() + yaml.dump(doc);
        }
        else {
            // There is no way to provide the header comment in a json document
            return JSON.stringify(doc);
        }
    }
    /**
       * Prepend header line with # if it does not start with #.
       * If no header is provided, this method will return an emptry string.
       * Headers are only available in yaml format.
       */
    headerWithComments() {
        if (this.header == undefined) {
            return '';
        }
        return this.header.split(os_1.EOL).map(line => line.trim().startsWith('#') ? line : '# ' + line).join(os_1.EOL) +
            // "---" demarks the end of the header and the start of the document.
            '\n---\n';
    }
}
exports.SsmDocument = SsmDocument;
_a = JSII_RTTI_SYMBOL_1;
SsmDocument[_a] = { fqn: "@cdklabs/cdk-ssm-documents.SsmDocument", version: "0.0.29" };
//# sourceMappingURL=data:application/json;base64,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