"""
Tests to check the capacitance model_threshold_1 works for double dot arrays
"""

import unittest

import matplotlib.pyplot as plt
import numpy as np

from qarray import (ground_state_open_rust, ground_state_closed_rust, ground_state_open_python,
                    ground_state_closed_python, optimal_Vg, compute_threshold)
from qarray.jax_core import ground_state_open_jax, ground_state_closed_jax
from .GLOBAL_OPTIONS import N_ITERATIONS, N_VOLTAGES
from .helper_functions import randomly_generate_matrices, too_different


class DoubleDotTests(unittest.TestCase):
    def test_python_vs_rust_open(self):
        """
        Test that the python and rust open double dot ground state functions return the same result.

        The threshold is set to 1, so every nearest neighbour change state is considered
        """

        for _ in range(N_ITERATIONS):
            cdd, cdd_inv, cgd = randomly_generate_matrices(4)
            vg = np.random.uniform(-5, 5, size=(N_VOLTAGES, 4))
            n_rust = ground_state_open_rust(vg, cgd, cdd_inv, 1)
            n_python = ground_state_open_python(vg, cgd, cdd_inv, 1)
            n_jax = ground_state_open_jax(vg, cgd, cdd_inv)

            debug = False
            if debug:
                if not np.allclose(n_rust, n_python, n_jax):
                    print(cdd_inv)
                    fig, ax = plt.subplots(1, 4)
                    ax[0].imshow(n_rust, aspect='auto')
                    ax[1].imshow(n_python, aspect='auto')
                    ax[2].imshow(n_rust - n_python, aspect='auto')
                    plt.show()

            self.assertFalse(too_different(n_rust, n_python))
            self.assertFalse(too_different(n_rust, n_jax))

    def test_threshold(self):
        """
        Test that the python and rust open double dot ground state functions return the same result.

        The threshold is set to 1, so every nearest neighbour change state is considered
        """

        for _ in range(N_ITERATIONS):
            cdd, cdd_inv, cgd = randomly_generate_matrices(4)
            vg = np.random.uniform(-10, 5, size=(N_VOLTAGES, 4))
            n_threshold_of_1 = ground_state_open_rust(vg, cgd, cdd_inv, 1.)

            threshold = compute_threshold(cdd)
            n_threshold_not_of_1 = ground_state_open_rust(vg, cgd, cdd_inv, threshold)

            self.assertTrue(np.allclose(n_threshold_of_1, n_threshold_not_of_1),
                            msg=f"threshold {threshold}"
                            )

    def test_python_vs_rust_one_charge(self):
        """
        Test that the python and rust isolated double dot array with one change in it ground state functions return the
        same result.

        The threshold is set to 1, so every nearest neighbour change state is considered
        """

        for _ in range(N_ITERATIONS):
            cdd, cdd_inv, cgd = randomly_generate_matrices(4)
            vg = np.random.uniform(-5, 5, size=(N_VOLTAGES, 4))
            n_rust = ground_state_closed_rust(vg, 1, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd, threshold=1)
            n_python = ground_state_closed_python(vg, 1, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd, threshold=1)
            n_jax = ground_state_closed_jax(vg, n_charge=1, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd)

            self.assertFalse(too_different(n_rust, n_python))
            self.assertFalse(too_different(n_rust, n_jax))

    def test_python_vs_rust_two_charge(self):
        """
        Test that the python and rust isolated double dot array with two change in it ground state functions return the same result.

        The threshold is set to 1, so every nearest neighbour change state is considered
        """

        for _ in range(N_ITERATIONS):
            cdd, cdd_inv, cgd = randomly_generate_matrices(4)
            vg = np.random.uniform(-5, 5, size=(N_VOLTAGES, 4))
            n_rust = ground_state_closed_rust(vg, 2, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd, threshold=1)
            n_python = ground_state_closed_python(vg, 2, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd, threshold=1)
            n_jax = ground_state_closed_jax(vg, n_charge=2, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd)

            self.assertFalse(too_different(n_rust, n_python))
            self.assertFalse(too_different(n_rust, n_jax))

    def test_python_vs_rust_three_charge(self):
        """
        Test that the python and rust isolated double dot array with three changes in it ground state functions return the same result.

        The threshold is set to 1, so every nearest neighbour change state is considered
        """

        for _ in range(N_ITERATIONS):
            cdd, cdd_inv, cgd = randomly_generate_matrices(4)
            vg = np.random.uniform(-5, 5, size=(N_VOLTAGES, 4))
            n_rust = ground_state_closed_rust(vg, 4, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd, threshold=1)
            n_python = ground_state_closed_python(vg, 4, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd, threshold=1)

            n_jax = ground_state_closed_jax(vg, n_charge=4, cdd=cdd, cdd_inv=cdd_inv, cgd=cgd)

            self.assertFalse(too_different(n_rust, n_python))
            self.assertFalse(too_different(n_rust, n_jax))

    def test_optimal_vg(self):
        """

        Test of the optimal dot voltage function which computes the dot voltages which minimise the free energy of a
        particular change configuration. For double quantum dots with two gates this means that change state will be the
        ground state.
        """

        for _ in range(N_ITERATIONS):
            cdd, cdd_inv, cgd = randomly_generate_matrices(4)
            n_charges = np.random.choice(np.arange(0, 10), size=(N_VOLTAGES, 4)).astype(int)
            vg = optimal_Vg(cdd_inv=cdd_inv, cgd=cgd, n_charges=n_charges)
            n_rust = ground_state_open_rust(vg, cgd, cdd_inv, 1).astype(int)
            self.assertTrue(np.allclose(n_rust, n_charges))


if __name__ == '__main__':
    unittest.main()
