# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['idex_sdk',
 'idex_sdk.client',
 'idex_sdk.client.order_book',
 'idex_sdk.client.rest',
 'idex_sdk.client.websocket',
 'idex_sdk.idex_types',
 'idex_sdk.idex_types.rest',
 'idex_sdk.idex_types.websocket',
 'idex_sdk.order_book']

package_data = \
{'': ['*']}

install_requires = \
['pyee>=9.0.4,<10.0.0', 'web3>=5.30.0,<6.0.0', 'websockets==9.1']

setup_kwargs = {
    'name': 'idex-sdk',
    'version': '1.0.0b2',
    'description': 'IDEX v3 SDK for Python',
    'long_description': '# <img src="https://github.com/idexio/idex-sdk-python/raw/main/assets/logo.png" alt="IDEX" height="37px" valign="top"> Python SDK\n\n![Discord](https://img.shields.io/discord/455246457465733130?label=Discord&style=flat-square)\n![GitHub](https://img.shields.io/github/license/idexio/idex-sdk-js?style=flat-square)\n![npm](https://img.shields.io/pypi/v/idex-sdk?style=flat-square)\n![GitHub issues](https://img.shields.io/github/issues/idexio/idex-sdk-python?style=flat-square)\n\n\n![Twitter Follow](https://img.shields.io/twitter/follow/idexio?style=social)\n\nThe official library for [IDEX v3\'s](https://idex.io) REST and WebSocket APIs, featuring a real time order book implementation with support for [hybrid liquidity](https://api-docs-v3.idex.io/#hybrid-liquidity).\n\nComplete documentation for the IDEX v3 API is available at https://api-docs-v3.idex.io. \n\n**The IDEX Python SDK is currently in beta.**\n## Installation\n\n```bash\npip install idex-sdk\n```\n\n## Getting Started\n\n- Sign up for [API keys](https://exchange.idex.io/user/signup). Market data endpoints do not require an account.\n- A fully-functional [testnet sandbox](https://api-docs-v3.idex.io/#sandbox) is available for development.\n- In-depth usage documentation by endpoint is [available here](docs.md).\n\n## Usage Examples\n\n### Public REST API Client\n\n```python\nfrom idex_sdk.client.rest.public import RestPublicClient\n\nc = RestPublicClient()\nexchange_info = c.get_exchange_info()\nassets = c.get_assets()\nmarkets = c.get_markets()\ncandles = c.get_candles({"market": "ETH-USDC", "interval": "1h"})\norder_book_l2 = c.get_order_book_level2("ETH-USDC")\n\n```\n\n### Authenticated REST API Client\n\n```python\nfrom uuid import uuid1\n\nfrom idex_sdk.client.rest.authenticated import RestAuthenticatedClient\n\ndef uuid() -> str:\n    return str(uuid1())\n\nc = RestAuthenticatedClient(\n    api_key="<API key>",\n    api_secret="<API secret>",\n    wallet_private_key="<wallet private key>",\n)\nwallet_address = "<wallet address>"\n\nget_balances = c.get_balances(\n    {\n        "nonce": uuid(),\n        "wallet": wallet_address,\n    }\n)\n\nnew_order = c.create_order(\n    {\n        "nonce": uuid(),\n        "wallet": wallet_address,\n        "market": "ETH-USDC",\n        "type": "limit",\n        "side": "buy",\n        "quantity": "1.00000000",\n        "price": "1000.00000000",\n    }\n)\n```\n\n### Real Time Order Book Client\n\n```python\nimport asyncio\n\nfrom idex_sdk.client.order_book.real_time import OrderBookRealTimeClient\n\ndef update_l2_order_book(market: str) -> None:\n    real_time_order_book = client.get_order_book_l2(market, 10)\n    print(real_time_order_book)\n\ndef make_client() -> OrderBookRealTimeClient:\n    client = OrderBookRealTimeClient()\n    client.on("l2", update_l2_order_book)\n    client.on("ready", update_l2_order_book)\n    client.on("error", lambda error: print(f"error {error}"))\n    return client\n\nclient = make_client()\n\nasync def test() -> None:\n    task = asyncio.create_task(client.start(["IDEX-USDC"]))\n    await task\n\nif __name__ == "__main__":\n    asyncio.run(test())\n```\n\n## About the Python SDK\n\nThis is a python conversion of the [IDEX Typescript SDK](https://github.com/idexio/idex-sdk-js). There is a conversion of all functionality, including typing with `mypy`. There are some small differences in how the `OrderBookRealTimeClient` is run by external code due to differences in how Python\'s `asyncio` library handles asynchronous code compares to Javascript. See the example code below for more information.\n\n\n### Setup Repository\n\n- Ensure python3 and [poetry](https://python-poetry.org/docs/) are installed.\n- Clone repo\n- Cd into repo directory\n- Run `poetry install`\n\nThe project is now installed to a virtual environment at `./.venv`. Use this environment in the terminal by running `poetry shell`, or by pointing your IDE to it. For example, loading the project directory in VSCode with the Python extension installed should automatically prompt to select the virtual environment.\n\nThe project includes configuration (in `pyproject.toml`) for type-checking wih `mypy`, debugging with `debugpy` (for IDEs like neovim), linting with `flake8`, auto-formatting with `black`, and sorting imports with `isort`. If you configure your IDE appropriately, all of this will be automatic. An example VSCode configuration is included.\n\n## Testing\nUnit tests are available (using [python unittest](https://docs.python.org/3/library/unittest.html)) for logic-heavy functionality. To run them, run `make test`.\n\n## Generating docs\n\n```\npydoc-markdown -I idex_sdk --render-toc > docs.md\n```\n\n## Contract ABIs\n\nIncluded in the `contracts/` directory contains the ABIs necessary for interacting with IDEX v3\'s smart contracts.\n\n- The [Exchange ABI](contracts/Exchange.abi.json) can be used to query contract state, [deposit funds](https://api-docs-v3.idex.io/#deposit-funds), [add liquidity](https://api-docs-v3.idex.io/#add-liquidity-via-smart-contract-function-call), [remove liquidity](https://api-docs-v3.idex.io/#remove-liquidity-via-smart-contract-function-call) or [exit wallets](https://api-docs-v3.idex.io/#exit-wallet).\n- The [FaucetToken ABI](contracts/FaucetToken.abi.json) is implemented by the [API sandbox](https://api-docs-v3.idex.io/#sandbox) testnet tokens and features a [faucet](https://api-docs-v3.idex.io/#faucets)\nfunction for dispensing tokens.\n\n## Getting Help\n\nPlease reach out in the [#developers Discord](https://discord.com/invite/UHAGGBz) for help, or [open an issue](https://github.com/idexio/idex-sdk-python/issues) to report bugs.\n## License\n\nThe IDEX Python SDK is released under the [MIT License](https://opensource.org/licenses/MIT).\n',
    'author': 'IDEX',
    'author_email': 'support@idex.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://idex.io/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>3.7.1,<3.10',
}


setup(**setup_kwargs)
