# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ledgerautosync']

package_data = \
{'': ['*']}

install_requires = \
['ofxclient>=2.0.3,<3.0.0', 'ofxparse>=0.21,<0.22']

entry_points = \
{'console_scripts': ['hledger-autosync = ledgerautosync.cli:run',
                     'ledger-autosync = ledgerautosync.cli:run']}

setup_kwargs = {
    'name': 'ledger-autosync',
    'version': '1.0.3',
    'description': "Automatically sync your bank's data with ledger",
    'long_description': 'ledger-autosync\n===============\n\n.. image:: https://travis-ci.org/egh/ledger-autosync.svg?branch=master\n  :target: https://travis-ci.org/egh/ledger-autosync\n\nledger-autosync is a program to pull down transactions from your bank\nand create `ledger <http://ledger-cli.org/>`__ transactions for them.\nIt is designed to only create transactions that are not already\npresent in your ledger files (that is, it will deduplicate\ntransactions). This should make it comparable to some of the automated\nsynchronization features available in products like GnuCash, Mint,\netc. In fact, ledger-autosync performs OFX import and synchronization\nbetter than all the alternatives I have seen.\n\nNews\n----\n\nv1.0.0\n~~~~~~\n\nVersions of ledger-autosync before 1.0.0 printed the ofxid in a\nslightly incorrect position. This should not effect usage of the\nprogram, but if you would like to correct the error, see below for\nmore details.\n\nFeatures\n--------\n\n-  supports `ledger <http://ledger-cli.org/>`__ 3 and\n   `hledger <http://hledger.org/>`__\n-  like ledger, ledger-autosync will never modify your files directly\n-  interactive banking setup via\n   `ofxclient <https://github.com/captin411/ofxclient>`__\n-  multiple banks and accounts\n-  support for non-US currencies\n-  support for 401k and investment accounts\n\n   -  tracks investments by share, not dollar value\n   -  support for complex transaction types, including transfers, buys,\n      sells, etc.\n\n-  import of downloaded OFX files, for banks not supporting automatic\n   download\n-  import of downloaded CSV files from Paypal, Simple, Amazon and Mint\n-  any CSV file can be supported via plugins\n\nPlatforms\n---------\n\nledger-autosync is developed on Linux with ledger 3 and python 3; it\nhas been tested on Windows (although it will run slower) and should\nrun on OS X. It requires ledger 3 or hledger, but it should run faster\nwith ledger, because it will not need to start a command to check\nevery transaction.\n\n\nQuickstart\n----------\n\nInstallation\n~~~~~~~~~~~~\n\nIf you are on Debian or Ubuntu, an (older) version of ledger-autosync\nshould be available for installation. Try:\n\n::\n\n    $ sudo apt-get install ledger-autosync\n\nIf you use pip, you can install the latest released version:\n\n::\n\n    $ pip install ledger-autosync\n\nYou can also install from source, if you have downloaded the source:\n\n::\n\n    $ python setup.py install\n\nYou may need to install the following libraries (on debian/ubuntu):\n\n::\n\n    $ sudo apt-get install libffi-dev libpython-dev libssl-dev libxml2-dev python-pip libxslt-dev\n\nRunning\n~~~~~~~\n\nOnce you have ledger-autosync installed, you can download an OFX file\nfrom your bank and run ledger-autosync against it:\n\n::\n\n    $ ledger-autosync download.ofx\n\nThis should print a number of transactions to stdout. If you add these\ntransactions to your default ledger file (whatever is read when you\nrun ``ledger`` without arguments), you should find that if you run\nledger-autosync again, it should print no transactions. This is\nbecause of the deduplicating feature: only new transactions will be\nprinted for insertion into your ledger files.\n\nUsing the ofx protocol for automatic download\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nledger-autosync also supports using the OFX protocol to automatically\nconnect to banks and download data. You can use the ofxclient program\n(which should have been installed with ledger-autosync) to set up\nbanking:\n\n::\n\n    $ ofxclient\n\nWhen you have added your institution, quit ofxclient.\n\n(At least one user has reported being signed up for a pay service by\nsetting up OFX direct connect. Although this seems unusual, please be\naware of this.)\n\nEdit the generated ``~/ofxclient.ini`` file. Change the\n``description`` field of your accounts to the name used in ledger.\nOptionally, move the ``~/ofxclient.ini`` file to your ``~/.config``\ndirectory.\n\nRun:\n\n::\n\n    ledger-autosync\n\nThis will download a maximum of 90 days previous activity from your\naccounts. The output will be in ledger format and printed to stdout.\nAdd this output to your ledger file. When that is done, you can call:\n\n::\n\n    ledger-autosync\n\nagain, and it should print nothing to stdout, because you already have\nthose transactions in your ledger.\n\nHow it works\n------------\n\nledger-autosync stores a unique identifier as metadata with each\ntransaction. (For OFX files, this is a unique ID provided by your\ninstitution for each transaction.) When syncing with your bank, it\nwill check if the transaction exists by running the ledger or hledger\ncommand. If the transaction exists, it does nothing. If it does not\nexist, the transaction is printed to stdout.\n\nPayee matching\n~~~~~~~~~~~~~~\n\nWhen generating transactions, ledger-autosync attempts to match\nprevious payees to determine the appropriate accounts. For instance,\nif a previous payee was ``Grocery store``, and one posting was to the\naccount ``Expenses:Food``, ledger-autosync will use that account again.\nIf ledger-autosync can’t find a matching payee, it will use\n``Expenses:Misc`` or the value of the ``--unknown-account`` argument.\n\nThe matching is not very sophisticated: it only does exact matching on\nthe payee, and it uses transaction with the matching payee. It is also\nnot currently working with CSV files.\n\nIf you prefer to modify the payees to make them shorter than what is\ngenerated by ledger-autosync, you can use the ``AutosyncPayee`` metadata\nfield to indicate to ledger-autosync that it should use the longer\nform for matching. For instance:\n\n::\n\n    2011/03/31 Grocery\n      ; AutosyncPayee: Payment to Grocery store #12345 CALIFORNIA\n      Assets:Bank                                  -$0.01\n      Expenses:Food                                 $0.01\n\nwould indicate to ledger-autosync that any payee with the name\n``Payment to Grocery store #12345 CALIFORNIA`` should use the\n``Expenses:Food`` account.\n\nofxid/csvid metadata tag\n~~~~~~~~~~~~~~~~~~~~~~~~\n\nledger-autosync stores a metatag with every posting that it outputs to\nsupport deduplication. This metadata tag is either ``ofxid`` (for OFX\nimports) or ``csvid`` for CSV imports.\n\nPre-1.0.0 versions of ledger-autosync put this metadata tag in a\nslightly incorrect place, associating the metadata tag with the\ntransaction itself, and not simply one posting. This should not effect\nthe usage of ledger-autosync, but if you would like to correct your\nledger files, there is a small python script ``fix_ofxid.py`` included\nwith ledger-autosync. It can be run as:\n\n::\n\n   python fix_ofxid.py <input file>\n\nand will print a corrected file to stdout.\n\nSyncing a CSV file\n------------------\n\nIf you have a CSV file, you may also be able to import it using a\nrecent (installed via source) version of ledger-autosync.\nledger-autosync can currently process CSV files as provided by Paypal,\nAmazon, or Mint. You can process the CSV file as follows:\n\n::\n\n    ledger-autosync /path/to/file.csv -a Assets:Paypal\n\nWith Amazon and Paypal CSV files, each row includes a unique\nidentifier, so ledger-autosync will be able to deduplicate against any\npreviously imported entries in your ledger files.\n\nWith Mint, a unique identifier based on the data in the row is\ngenerated and stored. If future downloads contain identical rows, they\nwill be deduplicated. This method is probably not as robust as a\nmethod based on unique ids, but Mint does not provide a unique id, and\nit should be better than nothing. It is likely to generate false\nnegatives: transactions that seem new, but are in fact old. It will\nnot generate false positives: transactions that are not generated\nbecause they seem old.\n\nIf you are a developer, you should fine it easy enough to add a new\nCSV format to ledger-autosync. See, for example, the ``MintConverter``\nclass in the ``ledgerautosync/converter.py`` file in this repository.\nSee below for how to add these as plugins.\n\nAssertions\n----------\n\nIf you supply the ``--assertions`` flag, ledger-autosync will also\nprint out valid ledger assertions based on your bank balances at the\ntime of the sync. These otherwise empty transactions tell ledger that\nyour balance *should* be something at a given time, and if not, ledger\nwill fail with an error.\n\n401k and investment accounts\n----------------------------\n\nIf you have a 401k account, ledger-autosync can help you to track the\nstate of it. You will need OFX files (or an OFX protocol connection as\nset up by ofxclient) provided by your 401k.\n\nIn general, your 401k account will consist of buy transactions,\ntransfers and reinvestments. The type will be printed in the payee\nline after a colon (``:``)\n\nThe buy transactions are your contributions to the 401k. These will be\nprinted as follows:\n\n::\n\n    2016/01/29 401k: buymf\n      Assets:Retirement:401k                                 1.12345 FOOBAR @ $123.123456\n      ; ofxid: 1234\n      Income:Salary                                            -$138.32\n\nThis means that you bought (contributed) $138.32 worth of FOOBAR (your\ninvestment fund) at the price of $123.123456. The money to buy the\ninvestment came from your income. In ledger-autosync, the\n``Assets:Retirement:401k`` account is the one specified using the\n``--account`` command line, or configured in your ``ofxclient.ini``.\nThe ``Income:Salary`` is specified by the ``--unknown-account``\noption.\n\nIf the transaction is a “transfer” transaction, this usually means\neither a fee or a change in your investment option:\n\n::\n\n    2014/06/30 401k: transfer: out\n      Assets:Retirement:401k                                -1.61374 FOOBAR @ $123.123456\n      ; ofxid: 1234\n      Transfer                                                  $198.69\n\nYou will need to examine your statements to determine if this was a\nfee or a real transfer back into your 401k.\n\nAnother type of transaction is a “reinvest” transaction:\n\n::\n\n    2014/06/30 401k: reinvest\n      Assets:Retirement:401k                                0.060702 FOOBAR @ $123.123456\n      ; ofxid: 1234\n      Income:Interest                                            -$7.47\n\nThis probably indicates a reinvestment of dividends. ledger-autosync\nwill print ``Income:Interest`` as the other account.\n\nresync\n------\n\nBy default, ledger-autosync will process transactions backwards, and\nstop when it sees a transaction that is already in ledger. To force it\nto process all transactions up to the ``--max`` days back in time\n(default: 90), use the ``--resync`` option. This can be useful when\nincreasing the ``--max`` option. For instance, if you previously\nsynchronized 90 days and now want to get 180 days of transactions,\nledger-autosync would stop before going back to 180 days without the\n``--resync`` option.\n\npayee format\n------------\n\nBy default, ledger-autosync attempts to generate a decent payee line\n(the information that follows the date in a ledger transaction).\nUnfortunately, because of differences in preference and in the format\nof OFX files, it is not always possible to generate the user’s\npreferred payee format. ledger-autosync supports a ``payee-format``\noption that can be used to generate your preferred payee line. This\noption is of the format ``Text {memo}``, where ``memo`` is a\nsubstitution based on the value of the transaction. Available\nsubstitutions are ``memo``, ``payee``, ``txntype``, ``account`` and\n``tferaction``. For example:\n\n::\n\n   $ ledger-autosync --payee-format "Memo: {memo}"\n   2011/03/31 Memo: DIVIDEND EARNED FOR PERIOD OF 03/01/2011 THROUGH 03/31/2011 ANNUAL PERCENTAGE YIELD EARNED IS 0.05%\n\nThis option is also available for CSV conversion. For CSV files, you\ncan substitution any of the values of the rows in the CSV file by\nname. For instance, for Paypal files:\n\n::\n\n   $ ledger-autosync --payee-format "{Name} ({To Email Address})" -a Paypal paypal.csv\n   2016/06/04 Jane Doe (someone@example.net)\n\npython bindings\n---------------\n\nIf the ledger python bindings are available, ledger-autosync can use\nthem if you pass in the ``--python`` argument. Note, however, they can\nbe buggy, which is why they are disabled by default\n\nPlugin support\n--------------\n\nledger-autosync has support for plugins. By placing python files a\ndirectory named ``~/.config/ledger-autosync/plugins/`` it should be\npossible to automatically load python files from there. You may place\n``CsvCconverter`` subclasses here, which will be selected based on the\ncolumns in the CSV file being parsed and the FIELDSET of the CSV\nconverters. You may also place a single ``OfxConverter`` in the plugin\ndirectory, which will be used in place of the stock ``OfxConverter``.\n\nBelow is an example CSV converter, starting with the input CSV file:\n\n::\n\n    "Date","Name","Amount","Balance"\n    "11/30/2016","Dividend","$1.06","$1,000“\n\nThe following converter in the file ``~/.config/ledger-autosync/plugins/my.py``:\n\n::\n\n    from ledgerautosync.converter import CsvConverter, Posting, Transaction, Amount\n    import datetime\n    import re\n\n    class SomeConverter(CsvConverter):\n        FIELDSET = set(["Date", "Name", Amount", "Balance"])\n\n        def __init__(self, *args, **kwargs):\n            super(SomeConverter, self).__init__(*args, **kwargs)\n\n        def convert(self, row):\n            md = re.match(r"^(\\(?)\\$([0-9,\\.]+)", row[\'Amount\'])\n            amount = md.group(2).replace(",", "")\n            if md.group(1) == "(":\n                reverse = True\n            else:\n                reverse = False\n            if reverse:\n                account = \'expenses\'\n            else:\n                account = \'income\'\n            return Transaction(\n                date=datetime.datetime.strptime(row[\'Date\'], "%m/%d/%Y"),\n                payee=row[\'Name\'],\n                postings=[Posting(self.name, Amount(amount, \'$\', reverse=reverse)),\n                          Posting(account, Amount(amount, \'$\', reverse=not(reverse)))])\n\nRunning ``ledger-autosync file.csv -a assets:bank`` will generate:\n\n::\n\n    2016/11/30 Dividend\n        assets:bank                                $1.06\n        income                                    -$1.06\n\nFor more examples, see\nhttps://gitlab.com/egh/ledger-autosync/blob/master/ledgerautosync/converter.py#L421\nor the `example plugins directory <examples/plugins>`_.\n\nIf you develop a converter that you think will be generally\nuseful, please consider submitting a pull request.\n\nTesting\n-------\n\nledger-autosync uses pytest for tests. To test, run pytest in the project directory. This will test the ledger, hledger and ledger-python interfaces. If hledger or the ledger-python interface is not found, these tests will be skipped.\n',
    'author': 'Erik Hetzner',
    'author_email': 'egh@e6h.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/egh/ledger-autosync',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6',
}


setup(**setup_kwargs)
