"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachineCustomResourceProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
/**
 * A state machine custom resource provider.
 *
 * @stability stable
 */
class StateMachineCustomResourceProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        const cfnResponseSuccessFn = this.createCfnResponseFn('Success');
        const cfnResponseFailedFn = this.createCfnResponseFn('Failed');
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [cfnResponseSuccessFn.functionArn, cfnResponseFailedFn.functionArn],
        }));
        // https://docs.aws.amazon.com/step-functions/latest/dg/stepfunctions-iam.html
        // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution'],
            resources: [props.stateMachine.stateMachineArn],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:DescribeExecution', 'states:StopExecution'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'states',
                    resource: 'execution',
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: `${aws_cdk_lib_1.Stack.of(this).splitArn(props.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                })],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'events',
                    resource: 'rule',
                    resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                })],
        }));
        const definition = aws_cdk_lib_1.Stack.of(this).toJsonString({
            StartAt: 'StartExecution',
            States: {
                StartExecution: {
                    Type: 'Task',
                    Resource: 'arn:aws:states:::states:startExecution.sync:2',
                    Parameters: {
                        'Input.$': '$',
                        'StateMachineArn': props.stateMachine.stateMachineArn,
                    },
                    TimeoutSeconds: ((_b = props.timeout) !== null && _b !== void 0 ? _b : aws_cdk_lib_1.Duration.minutes(30)).toSeconds(),
                    Next: 'CfnResponseSuccess',
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'CfnResponseFailed',
                        }],
                },
                CfnResponseSuccess: {
                    Type: 'Task',
                    Resource: cfnResponseSuccessFn.functionArn,
                    End: true,
                },
                CfnResponseFailed: {
                    Type: 'Task',
                    Resource: cfnResponseFailedFn.functionArn,
                    End: true,
                },
            },
        });
        const stateMachine = new aws_cdk_lib_1.CfnResource(this, 'StateMachine', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        stateMachine.node.addDependency(role);
        const startExecution = new lambda.Function(this, 'StartExecution', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            handler: 'index.startExecution',
            runtime: lambda.Runtime.NODEJS_14_X,
        });
        startExecution.addToRolePolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution'],
            resources: [stateMachine.ref],
        }));
        startExecution.addEnvironment('STATE_MACHINE_ARN', stateMachine.ref);
        this.serviceToken = startExecution.functionArn;
    }
    createCfnResponseFn(status) {
        return new lambda.Function(this, `CfnResponse${status}`, {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            handler: `index.cfnResponse${status}`,
            runtime: lambda.Runtime.NODEJS_14_X,
        });
    }
}
exports.StateMachineCustomResourceProvider = StateMachineCustomResourceProvider;
_a = JSII_RTTI_SYMBOL_1;
StateMachineCustomResourceProvider[_a] = { fqn: "cloudstructs.StateMachineCustomResourceProvider", version: "0.4.18" };
//# sourceMappingURL=data:application/json;base64,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