# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['servo', 'servo.connectors', 'servo.utilities']

package_data = \
{'': ['*']}

install_requires = \
['backoff>=1.10.0,<2.0.0',
 'bullet>=2.1.0,<3.0.0',
 'devtools>=0.6.0,<0.7.0',
 'httpx>=0.16.1,<0.17.0',
 'jsonschema>=3.2.0,<4.0.0',
 'kubernetes_asyncio>=11.3,<13.0',
 'loguru>=0.5.1,<0.6.0',
 'orjson>=3.3.1,<4.0.0',
 'pyaml>=20.4.0,<21.0.0',
 'pydantic>=1.5.1,<2.0.0',
 'pygments>=2.6.1,<3.0.0',
 'python-dotenv>=0.15.0,<0.16.0',
 'pytz>=2020.4,<2021.0',
 'semver>=2.10.1,<3.0.0',
 'statesman>=1.0.0,<2.0.0',
 'tabulate>=0.8.7,<0.9.0',
 'timeago>=1.0.14,<2.0.0',
 'typer>=0.3.0,<0.4.0',
 'uvloop>=0.14.0,<0.15.0']

entry_points = \
{'console_scripts': ['servo = servo.entry_points:run_cli'],
 'servo.connectors': ['kubernetes = '
                      'servo.connectors.kubernetes:KubernetesConnector',
                      'opsani_dev = '
                      'servo.connectors.opsani_dev:OpsaniDevConnector',
                      'prometheus = '
                      'servo.connectors.prometheus:PrometheusConnector',
                      'vegeta = servo.connectors.vegeta:VegetaConnector']}

setup_kwargs = {
    'name': 'servox',
    'version': '0.9.0',
    'description': 'Opsani Servo: The Next Generation',
    'long_description': '# Opsani ServoX\n\n![Run Tests](https://github.com/opsani/servox/workflows/Run%20Tests/badge.svg)\n[![license](https://img.shields.io/github/license/opsani/servox.svg)](https://github.com/opsani/servox/blob/main/LICENSE)\n[![PyPI](https://img.shields.io/pypi/v/servox.svg)](https://pypi.org/project/servox/)\n[![release](https://img.shields.io/github/release/opsani/servox.svg)](https://github.com/opsani/servox/releases/latest)\n[![GitHub release\ndate](https://img.shields.io/github/release-date/opsani/servox.svg)](https://github.com/opsani/servox/releases)\n\nThis repository contains the source code of the Opsani Servo agent.\n\nThe servo connects applications to the Opsani cloud optimization engine\nto identify cost savings and performance enhancements by applying machine\nlearning technology. Servos are lightweight Python applications and are\ntypically deployed as a container under an orchestration layer such as\nKubernetes, ECS, or Docker Compose.\n\nServos are composed of connectors, which provide the core functionality for\nintegrating with metrics, orchestration, and load generation systems/utilities.\nThe ServoX codebase provides core functionality shared by all servos and a rich\nlibrary supporting the development of connectors.\n\n## Quick Start\n\nServoX is a modern Python application distributed as an installable Python\npackage. Development is done in a Python install managed with\n[Pyenv](https://github.com/pyenv/pyenv) and a virtual environment managed by\n[Poetry](https://python-poetry.org/). This is the path of least resistance but\nany Python package management system should work.\n\n* Clone the repo: `git clone git@github.com:opsani/servox`\n* Install required Python: `cd servox && pyenv install`\n* Install Poetry: `curl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py |\n  python`\n* Link Poetry with pyenv version: ``poetry env use `cat .python-version` ``\n* Install dependencies: `poetry install`\n* Activate the venv: `poetry shell`\n* Initialize your environment: `servo init`\n* Start interacting with the servo: `servo --help`\n\n## Overview\n\n### Getting Started with Opsani\n\nAccess to an Opsani optimizer is required to deploy the servo and run the end to\nend integration tests. If you do not currently have access to an Opsani\nenvironment but are otherwise interested in working with the optimizer and\nServo, please reach out to us at [info@opsani.com](mailto:info@opsani.com) and\nwe will get back with you.\n\n### Usage\n\n#### Displaying Help\n\n```console\n❯ servo --help\n```\n\n#### Initializing an Assembly\n\n**NOTE**: A Dotenv file is recommended during development to keep the CLI\noptions under control. The init command will generate one for you.\n\n```console\n❯ servo init\n```\n\n#### Configuration\n\n```console\n# Generate a config file with all available connectors\n❯ servo generate\n\n# Validate a config file\n❯ servo validate\n\n# Display config schema\n❯ servo schema\n```\n\n#### Displaying Info\n\n```console\n# Display all available connectors\n❯ servo connectors\n\n# Display instance specific info (requires configuration)\n❯ servo show connectors\n❯ servo show events\n❯ servo show components\n❯ servo show metrics\n```\n\n#### Running Operations\n\n```console\n# Check servo readiness\n❯ servo check\n\n# Describe application state\n❯ servo describe\n\n# Capture and display a measurement\n❯ servo measure\n\n# Adjust the memory of web-server to 512MB and cpu of gateway to .8 millicores\n❯ servo adjust web.mem=512 gateway.cpu=0.8\n\n# Run the servo to start optimizing\n❯ servo run\n```\n\n## Architecture\n\nServoX has been designed to provide a delightful experience for engineers\nintegrating cloud optimization into their systems and workflow. Developer\nergonomics and operator efficiency are primary concerns as integrating and\norchestrating disparate components can quickly become tiresome and complex. As a\nlibrary, ServoX aspires to be as "batteries included" as possible and support\ndevelopers with well designed, implemented, and tested solutions for common\nconcerns. As a tool, ServoX strives to support system operators and devops\nengineers with strong support for common tasks and a high-velocity workflow.\n\nThere are a few key components that form the foundation of the architecture:\n\n* **Connectors** - Connectors are pluggable components that enable the servo to\n  interact with external systems such as metrics providers (Prometheus, Datadog,\n  New Relic, etc), orchestration technologies (Kubernetes, cloud provider APIs,\n  etc), or load generators. Every major functional component (including the\n  servo itself) is a connector that inherits from the `Connector` base class.\n  Connectors can process events dispatched from the servo (see Events below),\n  provide services to the user (see CLI below), and interact with other\n  connectors.\n* **Servo** - The Servo class models the active set of connectors and\n  configuration that is executing. The servo handles connectivity with the\n  Opsani Optimizer API (see Optimizer below) and is responsible for the primary\n  concerns of connectivity management and event handling.\n* **Configuration** - Configuration is a major shared concern in tools such as\n  Opsani that are designed to integrate with arbitrary systems. Ensuring that\n  configuration is valid, complete, and functional is a non-trivial task for any\n  component with more than a few knobs and levers. ServoX provides a rich\n  configuration subsystem built on Pydantic that makes modeling and processing\n  configuration very straightforward. Out of the box support is provided for\n  common needs such as environment variables and dotenv files. Configuration is\n  strongly validated using JSON Schema and support is provided for generating\n  config files directly from the connectors.\n* **Optimizer** - The Optimizer class represents an Opsani optimization engine\n  that the servo interacts with via an API. The optimizer can be configured via\n  CLI arguments, from the environment, or via assets such as Kubernetes secrets.\n* **Events** - The Event subsystem provides the primary interaction point\n  between the Servo and Connectors in a loosely coupled manner. Events are\n  simple string values that have connector defined semantics and can optionally\n  return a result. The Servo base class defines the primary events of\n  `DESCRIBE`, `MEASURE`, `ADJUST`, and `PROMOTE` which correspond to declaring\n  the metrics & components that the connector is interested in, taking\n  measurements and returning normalized scalar or time series data points,\n  making changes to the application under optimization, or promoting an\n  optimized configuration to the broader system.\n* **Checks** - Checks provide a mechanism for verifying the correctness and\n  health of connector configuration and operations. They are designed to support\n  a high throughput integration and debugging experience by providing feedback\n  loop driven workflow. Checks are implemented on top of the events subsystem\n  and provide a rich interface via the `servo check` CLI command. The design of\n  the checks subsystem is covered in depth [in the docs](docs/checks.md).\n* **Assembly** - The Servo Assembly models the runtime environment of the servo\n  outside of a particular configuration. The assembly is the parent of the servo\n  and is responsible for "assembling" it by instantiating connectors as\n  configured by the operator. Connectors can be used multiple times (e.g. you\n  may want to connect to multiple discrete Prometheus services) or may not be\n  used at all (e.g. you have a New Relic connector in the container image but\n  aren\'t using it).\n* **CLI** - The CLI provides the primary interface for interacting with the\n  servo. The CLI is modular and contains a number of root level commands and\n  connectors can optionally register additional commands. Most of the root level\n  commands are driven through the event subsystem and connectors which respond\n  to the relevant events will automatically become accessible through the CLI.\n  For example, executing `servo schema` will emit a complete JSON Schema\n  document for the assembly while `servo schema kubernetes` will emit a JSON\n  Schema specific to the Kubernetes connector.\n\n### Understanding Events\n\nThe servo is built around an event driven architecture and utilizes a\nlightweight eventing system to pass messages between connectors. Eventing is\nimplemented in asynchronous Python on top of asyncio. Events are simple strings\nidentifiers that are bound to a Python\n[inspect.Signature](https://docs.python.org/3/library/inspect.html#inspect.Signature)\nobject. The signature is used when event handlers are registered to enforce a\ncontract around the parameter and return types, method arity, etc.\n\nAny connector can define an event provided that no other connector has already\nregistered an event with the desired name. The `Servo` class defines several\nbasic events covering essential functionality such as declaring metrics and\ncomponents, capturing measurements, and performing adjustments. These events are\nintegrated into the CLI and readily accessible. For example, executing `servo\nshow metrics` dispatches the `metrics` event to all active connectors and\ndisplays the aggregate results in a table. More substantial commands such as\n`measure` and `adjust` work in the same manner -- dispatching events and\nvisualizing the results. Whenever possible functionality is implemented via\neventing.\n\nWhen the servo is run, it connects to the Opsani optimizer via the API and\nreceives instructions about which actions to take to facilitate the optimization\nactivity. These commands are dispatched to the connectors via events.\n\nThe default events are available on the `servo.servo.Events` enumeration and\ninclude:\n\n| Event | Category | Description |\n|-------|----------|-------------|\n| startup | Lifecycle | Dispatched when the servo is assembled. |\n| shutdown | Lifecycle | Dispatched when the servo is being shutdown. |\n| metrics | Informational | Dispatched to gather the metrics being measured. |\n| components | Informational | Dispatched to gather the components & settings available for adjustment. |\n| check | Operational | Asks connectors to check if they are ready to run by validating settings, etc. |\n| describe | Operational | Gathers the current state of the application under optimization. |\n| measure | Operational | Takes a measurement of target metrics and reports them to the optimizer. |\n| adjust | Operational | Applies a change to the components/settings of the application under optimization and reports status to the optimizer. |\n\n#### Event Handlers & Prepositions\n\nEvent handlers are easily registered via a set of method decorators available on\nthe `servo.connector` module. Handlers are registered with a *preposition* which\ndetermines if it is invoked before, on, or after the event has been processed.\nHandlers are invoked when the servo or another connector dispatches an event.\nEvent handlers can be implemented either synchronously or asynchronously\ndepending on if the method is a coroutined declared with the async prefix.\n\n```python\nfrom typing import List\nimport servo\n\n\nclass SomeConnector(servo.BaseConnector):\n    @servo.before_event(\'measure\')\n    def notify_before_measure(self) -> None:\n        self.logger.info("We are about to measure...")\n\n    @servo.on_event(\'metrics\')\n    def metrics(self) -> List[servo.Metric]:\n        return [servo.Metric(\'throughput\', servo.Unit.REQUESTS_PER_MINUTE)]\n\n    @servo.after_event(\'adjust\')\n    def analyze_results(self, results: List[servo.EventResult]) -> None:\n        self.logger.info(f"We got some results: {results}")\n```\n\nEach preposition has different capabilities available to it. Before event\nhandlers can cancel the execution of the event by raising a `EventCancelledError`.\nOn event handlers can return results that are aggregated and available for\nprocessing. After event handlers get access to all of the results returned by\nactive connectors via the on event handlers.\n\n#### Creating a new Event\n\nEvents can be created either programmatically via the `Connector.create_event()`\nclass method or declaratively via the `event()` decorator:\n\n```python\nimport servo\n\n\nclass EventExample(servo.BaseConnector):\n    @servo.event()\n    async def trace(self, url: str) -> str:\n        ...\n```\n\nThe `event` decorator uses the parameters and return type of the decorated\nmethod to define the signature requirements for on event handlers registered\nagainst the event. The body of the decorated method must be `...`, `pass`, or an\nasync generator that yields `None` exactly once.\n\nThe body of the decorated method can be used to define setup and tear-down\nactivities around on event handlers. This allows for common setup and tear-down\nfunctionality to be defined by the event creator. This is achieved by\nimplementing the body of the decorated method as an async generator that yields\ncontrol to the on event handler:\n\n```python\nfrom typing import AsyncIterator\nimport servo\n\n\nclass SetupAndTearDownExample(servo.BaseConnector):\n    @servo.event()\n    async def trace(self, url: str) -> AsyncIterator[str]:\n        print("Entering event handler...")\n        yield\n        print("Exited event handler.")\n```\n\nThe event decorator can also be used to create an event and register a handler\nfor the event at the same time. In this example, the `handler=True` keyword\nargument is provided to created the event **and** register the decorated method\nas an on event handler for the new event.\n\n```python\nimport servo\n\n\nclass AnotherConnector(servo.BaseConnector):\n    @servo.event(\'load_test\', handler=True)\n    async def run_load_test(self, url: str, duration: int = 60) -> str:\n        return "Do something..."\n```\n\nOnce an event is created, the connector can dispatch against it to notify other\nconnectors of changes in state or to request data from them.\n\n#### Dispatching an Event\n\nConnectors can notify or interact with other connectors by dispatching an event.\nWhen the servo is assembled, an event bus is transparently established between\nthe connectors to facilitate event driven interaction.\n\n```python\nfrom typing import List\nimport servo\n\n\nclass ExampleConnector(servo.BaseConnector):\n    async def do_something(self) -> None:\n        # Gather metrics from other connectors\n        results: List[servo.EventResult] = await self.dispatch_event("metrics")\n        for result in results:\n            print(f"Gathered metrics: {result.value}")\n```\n\n### Environment Variables & Dotenv\n\nPay attention to the output of `servo --help` and `servo schema` to identify\nenvironment variables that can be used for configuration. The servo handles\nconfiguration of deeply nested attributes by building the environment variable\nmapping on the fly at assembly time.\n\nFor convenience, the `servo` CLI utility automatically supports `.env` files for\nloading configuration and is already in the `.gitignore`. Interacting with the\nCLI is much cleaner if you drop in a dotenv file to avoid having to deal with\nthe options to configure the optimizer. The `servo init` command will help set\nthis up interactively.\n\n### Logging\n\nThe servo base library provides logging services for all connectors and core\ncomponents. By default, the `servo` CLI utility runs at the `INFO` log level\nwhich is designed to provide balanced output that informs you of what is\nhappening operationally without becoming overwhelming and pedantic. During\ndevelopment, debugging, or troubleshooting it may become desirable to run at an\nelevated log level. The log level can be set via a commandline option on the\n`servo` utility or via the `SERVO_LOG_LEVEL` environment variable. The servo\nwill emit ANSI colored output to `stderr` when the terminal is a TTY. Coloring\ncan be suppressed via the `--no-color` commandline option or with the\n`SERVO_NO_COLOR` or `NO_COLOR` environment variables.\n\n```console\n  -l, --log-level [TRACE|DEBUG|INFO|SUCCESS|WARNING|ERROR|CRITICAL]\n                                  Set the log level  [env var:\n                                  SERVO_LOG_LEVEL; default: INFO]\n  --no-color                      Disable colored output  [env var:\n                                  SERVO_NO_COLOR, NO_COLOR]\n```\n\nBy default, log messages are written to `stderr` and a file sink at the `logs/`\nsubdirectory relative to the servo root. Backtraces are preformatted and as much\ncontext as possible is provided when an exception is logged.\n\nThe `servo.logging` module exposes some interesting functionality for operators\nand developers alike. Logging is aware of the eventing subsystem and will\nautomatically attribute log messages to the currently executing connector and\nevent context. Long running operations can be automatically reported to the\nOpsani API by including a `progress` key with a numeric percentage value ranging\nfrom 0.0 to 100.0. There are several function decorators available that can\nprovide automatic logging output for entry and exit, execution timing, etc.\n\nDependent libraries such as `backoff` have been configured to emit their logs\ninto the servo logging module. Every component that has logging support is\nintercepted and handled by the logging subsystem and conforms to the log levels\noutlined above.\n\n### Connector Discovery\n\nConnectors are set up to be auto-discovered using the setuptools entry point\nfunctionality available from the Python standard library. When a new connector\nis installed into the assembly, it will be automatically discovered and become\navailable for interaction.\n\nThe specific of how this mechanism works is discussed in detail on the [Python\nPackaging\nGuide](https://packaging.python.org/guides/creating-and-discovering-plugins/).\n\nThe bundled connectors are registered and discovered using this mechanism via\nentries in the `pyproject.toml` file under the\n`[tool.poetry.plugins."servo.connectors"]` stanza.\n\n### Running Multiple Connector Instances\n\nServoX is designed to support assemblies that contain an arbitrary number of\nconnectors that may or may not be active and enable the use of multiple\ninstances of a connector with different settings. This introduces a few modes of\nconfiguration.\n\nThe servo looks to a `connectors` configuration key that explicitly declares\nwhich connectors are active within the assembly. If a `connectors` key is not\npresent in the config file, then all available connectors become optionally\navailable based on the presence or absence of their default configuration key.\nFor example, an assembly that includes New Relic, Datadog, and SignalFX\nconnectors installed as Python packages with the following configuration would\nonly activate Datadog due to the presence of its configuration stanza:\n\n```yaml\ndatadog:\n  setting_1: some value\n  setting_2: another value\n```\n\nThis mode supports the general case of utilizing a small number of connectors in\n"off the shelf" configurations.\n\nFrom time to time, it may become necessary to connect to multiple instances of a\ngiven service -- we have seen this a few times with Prometheus in canary mode\ndeployments where metrics are scattered across a few instances. In these cases,\nit can become necessary to explicitly alias a connector and utilize it under two\nor more configurations. In such cases, the `connectors` key becomes required in\norder to disambiguate aliases from configuration errors. In such cases, the\n`connectors` key can be configured as a dictionary where the key identifies the\nalias and the value identifies the connector:\n\n```yaml\nconnectors:\n  prom1: prometheus\n  prom2: prometheus\n\nprom1:\n  setting_1: some value\n  setting_2: another value\n\nprom2:\n  setting_1: some value\n  setting_2: another value\n```\n\nIt is also possible to utilize the `connectors` key in order to exclude\nconnectors from the active set. This can be done with the dictionary syntax\nreferenced above or using an array syntax if aliasing is not being utilized. For\nexample, given a configuration with New Relic and Prometheus active but some\nsort of issue warranting the isolation of Prometheus from the active set, the\nconfig file might be configured like:\n\n```yaml\nconnectors:\n  - new_relic\n\nprometheus:\n  setting_1: some value\n  setting_2: another value\n\nnew_relic:\n  setting_1: some value\n  setting_2: another value\n```\n\nThese configurations can be generated for you by providing arguments to `servo\ngenerate`. A space delimited list of connector names will explicitly populate\nthe `connectors` key and a syntax of `alias:connector` will configure aliases:\n\n```console\n❯ servo generate foo:vegeta bar:kubernetes\nbar:\n  description: Update the namespace, deployment, etc. to match your Kubernetes cluster\n  namespace: default\nconnectors:\n  bar: kubernetes\n  foo: vegeta\nfoo:\n  description: Update the rate and target/targets to match your load profile\n  duration: 5m\n  rate: 50/1s\n  target: https://example.com/\n\nGenerated servo.yaml\n```\n\n### Running Multiple Servos\n\nServoX is capable of running multiple servos within the same assembly and servos\ncan be added and removed dynamically at runtime. This is useful for optimizing\nseveral applications at one time from a single servo deployment to simplify\noperations or more interestingly to support the integration and automation of\noptimization into CI and CD pipelines. For example, it is possible to configure\nthe build system to trigger optimization for apps as they head into staging or\nupon emergence into production.\n\nMulti-servo execution mode is straightforward. When the `servo.yaml` config file\nis found to contain multiple documents (delimited by `---`), a servo instance is\nconstructed for each entry in the file and added to the assembly. There are\nhowever a few differences in configuration options.\n\nWhen multi-servo mode is enabled, the `--optimizer`, `--token`, `--token-file`,\n`--base-url`, and `--url` options are unavailable. The optimizer and\nconnectivity configuration must be provided via the `optimizer` stanza within\neach configuration *document* in the config file. The CLI will raise errors if\nthese options are utilized with a multi-servo configuration because they are\nambiguous. This does not preclude a single servo being promoted into a\nmulti-servo configuration at runtime -- it is a configuration resolution\nconcern.\n\nWhen running multi-servo, logging is changed to provide context about the servo\nthat is active and generating the output. The `servo.current_servo()` method\nreturns the active servo at runtime.\n\nBecause ServoX is based on `asyncio` and functions as an orchestrator, it is\ncapable of managing a large number of optimizations in parallel (we have tested\ninto the thousands). Most operations performed are I/O bound and asynchronous\nbut the specifics of the connectors used in a multi-servo configuration will\nhave a significant impact on the upper bounds of concurrency.\n\n#### Configuring Multi-servo Mode\n\nBasically all that you need to do is use the `---` delimiter to create multiple\ndocuments within the `servo.yaml` file and configure an `optimizer` within\neach one. For example:\n\n```yaml\n---\noptimizer:\n  id: newco.com/awesome-app1\n  token: 6686e4c3-2c6a-4c28-9c87-b304d7c1427b\nconnectors: [vegeta]\nvegeta:\n  duration: 5m\n  rate: 50/1s\n  target: GET https://app1.example.com/\n---\noptimizer:\n  id: newco.com/awesome-app2\n  token: 5d6e004d-cf7b-4121-b66f-d72f0fd44953\nconnectors: [vegeta]\nvegeta:\n  duration: 5m\n  rate: 50/1s\n  target: GET https://app2.example.com/\n```\n\n#### Adding & Removing Servos at Runtime\n\nServos can be added and removed from the assembly at runtime via methods on the\n`servo.Assembly` class:\n\n```python\nimport servo\n\nassembly = servo.current_assembly()\nnew_servo = servo.Servo()\nassembly.add_servo(new_servo)\nassembly.remove_servo(new_servo)\n```\n\n### Extending the CLI\n\nShould your connector wish to expose additional commands to the CLI, it can do\nso via the `ConnectorCLI` class. Instances are automatically registred with the\nCLI and the `Context` is configured appropriately when commands are invoked. All\nCLI extensions are namespaced as subcommands to keep things tidy and avoid\nnaming conflicts.\n\nA simple example of a CLI extension that will register `servo vegeta attack` is:\n\n```python\nimport servo\nimport servo.cli\nimport servo.connectors.vegeta\n\n\ncli = servo.cli.ConnectorCLI(servo.connectors.vegeta.VegetaConnector, help="Load testing with Vegeta")\n\n\n@cli.command()\ndef attack(context: servo.cli.Context):\n    """\n    Run an adhoc load generation\n    """\n    context.connector.measure()\n```\n\n### Requirements & Dependencies\n\nServoX is implemented in Python and supported by a handful of excellent\nlibraries from the Python Open Source community. Additional dependencies in the\nform of Python packages or system utilities are imposed by connectors (see\nbelow).\n\n* [Python](https://www.python.org/) 3.6+ - ServoX makes liberal use of type\n  hints to annotate the code and drive some functionality.\n* [Pydantic](https://pydantic-docs.helpmanual.io/) - Pydantic is a fantastic\n  parsing and validation library that underlies most classes within ServoX. It\n  enables the strong modeling and validation that forms the core of the\n  configuration module.\n* [Typer](https://typer.tiangolo.com/) - Typer provides a nice, lightweight\n  enhancement on top of [Click](https://click.palletsprojects.com/en/7.x/) for\n  building CLIs in Python. The CLI is built out on top of Typer.\n* [httpx](https://www.python-httpx.org/) - httpx is a (mostly) requests\n  compatible HTTP library that provides support for HTTP/2, is type annotated,\n  has extensive test coverage, and supports async interactions on top of\n  asyncio.\n* [loguru](https://loguru.readthedocs.io/en/stable/index.html) - A rich Python\n  logging library that builds on the foundation of the standard library logging\n  module and provides a number of enhancements.\n\n## Development\n\n### Contributing to ServoX\n\nOpen Source contributions are always appreciated. Should you wish to get\ninvolved, drop us a line via GitHub issues or email to coordinate efforts.\n\nIt is expected that most Open Source contributions will come in the form of new\nconnectors. Should you wish to develop a connector, reach out to us at Opsani as\nwe have connector developer guides that are in pre-release while ServoX matures.\n\n### Visual Studio Code\n\nThe core development team typically works in VSCode. Poetry and VSCode have not\nquite yet become seamlessly integrated. For your convenience, there are a couple\nof Makefile tasks that can simplify configuration:\n\n* `make init` - Initialize a Poetry environment, configure `.vscode/settings.json`,\n  and then run the `servo initialize command.\n* `make vscode` - Export the Poetry environment variables and then open the\n  local working copy within VSCode. The built-in terminal and Python extension\n  should auto-detect the Poetry environment and behave.\n\n### Pre-commit Hook\n\nThe project is configured with a [pre-commit](https://pre-commit.com/) hook to\nenforce as much of the coding standards and style guide as possible. To install\nit into your working copy, run:\n\n```console\npoetry run pre-commit install\n```\n\n### Developing with Local Packages\n\nWhen developing against dependencies or building out new connectors, it can be\nuseful to utilize a local package so that development can be done in the\ndependency and within the servox package at the same time.\n\nTo do so, utilize Poetry Path based dependencies with the `develop = true` flag\nby adding a path reference to the package into the to `tool.poetry.dev-dependencies`\nstanza of the `pyproject.toml` file, and then run `poetry update [dependency name]`.\n\nFor example, if developing on servox and the statesman state machine library in\na working copy in the parent directory, you would add:\n\n```toml\n[tool.poetry.dev-dependencies]\n# ...\nstatesman = {path = "../statesman", develop = true}\n```\n\nAnd then run `poetry update statesman`.\n\nChanges made to the dependency are immediately visible to servox, making it\neasy to develop across package boundaries efficiently.\n\n### Linting and Formatting\n\nThe project is structured to support and enforce consistent, idiomatic code. A\nnumber of tools have been integrated to provide linting and automatic\nreformatting when possible/appropriate.\n\nThis functionality is exposed via tasks in the `Makefile`, Git commit hooks, and\nGitHub Actions.\n\nDuring local development, you may wish to invoke these tools as follows:\n\n* `make lint` - Run the linters and output actionable feedback for conformance\n  with the engineering standards and style of the project.\n* `make format` - Automatically apply changes to the working copy to conform\n  with project standards where possible (e.g., reformatting imports).\n* `make pre-commit` - Run all pre-commit hooks in the manual hook stage. There\n  are a number of standards that are currently soft enforced in the interest of\n  pragmatism. Eventually the set of pre-commit and lint checks will form a\n  perfect union but we are not there yet. Most hooks set in manual mode have to\n  do with fit and finish concerns such as docstring tone/formatting.\n* `make test` - Execute the test suite and generate a code coverage report.\n\nLinting is automatically performed when a new branch is pushed or a PR is opened\nand conformance can be remediated during post-implementation code review.\n\n### Connecting to Development APIs\n\nBy default, the servo will connect to the primary Opsani API hosted on\n[https://api.opsani.com](https://api.opsani.com). This behavior can be\noverridden via CLI options and environment variables:\n\n| Option | Environment Variable | Description | Example |\n|--------|----------------------|-------------|---------|\n| `--base-url` | `OPSANI_BASE_URL` | Sets an alternate base URL. The path is computed using the optimizer. Useful for staging deployments and integration tests. | `servo --base-url https://staging-api.opsani.com:3456 run` |\n| `--url` | `OPSANI_URL` | Sets an explicit URL target. When given, the base URL is ignored and paths are not computed. Useful in unit tests and workstation development with a partial server stack. | `servo --url http://localhost:1234  run` |\n\n### Docker & Compose\n\n`Dockerfile` and `docker-compose.yaml` configurations are available in the\nrepository and have been designed to support both development and deployment\nworkflows. Configuration file mounts and environment variables can be used to\ninfluence the behavior of the servo within the container.\n\nThe `SERVO_ENV` build argument controls the target environment for the built\nimage. Building with `SERVO_ENV=production` excludes development packages from\nthe image to reduce size and build time.\n\nPre-built Docker images are available on\n[opsani/servox](https://hub.docker.com/repository/docker/opsani/servox) on\nDocker Hub. The documentation for these images is available within this\nrepository at [docs/README-DOCKER_HUB.md](docs/README-DOCKER_HUB.md).\n\nThe latest release version is available under the `opsani/servox:latest` tag.\nThe `main` development branch is published as the `opsani/servox:edge` tag.\n\nDocker images are built and published to Docker Hub via the\n[Docker GitHub Actions Workflow](.github/workflows/docker.yaml). The workflow\nbuilds branches, published releases, and the `main` integration branch. Pull\nRequests are not published to Docker Hub because as a publicly available\nrepository it could become an attack vector.\n\nGit branches and Docker images have differing naming constraints that impact how\ntag names are computed. For example, Docker tags cannot contain slashes, which\nis a common practice for namespacing branches and tags in Git. As such, slashes\nare converted to hyphens when computing tag names for branches and tags. The\nfull naming constraints on Docker image tags is covered in the [`docker\ntag`](https://docs.docker.com/engine/reference/commandline/tag/#extended-description)\ndocumentation.\n\nPre-built images are built using BuildKit and can be used as the basis for very\nfast customized builds:\n\n```console\n❯ DOCKER_BUILDKIT=1 docker build -t servox --build-arg BUILDKIT_INLINE_CACHE=1 --cache-from opsani/servox:edge .\n```\n\n### Switching Python Interpreters\n\nAfter changing Python interpreter versions you may find that you are "stuck" in\nthe existing virtual environment rather than your new desired version.\n\nThe problem is that Poetry is linked against the previous environment and needs\na nudge to select the new interpreter.\n\nThe project is bound to a local Python version via the `.python-version` file.\nTell Poetry to bind against the locally selected environment via:\n``poetry env use `cat .python-version` ``\n\n## Testing\n\nTests are implemented using [pytest](https://docs.pytest.org/en/stable/) and\nlive in the `tests` subdirectory. Tests can be executed directly via the\n`pytest` CLI interface (e.g., `pytest tests`) or via `make test`, which\nwill also compute coverage details.\n\nServoX is developed with a heavily test-driven workflow and philosophy. The\nframework strives to provide as much support for robust testing as possible\nand make things that you would think are very hard to test programmatically\nvery simple. You will want to familiarize yourself with what is available,\nthere are tools that can dramatically accelerate your development.\n\n### Test Types\n\nServoX divides the test suite into three types: **Unit**, **Integration**, and\n**System** tests.\n\nTests are identified within the suite in two ways:\n\n1. Use of pytest markers to annotate the tests in code.\n2. File location within the tests/ subdirectory.\n\nUnit tests are the default type and make up the bulk of the suite. They either\nexercise code that carries no outside dependencies or utilize isolation\ntechniques such as mocks and fakes. They are fast and highly effective for\nvalidating defined behavior and catching regressions.\n\nIntegration tests do not interact with an Opsani Optimizer but do interact with\nexternal systems and services such as Kubernetes and Prometheus. It is common to\nutilize a local environment (such as Docker, Compose, kind, or Minikube) or a\ndedicated cloud instance to host the services to interact with. But the focus of\nthe tests are on implementing and verifying the correct behaviors in a\nsupportive environment.\n\nSystem tests are much like integration tests except that theyn are highly\nprescriptive about the environment they are runnuing in and interact with a real\noptimizer backend as much as is practical. System tests sit at the top of the\npyramid and it is expected that there are comparatively few of them, buit they\ndeliver immense value late in a development cycle when code correctness has been\nestablished and deployment environments and compatibility concerns come to the\nforefront.\n\nAll test types can be implemented within any test module as appropriate by\nannotating the tests with pytest markers:\n\n```python\nimport pytest\n\n\n@pytest.mark.integration\nclass TestSomething:\n    ...\n```\n\nTests without a type mark are implicitly designated as unit tests for\nconvenience. When multiple type marks are in effect due to hierarchy, the\nclosest mark to the test node has precedence.\n\nThere are also dedicated directories for integration and system tests at\n`tests/integration` and `tests/system` respectively. These dedicated directories\nare home to cross cutting concerns and scenarios that do not clearly belong to a\nspecific module. Tests in these directories that are marked with other types\nwill trigger an error.\n\n### Running Tests\n\nA key part of any testing workflow is, well, running tests. pytest provides a\nwealth of capabilities out of the box and these have been further augmented with\ncustom pytest-plugins within the suite.\n\nLet\'s start with the basics: executing `pytest` standalone will execute the unit\ntests in `tests`. Running `pytest --verbose` (or -v) will provide a one test per\nline output which can be easier to follow in slower runs.\n\nIndividual files can be run by targetting them by name: `pytest\ntests/connector_test.py`. Individual test functions and container test classes\nwithin the file (known as nodes in pytest parlance) can be addressed with the\n`tests/path/to/test_something.py::TestSomething::test_name` syntax.\n\nTests can also be flexibly selected by marks and naming patterns. Invoking\n`pytest -m asyncio -k sleep` will select all tests with the asyncio mark and\nhave the word "sleep" in their name. These arguments support a matching syntax,\nlook into the pytest docs for details.\n\nThe ServoX test types have specific affordances exposed through pytest. Running\n`pytest -T integration` will select and run all integration tests, but deselect\nall other types. The `-T` also known as `--type` flag supports stem matching for\nbrevity: `pytest -T u` will select the unit tests.\n\nBecause they are slow and require supplemental configuration, integration and\nsystem tests are skipped by default. They can be enabled via the\n`-I, --integration` and `-S, --system` switches, respectively. Note the difference\nin behavior between the flags: `pytest -I -S` will result in all the tests being\nselected for run whereas `pytest -T sys` targets the system tests exclusively.\nOnce the `-I, -S` flags have been used to enable the tests, they can be further\nfiltered using `-m` and `-k`. If you are thoughtful about how you name your\ntests and leverage marks when it makes sense, it can become very easy to run\ninteresting subsets of the suite.\n\nBy default, pytest uses output buffering techniques to capture what is written\nto stdout and stderr. This can become annoying if you are trying to introspect\nstate, print debugging info, or get a look at the servo logs. You can suppress\noutput capture via the `-s` switch. This is typically only recommended when\nrunning a small number of tests because the output quickly becomes\nincomprehensible.\n\nIf you are working through a set of failures, you can rerun the tests that\nfailed on the last run via the `--lf, --last-failed` flag. The `--ff,\n--failed-first` flag will rerun all of the tests, but run the previously failed\ntests first. Similarly, `--nf, --new-first` will run the full suite but\nprioritize new files. The `pytest-picked` plugin provides additional targeting\nbased on git working copy status -- running `pytest --picked` will find unstaged\nfiles to run.\n\nFinally, the `--sw, --stepwise` and `--sw-skip, --stepwise-skip` flags allow you\nto methodically working through a stack of failures by resuming from your last\nfailure and then halting at the next one.\n\n### Makefile Tasks\n\nTest automation tasks are centralized into the `Makefile`. There are a number of\ntesting  tasks availble including:\n\n* `make test` - Run all available tests.\n* `make test-unit` - Run unit tests.\n* `make test-integration` - Run integration tests.\n* `make test-system` - Run system tests.\n* `make test-coverage` - Run all available tests and generate code coverage\n  report.\n* `make test-kubeconfig` - Generate a kubeconfig file at tests/kubeconfig. See\n  details in [Integration Testing](#integration-testing) below.\n* `make autotest` - Automatically run tests based on filesystem changes.\n\nTesting tasks will run in subprocess distributed mode by default (see below).\n\n### Integration Testing\n\nThe test suite includes support for integration tests for running tests against\nremote system components such as a Kubernetes cluster or Prometheus deployment.\nIntegration tests require a [kubeconfig](https://kubernetes.io/docs/concepts/configuration/organize-cluster-access-kubeconfig/)\nfile at `tests/kubeconfig`.\n\nBy convention, the default integration testing cluster is named `kubetest`\nand the `make test-kubeconfig` task is provided to export the cluster details\nfrom your primary kubeconfig, ensuring isolation.\n\nInteraction with the Kubernetes cluster is supported by the most excellent\n[kubetest](https://kubetest.readthedocs.io/en/latest/) library that provides\nfixtures, markers, and various testing utilities on top of pytest.\n\nTo run the integration tests, execute `pytest -I` to enable the\nmarker. Integration tests are much slower than the unit test suite\nand should be designed to balance coverage and execution time.\n\nSystem tests are enabled by running `pytest -S`. Systems tests are very similar\nto integration tests in implementation and performance, but differ in that they\nare prescriptively bound to particular deployment environments and interact with\nan actual\n\nTests can also be run in cluster by packaging a development container and\ndeploying it. The testing harness will detect the in-cluster state and utilize\nthe active service account.\n\n### Continuous Integration\n\nThe project is configured to run CI on unit tests for all branches, tags, and\npull requests opened against the repository. CI for integration and system tests\nis constrained by a few rules because they are so resource intensive and may be\nundesirable during experimental developmenmt or integrating multiple branches\ntogether.\n\nIntegration and system tests are run if any of the following conditions are\nmet:\n\n* A push is made to `main`.\n* A push is made to a branch prefixed with `release/`.\n* A push is made to a branch prefixed with `bugfix/`.\n* A tag is pushed.\n* A push is made to a branch with an open pull request.\n* The commit message includes `#test:integration` and/or\n  `#test:system`.\n\nThe default unit test job that is executed for all pushes generates code\ncoverage reports and XML/HTML report artifacts that are attached to the run. The\nintegration and system test jobs report on the runtime duration of the tests to\nhelp identify and manage runtime creep.\n\nThe unit, integration, and system test jobs all utilize the `pytest-xdist`\nplugin to split the test suite up across a set of subprocesses. This is\ndiscussed in the [Distributed Testing](#distributed-testing) section.\n\nDocker images are built and pushed to Docker Hub automatically for all\npushed refs. Release tags are handled automatically and the\n`opsani/servox:latest` tag is advanced when a new version is released. The\n`main` branch is built and pushed to the `opsani/servox:edge` tag.\n\n#### Distributed Testing\n\nThe project is configured to leverage locally distributed test execution by\ndefault. Servo workloads are heavily I/O bound and spend quite a bit\nof time awaiting data from external services. This characteristic makes tests\ninherently slower but also makes them very well suited for parallel execution.\nThe `Makefile` tasks and GitHub actions are configured to leverage a subprocess\ndivide & conquer strategy to speed things up.\n\nThis functionality is provided by [pytest-xdist](https://docs.pytest.org/en/3.0.1/xdist.html).\n\n### Manifest Templating\n\nAll manifests loaded through kubetest support Mustache templating.\nA context dictionary is provided to the template that includes references to all\nKubernetes resources that have been loaded at render time. The `namespace` and\n`objs` are likely to be the most interesting.\n\n## License\n\nServoX is distributed under the terms of the Apache 2.0 Open Source license.\n\nA copy of the license is provided in the [LICENSE](LICENSE) file at the root of\nthe repository.\n',
    'author': 'Blake Watters',
    'author_email': 'blake@opsani.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://opsani.com/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
