import logging
from dataclasses import dataclass
from datetime import datetime

from shellcord.scord import Command, ScordLog
from shellcord.config import TRIPLE_TICKS

logger = logging.getLogger(__name__)


@dataclass
class RunbookOptions(object):
    allow_exit_codes = None

    def __post_init__(self):
        self.allow_exit_codes = self.allow_exit_codes or [0]

    def include_command(self, command: Command):
        if command.exit_code not in self.allow_exit_codes:
            return False

        return True


class DefaultRunbookOptions(RunbookOptions):
    pass


@dataclass
class RunbookGenerator(object):
    """
    Generate a runbook from an ScordLog
    """
    scord_log: ScordLog
    opts: RunbookOptions = DefaultRunbookOptions()

    @classmethod
    def heading(cls, level: int, s: str):
        pre = "#" * level
        return f"{pre} {s}"

    @classmethod
    def linebreak(cls, lines: int = 1):
        br = "<br/>"
        return "\n".join(br for i in range(lines))

    @property
    def header(self):
        txt = self.heading(1, "Runbook") + "\n"

        return txt

    @property
    def footer_str(self):
        now = datetime.now()
        return f"Generated by shellcord session={self.scord_log.session_id} at {now}"

    def footer(self):
        txt = ""
        txt += self.linebreak(lines=5) + "\n\n"
        txt += self.heading(6, self.footer_str)
        return txt

    def code_part(self, s: str):
        return f"{TRIPLE_TICKS}bash\n{s}\n{TRIPLE_TICKS}"

    def make_command(self, command: Command, cmd_num: int):
        txt = ""

        if command.scord_id in self.scord_log.tags:
            tag = self.scord_log.tags[command.scord_id]
            txt += self.heading(3, tag.tag_str)
        else:
            d = cmd_num
            txt += self.heading(3, f"Command #{d}")
        txt += "\n"
        txt += self.code_part(command.cmd)
        return txt

    def make_markdown(self, fname: str):
        txt = self.header

        for cmd_num, command in enumerate(self.scord_log.cmds):
            if not self.opts.include_command(command):
                logger.debug("Skipping command %s", command)
            txt += self.make_command(command, cmd_num=cmd_num)
            txt += "\n\n"

        txt += self.footer()

        with open(fname, 'w') as f:
            f.write(txt)

        logger.info("Wrote runbook to %s", fname)
