"""
    batsim.sched.workloads
    ~~~~~~~~~~~~~~~~~~~~~~

    This module provides tools to generate workload files. It can also be used to generate workloads
    dynamically and submit them during the runtime of a scheduler.

    WorkloadDescriptions are used internally to manage dynamic jobs created with
    `Scheduler.register_dynamic_job` and with `Job.submit_sub_job`. For each job
    a workload is generated named after the id of the job which is then submitted
    to Batsim.
"""

import json
import sys
from datetime import datetime
from abc import ABCMeta, abstractmethod

from ..utils import ListView
from ..profiles import Profiles, Profile
from ...batsim import Batsim


class JobDescription:
    """A `JobDescription` contains information about a job which is not yet submitted
    to Batsim.

    It is also used to describe jobs which can be written to an external workload file.

    :param res: The number of requested resources

    :param walltime: The walltime of the job (max reserved runtime)

    :param profile: A `Profile` object describing the code to execute when this job is running

    :param subtime: The time of submission

    :param id: The id of this job. Can be omitted and generated by the workload.

    :param kwargs: Additional fields which will be set in the json data and sent to Batsim.

    """
    def __init__(
            self,
            res,
            walltime,
            profile,
            subtime=None,
            id=None,
            **kwargs):
        super().__setattr__("_submitted", False)

        self._id = id
        self._res = res
        self._subtime = subtime
        self._walltime = walltime
        self._profile = profile

        self._workload = None

        self._additional_profiles = None

        self._job = None

        for k, v in kwargs.items():
            setattr(self, k, v)

    def clone(self):
        return copy.deepcopy(self)

    @property
    def workload(self):
        """The `WorkloadDescription` if assigned. This is required in order to submit
        jobs since each job needs a workload.
        """
        return self._workload

    @property
    def job(self):
        """The job object from Batsim if the job was already submitted."""
        return self._job

    @job.setter
    def job(self, value):
        assert self.submitted, "Job was not yet submitted"
        assert not self._job, "Job was already assigned"
        self._job = value
        self.workload.update_job(self, value)

    @property
    def submitted(self):
        """Whether this job description was already sent to Batsim."""
        return self._submitted

    @property
    def id(self):
        """The id of this job."""
        return self._id

    @id.setter
    def id(self, id):
        assert not self.submitted, "Job was already submitted and can not be modified"
        has_workload = bool(self.workload)

        #if has_workload:
        #    self.workload.remove_job(self)

        self._id = id

        #if has_workload:
        #    self.workload.add_job(self)

    @property
    def res(self):
        """The requested number of resources."""
        return self._res

    @res.setter
    def res(self, res):
        assert not self.submitted, "Job was already submitted and can not be modified"
        self._res = res

    @property
    def subtime(self):
        """The time of submission of this job."""
        return self._subtime

    @subtime.setter
    def subtime(self, subtime):
        assert not self.submitted, "Job was already submitted and can not be modified"
        self._subtime = subtime

    @property
    def walltime(self):
        """The walltime of this job."""
        return self._walltime

    @walltime.setter
    def walltime(self, walltime):
        assert not self.submitted, "Job was already submitted and can not be modified"
        self._walltime = walltime

    @property
    def profile(self):
        """The `Profile` describing this job's execution."""
        return self._profile

    @profile.setter
    def profile(self, profile):
        assert not self.submitted, "Job was already submitted and can not be modified"
        self._profile = profile

    def to_dict(self, embed_references=False):
        """A dictionary representation close to the json representation in the workload file.

        :param embed_references: Whether or not the profile and references in the profile should be embedded.
        """
        subtime = self.subtime or 0
        walltime = self.walltime or 0
        res = self.res or 0

        # Export all properties with the exception of internal ones
        result = {k: v for k, v in self.__dict__.items()
                  if not k.startswith("_")}
        result.update(
            {"id": self.id, "profile": self.profile.to_dict(
                embed_references=True)
             if embed_references else self.profile, "res": res,
             "subtime": subtime, "walltime": walltime, })
        return result

    def to_json_string(self, flat=False):
        """Return a json representation similar to the workload file.

        :param flat: Whether the output should be printed without line breaks (if possible).
        """
        return json.dumps(
            self.to_dict(
                embed_references=True),
            sort_keys=True,
            indent=None if flat else 4)

    def get_profiles(self):
        """Return a list of all profiles which are related to this job."""
        return [self.profile] + self.get_additional_profiles()

    def get_additional_profiles(self):
        """Return a list of the additional profiles, i.e. profiles which are references
        but not directly executed.
        """
        if self._additional_profiles:
            return self._additional_profiles
        return self.profile.get_additional_profiles()

    def __str__(self):
        return self.to_json_string()

    def submit(self, scheduler):
        """Submit this job to the scheduler for execution.

        :param scheduler: The scheduler which should execute this job.
        """
        assert not self._submitted, "Job was already submitted"
        assert self.workload is not None, "Job has no workload"
        assert self.workload.name, "Workload has no name"
        assert self.id is not None, "Job.id was not set"
        assert self.profile is not None, "Job.profile was not set"
        assert self.profile.name, "Job.profile has no name"
        assert self.res is not None, "Job.res was not set"
        assert self.subtime is None or self.subtime >= 0, "Job.subtime was not set"
        assert self.walltime is not None and self.walltime >= 0 or self.walltime == -1, "Job.walltime was not set"

        scheduler.info(
            "Submitting job description (workload_name={workload_name}, job={job}, profile={profile_name})",
            job=self.to_json_string(
                flat=True),
            workload_name=self.workload.name,
            profile_name=self.profile.name,
            type="job_description_submit")

        # Freeze the profiles at the time of submission
        self._additional_profiles = self.get_additional_profiles()
        additional_profiles = {p.name: p.to_dict()
                               for p in self._additional_profiles}

        scheduler._batsim.register_profiles(
            self.workload.name,
            {self.profile.name : self.profile.to_dict()})
        scheduler._batsim.register_job(
            str(self.id),
            self.res,
            self.walltime,
            self.profile.name,
            self.subtime)
        scheduler._batsim.register_profiles(
            self.workload.name,
            additional_profiles)
        self._submitted = True

        # Keep track of the workload object in the scheduler to relate job
        # submissions back to the job description object.
        try:
            workload_obj = scheduler._workload_map[self.workload.name]
        except KeyError:
            workload_obj = self.workload
            scheduler._workload_map[self.workload.name] = workload_obj
        try:
            job_obj = workload_obj[self.id]
            if job_obj != self:
                raise ValueError(
                    "Workload name is ambiguous: {}".format(
                        self.workload.name))
        except KeyError:
            assert false, "Job is not correctly assigned to a workload"


class WorkloadDescription:
    """A `WorkloadDescription` manages a workload similar to those in workload files.

    :param name: The name of the workload. It can be omitted on initialisation but has to be set
                 before jobs can be submitted.

    :param nb_res: The number of required resources. If omitted it will be automatically set to
                   the required number when the workload is printed.
                   If the workload is dynamically submitted it the `nb_res` variable
                   does not matter at all.


    :param description: An additional description (optional)

    :param date: An additional date (optional)


    """

    def __init__(
            self,
            name=None,
            nb_res=None,
            description=None,
            date=None,
            **kwargs):
        self._last_job_id = 0

        self._nb_res = nb_res
        self._name = name
        self._description = description
        self._date = date

        self._jobs = []
        self._jobmap = {}

        for k, v in kwargs.items():
            setattr(self, k, v)

    def __str__(self):
        return self.to_json_string()

    def __getitem__(self, items):
        return self._jobmap[items]

    def __iter__(self):
        return iter(self.jobs)

    @property
    def nb_res(self):
        """The number of required resources."""
        return self._nb_res

    @nb_res.setter
    def nb_res(self, nb_res):
        assert not self.submitted_jobs, "A workload's properties can not be changed if it contains submitted jobs."
        assert nb_res is None or nb_res >= 0
        self._nb_res = nb_res

    @property
    def name(self):
        """The name of this workload."""
        return self._name

    @name.setter
    def name(self, name):
        assert not self.submitted_jobs, "A workload's properties can not be changed if it contains submitted jobs."
        self._name = name

    @property
    def description(self):
        """The description of this workload."""
        return self._description

    @description.setter
    def description(self, description):
        assert not self.submitted_jobs, "A workload's properties can not be changed if it contains submitted jobs."
        self._description = description

    @property
    def date(self):
        """The date of this workload."""
        return self._date

    @date.setter
    def date(self, date):
        assert not self.submitted_jobs, "A workload's properties can not be changed if it contains submitted jobs."
        self._date = date

    @property
    def jobs(self):
        """A view of all jobs."""
        return ListView(self._jobs)

    @property
    def unsubmitted_jobs(self):
        """A view of all unsubmitted jobs."""
        return ListView([job for job in self.jobs if not job.submitted])

    @property
    def submitted_jobs(self):
        """A view of all submitted jobs."""
        return ListView([job for job in self.jobs if job.submitted])

    @property
    def acknowledged_jobs(self):
        """A view of all acknowledged jobs (submitted and confirmed by Batsim)."""
        return ListView([job for job in self.jobs if job.job is not None])

    def new_job(self, *args, **kwargs):
        """Adds a new job to the workload.

        See `JobDescription` for more details on the arguments.
        """
        job = JobDescription(*args, **kwargs)
        self.add_job(job)
        return job

    def add_job(self, job):
        """Appends a `JobDescription` to the workload.

        :param job: The `JobDescription`.
        """
        assert job.workload is None, "Job already has a workload"
        assert job.id not in self._jobmap, "A job with the same id already exists in the workload"

        self._jobs.append(job)

        job._workload = self

        if job.id is None or job.id < self._last_job_id:
            job._id = self._last_job_id
            self._last_job_id += 1
        elif job.id > self._last_job_id:
            self._last_job_id = job.id + 1

        jid = job._workload.name + '!' + str(job.id)
        job.id = jid

        self._jobmap[job.id] = job

    def remove_job(self, job):
        """Removes a `JobDescription` from the workload.

        :param job: The `JobDescription`.
        """
        assert not job.submitted, "Job was already submitted and can not be removed from the workload"
        self._jobs.remove(job)
        job._workload = None

        del self._jobmap[job.id]

    def get_profiles(self, jobs=None):
        """Returns all profiles which are related to the given jobs.

        :param job: The jobs to collect the profiles from. As fallback all jobs in the workload are
                    considered.
        """
        # Recursively collect all profiles
        def expand_profiles(ps, visited=[], result=[]):
            sub_profiles = []
            for profile in ps:
                # id(profile) is necessary since profiles are considered as equal if their json
                # data is the same (the name of the profile instance does not
                # matter).
                if id(profile) not in visited:
                    sub_profiles += profile.get_additional_profiles()
                    visited.append(id(profile))
                    result.append(profile)
            if sub_profiles:
                sub_profiles += expand_profiles(sub_profiles, visited, result)
            return result

        if jobs is None:
            jobs = self.jobs

        return expand_profiles([profile for job in jobs
                                for profile in job.get_profiles()])

    def reduce_profiles(self):
        """Reduce the profile instances in this workload.

        By reducing the algorithm tries to re-use already used profiles to reduce
        their number so that Batsim has to store less profiles.
        """
        jobs = self.unsubmitted_jobs
        profiles = self.get_profiles()
        submitted_profiles = self.get_profiles(self.submitted_jobs)

        profiles_map = {}

        for profile in reversed(profiles):
            profiles_map[profile] = profile

        for profile in profiles:
            if profile not in submitted_profiles:
                profile.replace_additional_profiles(profiles_map)

        for job in jobs:
            job.profile = profiles_map[job.profile]

    def fill_profile_names(self, reset=False):
        """Fill empty profile names with automatically generated names since every
        profile needs to have a name in Batsim.

        :param reset: Reset also already set names to the automatically generated ones.
                      However, only the profiles of unsubmitted jobs are considered.
        """
        last_profile_id = 0
        profile_names = set()

        unsubmitted_profiles = self.get_profiles(self.unsubmitted_jobs)
        all_profiles = self.get_profiles()

        if reset:
            for profile in unsubmitted_profiles:
                profile.name = None

        for profile in all_profiles:
            if profile.name is not None:
                profile_names.add(profile.name)

        for profile in unsubmitted_profiles:
            if profile.name is None:
                while True:
                    new_name = "{}_profile_{}".format(
                        profile.type, last_profile_id)
                    last_profile_id += 1
                    if new_name not in profile_names:
                        profile_names.add(new_name)
                        break
                profile.name = new_name

    def to_dict(self):
        """Return a dictionary representation of this workload which can be given
        as parameter to Batsim.
        """
        date = self.date or datetime.now()
        datestr = date.strftime(
            "%a %d %b %H:%M:%S %Z %Y") if not isinstance(date, str) else date

        job_dicts = [job.to_dict(embed_references=False) for job in self.jobs]
        profiles = self.get_profiles()

        for profile in profiles:
            if profile.name is None:
                raise ValueError("Profile has no name: {}".format(profile))

        profile_dicts = {profile.name: profile.to_dict()
                         for profile in profiles}

        for job in job_dicts:
            job["profile"] = job["profile"].name
            assert job["profile"], "Job {} has no profile set.".format(job)

            if "deps" in job:
                deps = []
                for dep in job["deps"]:
                    if isinstance(dep, JobDescription):
                        if dep in jobs:
                            dep = str(dep.id)
                        else:
                            dep = str(dep.workload.name + "!" + dep.id)
                    else:
                        dep = str(dep)
                    assert dep, "Dependency of job {} is None".format(job)
                    deps.append(dep)
                job["deps"] = deps

        for job in job_dicts:
            for field in ["id", "profile", "res", "subtime", "walltime"]:
                if field not in job:
                    raise ValueError(
                        "Job {} is lacking required field: {}".format(
                            job, field))

        nb_res = self.nb_res
        if nb_res is None:
            nb_res = max([job.res for job in self.jobs])

        output_dict = {k: v for k, v in self.__dict__.items()
                       if not k.startswith("_")}
        output_dict.update({
            "name": self.name or "",
            "description": self.description or "",
            "date": datestr,
            "nb_res": nb_res,
            "jobs": job_dicts,
            "profiles": profile_dicts,
        })
        return output_dict

    def to_json_string(self):
        """Convert this workload to a string json representation."""
        return json.dumps(self.to_dict(), sort_keys=True, indent=4)

    def prepare(self):
        """Prepare this workload for execution by reducing the profiles and fill-in
        empty profile names.
        """
        self.reduce_profiles()
        self.fill_profile_names()

    def print(self, file=None):
        """Print this workload."""
        if file is None:
            file = sys.stdout
        file.write(self.to_json_string())
        file.write("\n")
        file.flush()

    def prepare_and_print(self, file=None):
        """Prepare this workload for execution and print its json representation to
        stdout or to the given file.
        """
        self.prepare()
        self.print(file=file)

    def submit(self, scheduler, prepare=True):
        """Submit this workload and all contained jobs to Batsim.

        :param scheduler: The scheduler to which the jobs should be submitted

        :param prepare: Whether the workload should be automatically prepared for execution
                        prior to sending it to Batsim.
        """
        assert self.name, "Workload has no name"

        if prepare:
            self.prepare()

        unsubmitted_jobs = self.unsubmitted_jobs
        if unsubmitted_jobs:
            scheduler.info(
                "Submitting workload ({workload_name})",
                workload_name=self.name,
                type="dynamic_workload_submit")

            for job in unsubmitted_jobs:
                job.submit(scheduler)

    def update_job(self, jobdesc, job):
        pass


def generate_workload(func, output_file=None, standalone=True, **kwargs):
    """Call the function `func(workload)` which should set-up the workload object.
    The workload will then be printed to the file or to stdout and afterwards the
    script terminates if `standalone` is set.

    :param func: the function generating the workload (`func(workload)`)

    :param output_file: the output file to write the workload to (or `None`)

    :param standalone: if `True`, the script will terminate after the workload has been
                       printed.

    :param kwargs: additional arguments will be forwarded to the `WorkloadDescription`
                   constructor.

    """
    w = WorkloadDescription(name=func.__name__, **kwargs)
    func(w)
    w.prepare_and_print(file=output_file)

    if standalone:
        sys.exit(0)
