"""
# data_plumber/fork.py

This module defines the `Fork`-class which enables more complex flows
in a `Pipeline.run`.
"""

from typing import Callable, Optional

from .component import _PipelineComponent
from .context import PipelineContext
from .ref import StageRef, StageById, StageByIncrement


class Fork(_PipelineComponent):
    """
    A `Fork` can be inserted into a `Pipeline` to control flow/execution
    order of a `Pipeline.run(...)`-command. This class is not indended
    for direct use, but taylored to be integrated into a `Pipeline`. It
    is initialized with a callable that returns either a `StageRef` to
    or a `str`-identifier of a `Stage`. Supported input arguments are
    (in the context of `Pipeline` execution, see `Stage` for more
    details)
    * all kwargs given to `Pipeline.run` are forwarded (note that this
      makes the following parameters reserved words in this context),
    * `out` (an object that is passed through the entire `Pipeline`; its
      initial value is generated by the `Pipeline`'s `initialize_output`
      kwarg),
    * `count` (index of `Stage` in execution of `Pipeline`)

    A return value of `None` for the callable is treated as a request to
    exit the `Pipeline` execution.

    Example usage:
     >>> from data_plumber import Fork
     >>> Fork(
             lambda **kwargs: None if "arg" in kwargs else "stage-default"
         )
     <data_plumber.fork.Fork object at ...>

    Keyword arguments:
    fork -- callable that returns a reference to a `Stage` as (StageRef
            | str | int)
            (kwargs: `out`, `count`)
    """

    def __init__(
        self,
        fork: Callable[..., Optional[StageRef | str | int]]
    ) -> None:
        self._fork = fork
        super().__init__()

    def eval(self, context: PipelineContext) -> Optional[StageRef]:
        """
        Returns a `StageRef` or `None` as given with `Fork`s conditional
        function.

        Keyword arguments:
        context -- `Pipeline` execution context
        """

        result = self._fork(
            **context.kwargs,
            out=context.out,
            count=context.count,
            records=context.records
        )

        # replace int or string by corresponding StageRef.
        if isinstance(result, str):
            return StageById(result)
        if isinstance(result, int):
            return StageByIncrement(result)
        # otherwise it is either a StageRef already or None
        return result
