from typing import Any, Dict, Optional, Union

import httpx

from ...client import AuthenticatedClient
from ...exceptions import BadRequestError, HTTPClientError, ParamValidationError, UnauthorizedTokenError
from ...models.http_validation_error import HTTPValidationError
from ...models.message_model import MessageModel
from ...models.user_password_input import UserPasswordInput
from ...types import UNSET, Response


def _get_kwargs(
    *,
    client: AuthenticatedClient,
    json_body: UserPasswordInput,
    oblivious_user_id: str,
) -> Dict[str, Any]:
    url = "{}/user/password".format(client.base_url)

    headers: Dict[str, str] = client.get_headers()
    cookies: Dict[str, Any] = client.get_cookies()

    params: Dict[str, Any] = {}
    params["oblivious_user_id"] = oblivious_user_id

    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    json_json_body = json_body.to_dict()

    return {
        "method": "put",
        "url": url,
        "headers": headers,
        "cookies": cookies,
        "timeout": client.get_timeout(),
        "json": json_json_body,
        "params": params,
    }


def _parse_response(*, response: httpx.Response) -> Optional[Union[Any, HTTPValidationError, MessageModel]]:
    if response.status_code == 200:
        response_200 = response.json()

        return response_200
    if response.status_code == 400:
        response_400_message = response.json()["message"]
        raise BadRequestError(message=response_400_message)
    if response.status_code == 500:
        response_500_request_id = response.headers["apigw-requestid"]
        raise HTTPClientError(request_id=response_500_request_id)
    if response.status_code == 422:
        response_422 = HTTPValidationError.from_dict(response.json())
        if response_422.detail[0].type.__contains__("regex"):
            report = "Invalid " + response_422.detail[0].loc[-1] + " provided"
        report = "Invalid " + response_422.detail[0].loc[-1] + " provided"
        raise ParamValidationError(report=report)
    if response.status_code == 403:
        raise UnauthorizedTokenError()
    return None


def _build_response(*, response: httpx.Response) -> Response[Union[HTTPValidationError, MessageModel]]:
    return Response(
        status_code=response.status_code,
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(response=response),
    )


def sync_detailed(
    *,
    client: AuthenticatedClient,
    json_body: UserPasswordInput,
    oblivious_user_id: str,
) -> Response[Union[HTTPValidationError, MessageModel]]:
    """Update User's Password

     API to update user's password

    Args:
        oblivious_user_id (str):
        json_body (UserPasswordInput):

    Returns:
        Response[Union[HTTPValidationError, MessageModel]]
    """

    kwargs = _get_kwargs(
        client=client,
        json_body=json_body,
        oblivious_user_id=oblivious_user_id,
    )

    response = httpx.request(
        verify=client.verify_ssl,
        **kwargs,
    )

    return _build_response(response=response)


def sync(
    *,
    client: AuthenticatedClient,
    json_body: UserPasswordInput,
    oblivious_user_id: str,
) -> Optional[Union[HTTPValidationError, MessageModel]]:
    """Update User's Password

     API to update user's password

    Args:
        oblivious_user_id (str):
        json_body (UserPasswordInput):

    Returns:
        Response[Union[HTTPValidationError, MessageModel]]
    """

    return sync_detailed(
        client=client,
        json_body=json_body,
        oblivious_user_id=oblivious_user_id,
    ).parsed


async def asyncio_detailed(
    *,
    client: AuthenticatedClient,
    json_body: UserPasswordInput,
    oblivious_user_id: str,
) -> Response[Union[HTTPValidationError, MessageModel]]:
    """Update User's Password

     API to update user's password

    Args:
        oblivious_user_id (str):
        json_body (UserPasswordInput):

    Returns:
        Response[Union[HTTPValidationError, MessageModel]]
    """

    kwargs = _get_kwargs(
        client=client,
        json_body=json_body,
        oblivious_user_id=oblivious_user_id,
    )

    async with httpx.AsyncClient(verify=client.verify_ssl) as _client:
        response = await _client.request(**kwargs)

    return _build_response(response=response)


async def asyncio(
    *,
    client: AuthenticatedClient,
    json_body: UserPasswordInput,
    oblivious_user_id: str,
) -> Optional[Union[HTTPValidationError, MessageModel]]:
    """Update User's Password

     API to update user's password

    Args:
        oblivious_user_id (str):
        json_body (UserPasswordInput):

    Returns:
        Response[Union[HTTPValidationError, MessageModel]]
    """

    return (
        await asyncio_detailed(
            client=client,
            json_body=json_body,
            oblivious_user_id=oblivious_user_id,
        )
    ).parsed
