#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Jun  5 10:48:05 2019

@author: robertson
"""
# =============================================================================
# IMPORTANT: This script is included for legasy purposes. It was how the original
# lemmatiser/stemmer was made, but is not designed to be run.
# =============================================================================

# =============================================================================
# create morpheme ---> lemma/stem mappings facilitate counting by lemma/stem 
# rather than inflection/derivation.  Additionally, any outputs of the following for loop
# will need to be adjusted by hand, i.e. because stemming results in non-dictionary root morphemes
# and some marking i.e. german mood must be retained. Code has been left in this script to make the methods 
# transparent as well as facilitate extensibility, should future researchers wish to add languages
# =============================================================================
def _make_initial_map(stem_langs=['english'],stem_features = ['adv_adj_poss','adv_adj_cert']):
    '''
    Note: only english adjectives and adverbs are stemmed, as they are derivations (ie different word classes). 
    Other features classes (i.e. present, mental_*, verb_*, *future*, and particle*, are either marked 
    only for person and number so share a root lemma (i.e. present tense verbs, mental state predicates),
    or are not marked (modal verbs, particles). Dutch and German epistemic adverbs/adjectives are mostly 
    not marked so only lemmatized, rather than stemmed (Nuyts 2000)
    '''
    #imports
    from models import MODELS
    from word_lists import WORD_LISTS
    from nltk.stem.snowball import SnowballStemmer
    
    #set up a new version to write into
    lemma_map = WORD_LISTS.copy()
    
    #load stemmers
    STEMMERS = {lang:SnowballStemmer(language=lang) for lang in stem_langs}
    
    #map the inflections derrivations on their lemmas/stems
    for lang,lang_list in lemma_map.items():
        for feature,word_list in lang_list.items():
            word_pairs = {}
            phrase_pairs = {}
            for phrase in word_list[0]:
                phrase_pairs[phrase] = phrase
            for word in word_list[1]:
                if lang in stem_langs and feature in stem_features:
                    word_pairs[word] = STEMMERS[lang].stem(word)
                else:
                    word_pairs[word] = MODELS[lang](word)[0].lemma_
            lemma_map[lang][feature] = (phrase_pairs,word_pairs)
    return lemma_map

##print
def _print_map(lemma_map,indent=4):
    #imports
    from word_lists import LANGUAGES
    from pprint import PrettyPrinter
    pp = PrettyPrinter(indent=indent)
    #print
    for lang in LANGUAGES:
        pp.pprint(lemma_map[lang])
    
# =============================================================================
# Lemma map (copy below hand edited from the output of the code above)
# =============================================================================
# =============================================================================
# English
# =============================================================================
english_lems = {   'adv_adj_cert': (   {'for sure': 'for sure'},
                        {   'assured': 'assured',
                            'assuredly': 'assured',
                            'certain': 'certain',
                            'certainly': 'certain',
                            'certainty': 'certain',
                            'clearly': 'clear',
                            'definetly': 'definite',
                            'definite': 'definite',
                            'definitely': 'definite',
                            'doubtless': 'doubtless',
                            'indubitably': 'indubitable',
                            'inevitably': 'inevitable',
                            'infallibly': 'infalable',
                            'irrefutable': 'irrefutable',
                            'irrefutably': 'irrefutable',
                            'necessarily': 'necessary',
                            'necessary': 'necessary',
                            'obvious': 'obvious',
                            'obviously': 'obvious',
                            'sure': 'sure',
                            'surely': 'sure',
                            'unavoidable': 'unavoidable',
                            'unavoidably': 'unavoidable',
                            'undeniable': 'undeniable',
                            'undeniably': 'undeniable',
                            'undoubted': 'undoubted',
                            'undoubtedly': 'undoubted',
                            'unquestionable': 'unquestionable',
                            'unquestionably': 'unquestionable'}),
    'adv_adj_poss': (   {},
                        {   'apparently': 'apparent',
                            'dubiously': 'dubiously',
                            'expectably': 'expect',
                            'expectedly': 'expect',
                            'improbability': 'improbable',
                            'improbable': 'improbable',
                            'improbably': 'improbable',
                            'likely': 'likely',
                            'maybe': 'maybe',
                            'mayhap': 'maybe',
                            'perchance': 'perchance',
                            'perhaps': 'perhaps',
                            'possibility': 'possible',
                            'possible': 'possible',
                            'possibly': 'possible',
                            'potentially': 'potential',
                            'presumably': 'presume',
                            'probability': 'probable',
                            'probable': 'probable',
                            'probably': 'probable',
                            'questionably': 'questionable',
                            'seemingly': 'seem',
                            'somewhat': 'somewhat',
                            'supposed': 'supposed',
                            'supposedly': 'supposed',
                            'uncertainly': 'uncertain'}),
    'future': (   {   'about to': 'about to',
                      'am going': 'be going to',
                      'are going': 'be going to',
                      'going to': 'be going to',
                      'is going': 'be going to'},
                  {   "'ll": 'will',
                      'shall': 'shall',
                      'theyll': 'will',
                      'wil': 'will',
                      'will': 'will'}),
    'go_future': (   {   'am going': 'be going to',
                         'are going': 'be going to',
                         'going to': 'be going to',
                         'is going': 'be going to'},
                     {}),
    'mental_cert': ({}, {'know': 'know', 'knows': 'know'}),
    'mental_poss': (   {'feel like': 'feel like'},
                       {   'believe': 'believe',
                           'believes': 'believe',
                           'believing': 'believe',
                           'doubt': 'doubt',
                           'doubting': 'doubt',
                           'doubts': 'doubt',
                           'expect': 'expect',
                           'expecting': 'expect',
                           'expects': 'expecting',
                           'guess': 'guess',
                           'guesses': 'guess',
                           'guessing': 'guess',
                           'plan': 'plan',
                           'planning': 'plan',
                           'plans': 'plan',
                           'reckon': 'reckon',
                           'reckoning': 'reckon',
                           'reckons': 'reckon',
                           'suppose': 'suppose',
                           'supposes': 'suppose',
                           'supposing': 'suppose',
                           'think': 'think',
                           'thinking': 'think',
                           'thinks': 'think'}),
    'particle': ({}, {}),
    'particle_poss': ({}, {}),
    'present': (   {},
                   {   "'m": 'be',
                       "'re": 'be',
                       'allow': 'allow',
                       'allows': 'allow',
                       'am': 'be',
                       'are': 'be',
                       'arrive': 'arrive',
                       'arrives': 'arrive',
                       'be': 'be',
                       'break': 'break',
                       'breaks': 'break',
                       'buy': 'buy',
                       'buys': 'buy',
                       'call': 'call',
                       'calls': 'call',
                       'cause': 'cause',
                       'causes': 'cause',
                       'climb': 'climb',
                       'climbs': 'climb',
                       'collapse': 'collapse',
                       'collapses': 'collapse',
                       'come': 'come',
                       'comes': 'come',
                       'congradulate': 'congratulate',
                       'congradulates': 'congratulate',
                       'congratulate': 'congratulate',
                       'congratulates': 'congratulate',
                       'crash': 'crash',
                       'crashes': 'crash',
                       'decline': 'decline',
                       'declines': 'decline',
                       'develop': 'develop',
                       'develops': 'develop',
                       'die': 'die',
                       'dies': 'die',
                       'do': 'do',
                       'does': 'do',
                       'drop': 'drop',
                       'drops': 'drop',
                       'fall': 'fall',
                       'falls': 'fall',
                       'flies': 'fly',
                       'fly': 'fly',
                       'forecast': 'forecast',
                       'forecasts': 'forecast',
                       'gain': 'gain',
                       'gains': 'gain',
                       'garden': 'garden',
                       'gardens': 'garden',
                       'get': 'get',
                       'gets': 'get',
                       'go': 'go',
                       'goes': 'go',
                       'help': 'help',
                       'helps': 'help',
                       'increase': 'increase',
                       'increases': 'increase',
                       'insist': 'insist',
                       'insists': 'insist',
                       'is': 'be',
                       'live': 'live',
                       'lives': 'life',
                       'make': 'make',
                       'makes': 'make',
                       'plan': 'plan',
                       'plans': 'plan',
                       'plummet': 'plummet',
                       'plummets': 'plummet',
                       'potter': 'potter',
                       'potters': 'potter',
                       'praise': 'praise',
                       'praises': 'praise',
                       'put': 'put',
                       'puts': 'put',
                       'rain': 'rain',
                       'rains': 'rain',
                       'receive': 'receive',
                       'receives': 'receive',
                       'rise': 'rise',
                       'rises': 'rise',
                       'risk': 'risk',
                       'risks': 'risk',
                       'say': 'say',
                       'says': 'say',
                       'sit': 'sit',
                       'sits': 'sit',
                       'spend': 'spend',
                       'spends': 'spend',
                       'stay': 'stay',
                       'stays': 'stay',
                       'suggest': 'suggest',
                       'suggests': 'suggest',
                       'thank': 'thank',
                       'thanks': 'thank',
                       'travel': 'travel',
                       'travels': 'travel',
                       'work': 'work',
                       'works': 'work'}),
    'verb_cert': (   {   'got to': 'got to',
                         'has to': 'has to',
                         'have to': 'has to'},
                     {'must': 'must'}),
    'verb_poss': (   {},
                     {   'can': 'can',
                         'could': 'could',
                         'may': 'may',
                         'might': 'might',
                         'ought': 'ought',
                         'should': 'should',
                         'sould': 'should'}),
    'will_future': (   {},
                       {   "'ll": 'will',
                           'theyll': 'will',
                           'wil': 'will',
                           'will': 'will'})}
    
    
# =============================================================================
#     Dutch lemma map
# =============================================================================
dutch_lems = {   'adv_adj_cert': (   {'wel degelijk': 'wel degelijk'},
                        {   'absoluut': 'absoluut',
                            'alleszins': 'alleszins',
                            'allicht': 'allicht',
                            'bepaald': 'bepaald',
                            'beslist': 'beslist',
                            'definitief': 'definitief',
                            'doorgansduidelijk': 'doorgansduidelijk',
                            'duidelijk': 'duidelijk',
                            'echt': 'echt',
                            'eenvoudigweg': 'eenvoudigweg',
                            'eigenlijk': 'eigenlijk',
                            'essentieel': 'essentieel',
                            'evident': 'evident',
                            'fietelijk': 'fietelijk',
                            'gedwongen': 'gedwongen',
                            'gegarandeerd': 'gegarandeerd',
                            'glashelder': 'glashelder',
                            'helder': 'helder',
                            'hoogstwaarschijnlijk': 'hoogstwaarschijnlijk',
                            'inderdaad': 'inderdaad',
                            'kennelijk': 'kennelijk',
                            'klaar': 'klaar',
                            'klaarblijkelijk': 'klaarblijkelijk',
                            'logisch': 'logisch',
                            'natuurlijk': 'natuurlijk',
                            'nodig': 'nodig',
                            'noodzakelijk': 'noodzakelijk',
                            'normaal': 'normaal',
                            'ondenkbaar': 'ondenkbaar',
                            'ongetwijfeld': 'ongetwijfeld',
                            'onmiskenbaar': 'onmiskenbaar',
                            'onmogelijk': 'onmogelijk',
                            'onomstotelijk': 'onomstotelijk',
                            'ontwijfelbaar': 'ontwijfelbaar',
                            'onvermijdelijk': 'onvermijdelijk',
                            'onwaarschijnlijk': 'onwaarschijnlijk',
                            'onweerlegbaar': 'onweerlegbaar',
                            'onwrikbaar': 'onwrikbaar',
                            'overduidelijk': 'overduidelijk',
                            'overtuigend': 'overtuigend',
                            'sowieso': 'sowieso',
                            'stellig': 'stellig',
                            'uiteraard': 'uiteraard',
                            'vanzelfsprekend': 'vanzelfspreken',
                            'vast': 'vast',
                            'verplicht': 'verplichten',
                            'voorgoed': 'voorgoed',
                            'werkelijk': 'werkelijk',
                            'wis': 'wis',
                            'zeker': 'zeker'}),
    'adv_adj_poss': (   {   'in aanmerking komendniet zeker': 'in aanmerking '
                                                              'komendniet '
                                                              'zeker',
                            'is er een kans': 'is er een kans',
                            'naar het schijnt': 'naar het schijnt',
                            'te komen': 'te komen'},
                        {   'aannemelijk': 'aannemelijk',
                            'allichet': 'allichet',
                            'bedenkelijk': 'bedenkelijk',
                            'blijkbaar': 'blijkbaar',
                            'denkelijk': 'denkelijk',
                            'geschiktlijkend': 'geschiktlijken',
                            'hypothetisch': 'hypothetisch',
                            'klaarblijkelijk': 'klaarblijkelijk',
                            'misschien': 'misschien',
                            'mogelijk': 'mogelijk',
                            'mogelijkerwijs': 'mogelijkerwijs',
                            'ogenschijnlijk': 'ogenschijnlijk',
                            'ongeveer': 'ongeveer',
                            'onzeker': 'onzeker',
                            'schijnbaar': 'schijnbaar',
                            'twijfelachtig': 'twijfelachtig',
                            'veelbelovend': 'veelbelovend',
                            'vermoedelijk': 'vermoedelijk',
                            'verwacht': 'verwacht',
                            'waarschijnlijk': 'waarschijnlijk',
                            'wellicht': 'wellicht',
                            'word': 'worden',
                            'worden': 'worden',
                            'wordt': 'worden'}),
    'future': (   {'staat op': 'staat op'},
                  {   'ga': 'gaan',
                      'gaan': 'gaan',
                      'gaanverliezen': 'gaan',
                      'gaat': 'gaan',
                      'zal': 'zullen',
                      'zul': 'zullen',
                      'zullen': 'zullen',
                      'zult': 'zullen'}),
    'go_future': (   {},
                     {   'ga': 'gaan',
                         'gaan': 'gaan',
                         'gaanverliezen': 'gaan',
                         'gaat': 'gaan'}),
    'mental_cert': ({}, {'weet': 'weten', 'weten': 'weten'}),
    'mental_poss': (   {'houden voor': 'houden voor'},
                       {   'annehm': 'annehm',
                           'annehmen': 'annehmen',
                           'annehmt': 'annehmt',
                           'betwijfelen': 'betwijfelen',
                           'betwijfelt': 'betwijfelen',
                           'denk': 'denken',
                           'denken': 'denken',
                           'denkt': 'denken',
                           'gehoord': 'horen',
                           'geloof': 'geloven',
                           'gelooft': 'geloven',
                           'geloven': 'geloven',
                           'hoor': 'horen',
                           'hoortbetwijfel': 'hoortbetwijfel',
                           'horen': 'horen',
                           'meen': 'menen',
                           'menen': 'menen',
                           'ment': 'menen',
                           'vermoed': 'vermoeden',
                           'vermoeden': 'vermoeden',
                           'vermoedt': 'vermoeden',
                           'veronderstel': 'veronderstellen',
                           'veronderstellen': 'veronderstellen',
                           'veronderstelt': 'veronderstellen',
                           'zeg': 'zeggen',
                           'zeggen': 'zeggen',
                           'zegt': 'zeggen'}),
    'particle': (   {'nou eenmaal': 'nu eenmaal', 'nu eenmaal': 'nu eenmaal'},
                    {   'eenseffe': 'eenseffe',
                        'entjes': 'entje',
                        'es': 'es',
                        'even': 'even',
                        'gewoon': 'gewoon',
                        'hoor': 'hoor',
                        'ja': 'ja',
                        'maar': 'maar',
                        'nounu': 'nounu',
                        'toch': 'toch'}),
    'particle_poss': ({'wel eens': 'wel eens'}, {'wel': 'wel'}),
    'present': (   {   'contact opnemen': 'contact opnemen',
                       'en ik in au': 'en ik in au',
                       'het stoort': 'het stoort',
                       'is het': 'is het',
                       'leuk vindt': 'leuk vindt',
                       'stort in': 'instorten',
                       'storten in': 'instorten',
                       'storten inen': 'instorten',
                       'val om': 'omvallen',
                       'vallen om': 'omvallen',
                       'valt om': 'omvallen',
                       'ze halen het': 'ze halen het'},
                   {   '.is': 'zijn',
                       'accepteer': 'accepteren',
                       'accepteert': 'accepteren',
                       'accepteren': 'accepteren',
                       'bedank': 'bedanken',
                       'bedanken': 'bedanken',
                       'bedankt': 'bedanken',
                       'bekijk': 'bekijk',
                       'bekijken': 'bekijken',
                       'bekijkt': 'bekijken',
                       'bel': 'bellen',
                       'bellen': 'bellen',
                       'belt': 'bellen',
                       'ben': 'zijn',
                       'bent': 'zijn',
                       'bereik': 'bereiken',
                       'bereiken': 'bereiken',
                       'bereikt': 'bereiken',
                       'blijf': 'blijven',
                       'blijft': 'blijven',
                       'blijven': 'blijven',
                       'breek': 'breken',
                       'breekt': 'breken',
                       'breken': 'breken',
                       'daal': 'dalen',
                       'daalt': 'dalen',
                       'dalen': 'dalen',
                       'dank': 'danken',
                       'danken': 'danken',
                       'dankt': 'danken',
                       'doden': 'doden',
                       'doe': 'doen',
                       'doen': 'doen',
                       'doet': 'doen',
                       'dood': 'dood',
                       'doodgaan': 'doden',
                       'doodt': 'doden',
                       'eet': 'eten',
                       'eten': 'eten',
                       'explodeer': 'exploderen',
                       'explodeert': 'exploderen',
                       'exploderen': 'exploderen',
                       'haal': 'halen',
                       'haalt': 'halen',
                       'halen': 'halen',
                       'heb': 'hebben',
                       'hebben': 'hebben',
                       'hebt': 'hebben',
                       'heeft': 'hebben',
                       'heel': 'helen',
                       'heelt': 'helen',
                       'helen': 'helen',
                       'ingestort': 'instorten',
                       'instort': 'instorten',
                       'is': 'zijn',
                       'is.': 'zijn',
                       'kijk': 'kijken',
                       'kijken': 'kijken',
                       'kijkt': 'kijken',
                       'knap': 'knappen',
                       'knappen': 'knappen',
                       'knapt': 'knappen',
                       'kom': 'komen',
                       'komen': 'komen',
                       'komt': 'komen',
                       'koop': 'kopen',
                       'koopt': 'kopen',
                       'kopen': 'kopen',
                       'krijg': 'krijgen',
                       'krijgen': 'krijgen',
                       'krijgt': 'krijgen',
                       'laat': 'laten',
                       'latten': 'laten',
                       'maak': 'maken',
                       'maaken': 'maken',
                       'maakt': 'maken',
                       'maken': 'maken',
                       'neem': 'nemen',
                       'neemt': 'nemen',
                       'nemen': 'nemen',
                       'omval': 'omvallen',
                       'omvalen': 'omvallen',
                       'omvalt': 'omvallen',
                       'regen': 'regenen',
                       'regenen': 'regenen',
                       'regent': 'regenen',
                       'reis': 'reizen',
                       'reist': 'reizen',
                       'reizen': 'reizen',
                       'resideer': 'resideren',
                       'resideert': 'resideren',
                       'resideren': 'resideren',
                       'rij': 'rijden',
                       'rijd': 'rijden',
                       'rijden': 'rijden',
                       'rijdt': 'rijden',
                       'slijt': 'slijten',
                       'slijten': 'slijten',
                       'sneeuw': 'sneeuwen',
                       'sneeuwen': 'sneeuwen',
                       'sneeuwt': 'sneeuwen',
                       'sort': 'storten',
                       'sotrt': 'storten',
                       'spaar': 'sparen',
                       'spaart': 'sparen',
                       'sparen': 'sparen',
                       'stel': 'stellen',
                       'stellen': 'stellen',
                       'stelt': 'stellen',
                       'sterf': 'sterven',
                       'sterft': 'sterven',
                       'sterven': 'sterven',
                       'stijg': 'stijgen',
                       'stijgen': 'stijgen',
                       'stijgt': 'stijgen',
                       'stjgt': 'stjgen',
                       'stort': 'storten',
                       'storten': 'storten',
                       'strort': 'strort',
                       'val': 'vallen',
                       'vallen': 'vallen',
                       'valt': 'vallen',
                       'verdubbel': 'verdubbelen',
                       'verdubbelen': 'verdubbelen',
                       'verdubbelt': 'verdubbelen',
                       'vergang': 'vergangen',
                       'vergangen': 'vergangen',
                       'vergangt': 'vergangt',
                       'verhuis': 'verhuizen',
                       'verhuist': 'verhuizen',
                       'verhuizen': 'verhuizen',
                       'verlies': 'verlies',
                       'verliest': 'verliezen',
                       'verliezen': 'verliezen',
                       'verminder': 'verminderen',
                       'verminderen': 'verminder',
                       'vermindert': 'verminderen',
                       'verslijt': 'verslijten',
                       'verslijten': 'verslijten',
                       'verslitjen': 'verslitj',
                       'verspil': 'verspillen',
                       'verspillen': 'verspillen',
                       'verspilt': 'verspillen',
                       'vervang': 'vervang',
                       'vervangen': 'vervangen',
                       'vervangt': 'vervangen',
                       'vind': 'vinden',
                       'vinden': 'vinden',
                       'vindt': 'vinden',
                       'voel': 'voelen',
                       'voelen': 'voelen',
                       'voelt': 'voelen',
                       'weetblijft': 'blijven',
                       'werk': 'werken',
                       'werken': 'werken',
                       'werkt': 'werken',
                       'wetkt': 'werken',
                       'win': 'winnen',
                       'winnen': 'winnen',
                       'wint': 'winnen',
                       'wonen': 'wonen',
                       'wonnen': 'winnen',
                       'woon': 'wonen',
                       'woont': 'wonen',
                       'wwon': 'wonen',
                       'zie': 'zien',
                       'zien': 'zien',
                       'ziet': 'zien',
                       'zij': 'zijn',
                       'zijn': 'zijn',
                       'zinnen': 'zijn',
                       'zit': 'zitten',
                       'zitten': 'zitten',
                       'zwel': 'zwellen',
                       'zwellen': 'zwellen',
                       'zwelt': 'zwellen'}),
    'verb_cert': (   {},
                     {   'gemoeten': 'moeten',
                         'moest': 'moeten',
                         'moesten': 'moeten',
                         'moet': 'moeten',
                         'moeten': 'moeten'}),
    'verb_poss': (   {},
                     {   'gekund': 'kunnen',
                         'kan': 'kunnen',
                         'kans': 'kunnen',
                         'kon': 'kunnen',
                         'kun': 'kunnen',
                         'kunnen': 'kunnen',
                         'kunt': 'kunnen',
                         'mag': 'mogen',
                         'mogen': 'mogen',
                         'vermogen': 'mogen',
                         'vermoogd': 'mogen'}),
    'will_future': (   {},
                       {   'zal': 'zullen',
                           'zul': 'zullen',
                           'zullen': 'zullen',
                           'zult': 'zullen'})}
    
    
# =============================================================================
# German lemmatization
# =============================================================================
german_lems = {'adv_adj_cert': (   {   'auf jeden fall': 'auf jeden fall',
                            'klipp und klar': 'klipp und klar'},
                        {   'aufjedenfall': 'aufjedenfall',
                            'augenscheinlich': 'augenscheinlich',
                            'bestimmt': 'bestimmen',
                            'definitiv': 'definitiv',
                            'deutlich': 'deutlich',
                            'eindeutig': 'eindeutig',
                            'gewiss': 'gewiss',
                            'jedenfalls': 'jedenfalls',
                            'klar': 'klar',
                            'offensichtlich': 'offensichtlich',
                            'sicher': 'sicher',
                            'sicherlich': 'sicherlich',
                            'zweifellos': 'zweifellos',
                            'zweifelsohne': 'zweifelsohne'}),
    'adv_adj_poss': (   {'unter umständen': 'unter umständen'},
                        {   'annehmbar': 'annehmbar',
                            'anscheinend': 'anscheinend',
                            'eventuell': 'eventuell',
                            'gegebenenfalls': 'gegebenenfalls',
                            'möglich': 'möglich',
                            'möglicherweise': 'möglicherweise',
                            'offenbar': 'offenbar',
                            'scheinbar': 'scheinbar',
                            'vermutlich': 'vermutlich',
                            'vielleicht': 'vielleicht',
                            'wahrscheinlich': 'wahrscheinlich',
                            'womöglich': 'womöglich'}),
    'future': (   {},
                  {   'werde': 'werden',
                      'werden': 'werden',
                      'werdet': 'werden',
                      'wird': 'werden',
                      'wirst': 'werden'}),
    'go_future': ({}, {}),
    'mental_cert': (   {},
                       {   'weiß': 'wissen',
                           'weißt': 'wissen',
                           'wissen': 'wissen',
                           'wisst': 'wissen'}),
    'mental_poss': (   {   'nehme an': 'aannemen',
                           'nehmen an': 'aannemen',
                           'nehmt an': 'aannemen',
                           'nimmst an': 'aannemen',
                           'nimmt an': 'aannemen'},
                       {   'denke': 'denken',
                           'denken': 'denken',
                           'denkst': 'denken',
                           'denkt': 'denken',
                           'glaube': 'glauben',
                           'glauben': 'glauben',
                           'glaubst': 'glauben',
                           'glaubt': 'glauben',
                           'meine': 'meinen',
                           'meinen': 'meinen',
                           'meinst': 'meinen',
                           'meint': 'meinen',
                           'rechne': 'rechnen',
                           'rechnen': 'rechnen',
                           'rechnest': 'rechnen',
                           'rechnet': 'rechnen',
                           'sage': 'sagen',
                           'sagen': 'sagen',
                           'sagst': 'sagen',
                           'sagt': 'sagen',
                           'vermute': 'vermuten',
                           'vermuten': 'vermuten',
                           'vermutest': 'vermuten',
                           'vermutet': 'vermuten'}),
    'particle': (   {},
                    {   'aber': 'aber',
                        'auch': 'auch',
                        'bloß': 'bloß',
                        'denn': 'denn',
                        'doch': 'doch',
                        'eben': 'eben',
                        'eigentlich': 'eigentlich',
                        'einfach': 'einfach',
                        'erst': 'erst',
                        'etwa': 'etwa',
                        'halt': 'halt',
                        'ja': 'ja',
                        'mal': 'mal',
                        'nun': 'nun',
                        'nur': 'nur',
                        'ruhig': 'ruhig',
                        'schon': 'schon',
                        'vielleicht': 'vielleicht'}),
    'particle_poss': ({}, {'wohl': 'wohl'}),
    'present': (   {   'gehe auf': 'aufgehen',
                       'gehe aus': 'ausgehen',
                       'gehen auf': 'aufgehen',
                       'gehen aus': 'ausgehen',
                       'gehst auf': 'aufgehen',
                       'gehst aus': 'ausgehen',
                       'geht auf': 'aufgehen',
                       'geht aus': 'ausgehen',
                       'nutze ab': 'abnutzen',
                       'nutzen ab': 'abnutzen',
                       'nutzt ab': 'abnutzen',
                       'rufe an': 'anrufen',
                       'rufen an': 'anrufen',
                       'rufst an': 'anrufen',
                       'ruft an': 'anrufen'},
                   {   'arbeite': 'arbeiten',
                       'arbeiten': 'arbeiten',
                       'arbeitest': 'arbeiten',
                       'arbeitet': 'arbeiten',
                       'bekomme': 'bekommen',
                       'bekommen': 'bekommen',
                       'bekommst': 'bekommen',
                       'bekommt': 'bekommen',
                       'bin': 'sein',
                       'bist': 'sein',
                       'fuehlst': 'fühlen',
                       'fühle': 'fühlen',
                       'fühlen': 'fühlen',
                       'fühlst': 'fühlen',
                       'fühlt': 'fühlen',
                       'gehe': 'gehen',
                       'gehen': 'gehen',
                       'gehst': 'gehen',
                       'geht': 'gehen',
                       'gewinne': 'gewinnen',
                       'gewinnen': 'gewinnen',
                       'gewinnst': 'gewinnen',
                       'gewinnt': 'gewinnen',
                       'habe': 'haben',
                       'haben': 'haben',
                       'habt': 'haben',
                       'hast': 'haben',
                       'hat': 'haben',
                       'ist': 'sein',
                       'kaufe': 'kaufen',
                       'kaufen': 'kaufen',
                       'kaufst': 'kaufen',
                       'kauft': 'kaufen',
                       'komme': 'kommen',
                       'kommen': 'kommen',
                       'kommst': 'kommen',
                       'kommt': 'kommen',
                       'lebe': 'leben',
                       'leben': 'leben',
                       'lebst': 'leben',
                       'lebt': 'leben',
                       'liebe': 'lieben',
                       'lieben': 'lieben',
                       'liebst': 'lieben',
                       'liebt': 'lieben',
                       'mache': 'machen',
                       'machen': 'machen',
                       'machst': 'machen',
                       'macht': 'machen',
                       'nutze': 'nutzen',
                       'nutzen': 'nutzen',
                       'nutzt': 'nutzen',
                       'regne': 'regnen',
                       'regnen': 'regnen',
                       'regnest': 'regnen',
                       'regnet': 'regnen',
                       'reise': 'reisen',
                       'reisen': 'reisen',
                       'reist': 'reisen',
                       'rufe': 'rufen',
                       'rufen': 'rufen',
                       'rufst': 'rufen',
                       'ruft': 'rufen',
                       'seid': 'sein',
                       'sein': 'sein',
                       'sind': 'sein',
                       'steige': 'steigen',
                       'steigen': 'steigen',
                       'steigst': 'steigen',
                       'steigt': 'steigen',
                       'sterbe': 'sterben',
                       'sterben': 'sterben',
                       'sterbt': 'sterben',
                       'stirbst': 'sterben',
                       'stirbt': 'sterben'}),
    'verb_cert': (   {},
                     {   'muss': 'müssen_IND',
                         'musst': 'müssen_IND',
                         'muß': 'müssen_IND',
                         'mußt': 'müssen_IND',
                         'müssen': 'müssen_IND',
                         'müsst': 'müssen_IND',
                         'müßen': 'müssen_IND',
                         'müßt': 'müssen_IND'}),
    'verb_poss': (   {},
                     {   'dürfte': 'dürfen_SUBJ',
                         'dürften': 'dürfen_SUBJ',
                         'dürftest': 'dürfen_SUBJ',
                         'dürftet': 'dürfen_SUBJ',
                         'kann': 'können_IND',
                         'kannst': 'können_IND',
                         'können': 'können_IND',
                         'könnt': 'können_IND',
                         'könnte': 'können_SUBJ',
                         'könnten': 'können_SUBJ',
                         'könntest': 'können_SUBJ',
                         'könntet': 'können_SUBJ',
                         'mag': 'mögen_IND',
                         'magst': 'mögen_IND',
                         'mögen': 'mögen_IND',
                         'mögt': 'mögen_IND',
                         'müsste': 'müssen_SUBJ',
                         'müssten': 'müssen_SUBJ',
                         'müsstest': 'müssen_SUBJ',
                         'müsstet': 'müssen_SUBJ',
                         'müßte': 'müssen_SUBJ',
                         'müßten': 'müssen_SUBJ',
                         'müßtest': 'müssen_SUBJ',
                         'müßtet': 'müssen_SUBJ',
                         'soll': 'sollen_IND',
                         'sollen': 'sollen_IND',
                         'sollst': 'sollen_IND',
                         'sollt': 'sollen_IND',
                         'sollte': 'sollen_SUBJ',
                         'sollten': 'sollen_SUBJ',
                         'solltest': 'sollen_SUBJ',
                         'solltet': 'sollen_SUBJ'}),
    'will_future': (   {},
                       {   'werde': 'werden',
                           'werden': 'werden',
                           'werdet': 'werden',
                           'wird': 'werden',
                           'wirst': 'werden'})}

# =============================================================================
# create lemma map with all languages
# =============================================================================
LEMMA_MAP = {'english':english_lems,
             'dutch':dutch_lems,
             'german':german_lems
             }
# =============================================================================
# save it as a pickle
# =============================================================================
if __name__ == '__main__':
    #from ftr_classifier.word_lists import _save_obj
    #_save_obj(LEMMA_MAP,'lemma_map')
    pass


