# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/00_data_acquisition.ipynb (unless otherwise specified).

__all__ = ['load_postgres_env', 'load_postgres_engine', 'load_statcast_db_schema', 'query_statcast', 'query_db']

# Internal Cell
from pybaseball import statcast
import pandas as pd
from fastscript import *
from sqlalchemy import create_engine
import psycopg2
from os import path
import os
from dotenv import find_dotenv, load_dotenv
import io

# Cell

def load_postgres_env():
    # retrieving engironment variables
    load_dotenv(find_dotenv(), override=True)
    dbname = os.getenv("POSTGRES_DB")
    user = os.getenv("POSTGRES_USER")
    password = os.getenv("POSTGRES_PASSWORD")
    host = os.getenv("POSTGRES_HOST")
    port = os.getenv("POSTGRES_PORT")
    return dbname, user, password, host, port

def load_postgres_engine():
    # postgres env variables
    dbname, user, password, host, port = load_postgres_env()

    # creating remote db connection
    engine = create_engine(f'postgresql://{user}:{password}@{host}/{dbname}')

    return engine

# Cell

def load_statcast_db_schema():
    schema = f"""
                index bigint,
                pitch_type text,
                game_date timestamp without time zone,
                release_speed double precision,
                release_pos_x double precision,
                release_pos_z double precision,
                player_name text,
                batter double precision,
                pitcher double precision,
                events text,
                description text,
                spin_dir double precision,
                spin_rate_deprecated double precision,
                break_angle_deprecated double precision,
                break_length_deprecated double precision,
                zone double precision,
                des text,
                game_type text,
                stand text,
                p_throws text,
                home_team text,
                away_team text,
                type text,
                hit_location double precision,
                bb_type text,
                balls double precision,
                strikes double precision,
                game_year double precision,
                pfx_x double precision,
                pfx_z double precision,
                plate_x double precision,
                plate_z double precision,
                on_3b double precision,
                on_2b double precision,
                on_1b double precision,
                outs_when_up double precision,
                inning double precision,
                inning_topbot text,
                hc_x double precision,
                hc_y double precision,
                tfs_deprecated double precision,
                tfs_zulu_deprecated double precision,
                fielder_2 double precision,
                umpire double precision,
                sv_id text,
                vx0 double precision,
                vy0 double precision,
                vz0 double precision,
                ax double precision,
                ay double precision,
                az double precision,
                sz_top double precision,
                sz_bot double precision,
                hit_distance_sc double precision,
                launch_speed double precision,
                launch_angle double precision,
                effective_speed double precision,
                release_spin_rate double precision,
                release_extension double precision,
                game_pk double precision,
                pitcher_1 double precision,
                fielder_2_1 double precision,
                fielder_3 double precision,
                fielder_4 double precision,
                fielder_5 double precision,
                fielder_6 double precision,
                fielder_7 double precision,
                fielder_8 double precision,
                fielder_9 double precision,
                release_pos_y double precision,
                estimated_ba_using_speedangle double precision,
                estimated_woba_using_speedangle double precision,
                woba_value double precision,
                woba_denom double precision,
                babip_value double precision,
                iso_value double precision,
                launch_speed_angle double precision,
                at_bat_number double precision,
                pitch_number double precision,
                pitch_name text,
                home_score double precision,
                away_score double precision,
                bat_score double precision,
                fld_score double precision,
                post_away_score double precision,
                post_home_score double precision,
                post_bat_score double precision,
                post_fld_score double precision,
                if_fielding_alignment text,
                of_fielding_alignment text
            """
    return schema

# Cell


@call_parse
def query_statcast(
    start_dt: Param(help="Beginning date to pull data from", type=str) = None,
    end_dt: Param(help="End date to pull data from", type=str) = None,
    team: Param(help="Abbreviation for team of interest", type=str) = None,
    verbose: Param(
        help="Whether or not to print verbose updates", type=bool_arg
    ) = True
):
    """
    Callable from the command-line or in Python. Pulls pitch-level MLB data from [statcast](https://baseballsavant.mlb.com/statcast_search).
    Saves to a pre-instantiated Postgres DB in AWS RDS.
    By default, drops the table.

    * inputs:
        - `start_dt`: `str`, Beginning date to pull data from = None
        - `end_dt`: `str`, End date to pull data from = None
        - `team`: `str`, abbreviation for team of interest = None
        - `verbose`: `bool`, Whether or not to print verbose updates

    * outputs:
        - None
    """
    # creating remote db engine
    engine = load_postgres_engine()

    # pulling data from statcast
    data = statcast(start_dt=start_dt, end_dt=end_dt, team=team, verbose=verbose)
    data.columns = [col.replace(".", "_") if "." in col else col for col in list(data.columns)]

    # syncing data to AWS RDS
    # below would be most readable, but it is _super_ slow
    # data.to_sql(f"statcast_{start_dt[:4]}", connection, if_exists=if_exists)

    # creating remote db connection
    conn = engine.raw_connection()
    cur = conn.cursor()

    # create the table
    cur.execute(f'drop table if exists statcast_{start_dt[:4]}')
    cur.execute(f"create table statcast_{start_dt[:4]}({load_statcast_db_schema()});")

    # copy data
    f = io.StringIO()
    data.to_csv(f, index=False, header=False, sep=";")
    f.seek(0)
    cur.copy_from(f, f"statcast_{start_dt[:4]}", sep=';', null="", columns=list(data.columns))
    conn.commit()

    return None


# Cell


def query_db(
    year: str = "2019",
    columns: str = "*",
    limit: int = None,
    verbose: bool = True,
):
    """
    Queries a sqlite db file. Assumes that it's been created by `query_statcast`.
    Only queries for a single year at a time.

    * intputs:
        - `year`: `str`, year of data to query
        - `columns`: `str`, which columns from the [statcast data](https://baseballsavant.mlb.com/csv-docs) to include in table
        - `limit`: `int`, the maximum number of rows to retrieve ([postgresql documentation](https://www.postgresql.org/docs/8.1/queries-limit.html))
        - `verbose`: `bool`, Whether or not to print verbose updates

    * output:
        - `df`: `pd.DataFrame`, DataFrame populated with data queried from database
    """
    if verbose:
        print(f"querying year {year} from db now.")

    engine = load_postgres_engine()

    query = f"""select {columns}
                from statcast_{year}"""
    if limit:
        query += f" limit {round(limit)}"

    # if year is not in db, return empty pd.DataFrame
    try:
        with engine.connect() as connection:
            df = pd.read_sql_query(query, connection)
    except:
        print(f"Year {year} not in database. Returning empty dataframe.")
        df = pd.DataFrame()

    return df
