import { Duration } from 'aws-cdk-lib';
import { IVpc, IInstance, ISecurityGroup, SubnetSelection } from 'aws-cdk-lib/aws-ec2';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { DockerVolumes } from './gitlab-runner-interfaces';
/**
 * GitlabContainerRunner Props.
 */
export interface GitlabContainerRunnerProps {
    /**
     * Gitlab token for the Register Runner .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN' });
     *
     * @default - You must to give the token !!!
     *
     */
    readonly gitlabtoken: string;
    /**
     * Image URL of Gitlab Runner.
     *
     * @example
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', gitlabRunnerImage: 'gitlab/gitlab-runner:alpine' });
     *
     * @default public.ecr.aws/gitlab/gitlab-runner:alpine
     *
     */
    readonly gitlabRunnerImage?: string;
    /**
     * Runner default EC2 instance type.
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', ec2type: 't3.small' });
     *
     * @default - t3.micro
     *
     */
    readonly ec2type?: string;
    /**
     * VPC for the Gitlab Runner .
     *
     * @example
     * const newvpc = new Vpc(stack, 'NEWVPC', {
     *   cidr: '10.1.0.0/16',
     *   maxAzs: 2,
     *   subnetConfiguration: [{
     *     cidrMask: 26,
     *     name: 'RunnerVPC',
     *     subnetType: SubnetType.PUBLIC,
     *   }],
     *   natGateways: 0,
     * });
     *
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', selfvpc: newvpc });
     *
     * @default - new VPC will be created , 1 Vpc , 2 Public Subnet .
     *
     */
    readonly selfvpc?: IVpc;
    /**
     * IAM role for the Gitlab Runner Instance .
     *
     * @example
     * const role = new Role(stack, 'runner-role', {
     *   assumedBy: new ServicePrincipal('ec2.amazonaws.com'),
     *   description: 'For Gitlab EC2 Runner Test Role',
     *   roleName: 'Myself-Runner-Role',
     * });
     *
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', ec2iamrole: role });
     *
     * @default - new Role for Gitlab Runner Instance , attach AmazonSSMManagedInstanceCore Policy .
     *
     */
    readonly ec2iamrole?: IRole;
    /**
     * Gitlab Runner register tag1  .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', tag1: 'aa' });
     *
     * @deprecated - use tags ['runner', 'gitlab', 'awscdk']
     *
     * @default - tag1: gitlab .
     *
     */
    readonly tag1?: string;
    /**
     * Gitlab Runner register tag2  .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', tag2: 'bb' });
     *
     * @deprecated - use tags ['runner', 'gitlab', 'awscdk']
     *
     * @default - tag2: awscdk .
     *
     */
    readonly tag2?: string;
    /**
     * Gitlab Runner register tag3  .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', tag3: 'cc' });
     *
     * @deprecated - use tags ['runner', 'gitlab', 'awscdk']
     *
     * @default - tag3: runner .
     *
     */
    readonly tag3?: string;
    /**
     * tags for the runner
     *
     * @default - ['runner', 'gitlab', 'awscdk']
     */
    readonly tags?: string[];
    /**
     * Gitlab Runner register url .
     *
     * @example
     * const runner = new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN',gitlaburl: 'https://gitlab.com/'});
     *
     * @default - gitlaburl='https://gitlab.com/' , please use https://yourgitlab.com/ do not use https://yourgitlab.com
     *
     */
    readonly gitlaburl?: string;
    /**
     * Gitlab Runner instance EBS size .
     *
     * @example
     * const runner = new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN',ebsSize: 100});
     *
     * @default - ebsSize=60
     *
     */
    readonly ebsSize?: number;
    /**
     * Gitlab Runner instance Use Spot Fleet or not ?!.
     *
     * @example
     * const runner = new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN',spotFleet: true});
     *
     * @default - spotFleet=false
     *
     */
    readonly spotFleet?: boolean;
    /**
     * SSH key name
     *
     * @default - no ssh key will be assigned , !!! only support spotfleet runner !!! .
     */
    readonly keyName?: string;
    /**
     * The behavior when a Spot Runner Instance is interrupted
     *
     * @default - InstanceInterruptionBehavior.TERMINATE , !!! only support spotfleet runner !!! .
     */
    readonly instanceInterruptionBehavior?: InstanceInterruptionBehavior;
    /**
     * the time when the spot fleet allocation expires
     *
     * @default - no expiration , !!! only support spotfleet runner !!! .
     */
    readonly validUntil?: string;
    /**
     * VPC subnet for the spot fleet
     *
     * @example
     * const vpc = new Vpc(stack, 'nat', {
     * natGateways: 1,
     * maxAzs: 2,
     * });
     * const runner = new GitlabContainerRunner(stack, 'testing', {
     *   gitlabtoken: 'GITLAB_TOKEN',
     *   ec2type: 't3.large',
     *   ec2iamrole: role,
     *   ebsSize: 100,
     *   selfvpc: vpc,
     *   vpcSubnet: {
     *     subnetType: SubnetType.PUBLIC,
     *   },
     * });
     *
     * @default - public subnet
     */
    readonly vpcSubnet?: SubnetSelection;
    /**
     * add another Gitlab Container Runner Docker Volumes Path at job runner runtime.
     *
     * more detail see https://docs.gitlab.com/runner/configuration/advanced-configuration.html#the-runnersdocker-section
     *
     * @default - already mount "/var/run/docker.sock:/var/run/docker.sock"
     *
     * @example
     * dockerVolumes: [
     *   {
     *     hostPath: '/tmp/cache',
     *     containerPath: '/tmp/cache',
     *   },
     * ],
     */
    readonly dockerVolumes?: DockerVolumes[];
}
/**
 * InstanceInterruptionBehavior enum.
 */
export declare enum InstanceInterruptionBehavior {
    /**
     * hibernate
     */
    HIBERNATE = "hibernate",
    /**
     * stop
     */
    STOP = "stop",
    /**
     * terminate
     */
    TERMINATE = "terminate"
}
/**
 * GitlabContainerRunner Construct for create a Gitlab Runner.
 */
export declare class GitlabContainerRunner extends Construct {
    /**
     * The IAM role assumed by the Runner instance .
     */
    readonly runnerRole: IRole;
    /**
     * This represents a Runner EC2 instance , !!! only support On-demand runner instance !!!
     */
    readonly runnerEc2: IInstance;
    /**
     * The EC2 runner's vpc.
     */
    readonly vpc: IVpc;
    /**
     * The time when the the fleet allocation will expire , !!! only support spotfleet runner !!!
     */
    private validUntil?;
    /**
     * The EC2 runner's default SecurityGroup.
     */
    readonly defaultRunnerSG: ISecurityGroup;
    /**
     * SpotFleetRequestId for this spot fleet , !!! only support spotfleet runner !!!
     */
    readonly spotFleetRequestId: string;
    /**
     * the first instance id in this fleet , !!! only support spotfleet runner !!!
     */
    readonly spotFleetInstanceId: string;
    private cfnSpotFleet?;
    constructor(scope: Construct, id: string, props: GitlabContainerRunnerProps);
    /**
     * Add expire time function for spotfleet runner !!! .
     *
     * @param duration - Block duration.
     */
    expireAfter(duration: Duration): void;
    private dockerVolumesList;
    /**
     * @param props
     * @param tokenParameterStoreName - the tokenParameterStoreName to put gitlab runner token.
     * @returns Array.
     */
    createUserData(props: GitlabContainerRunnerProps, tokenParameterStoreName: string): string[];
}
