"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NatInstanceImage = exports.NatInstanceProvider = exports.NatProvider = exports.NatTrafficDirection = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * Direction of traffic to allow all by default.
 */
var NatTrafficDirection;
(function (NatTrafficDirection) {
    /**
     * Allow all outbound traffic and disallow all inbound traffic.
     */
    NatTrafficDirection["OUTBOUND_ONLY"] = "OUTBOUND_ONLY";
    /**
     * Allow all outbound and inbound traffic.
     */
    NatTrafficDirection["INBOUND_AND_OUTBOUND"] = "INBOUND_AND_OUTBOUND";
    /**
     * Disallow all outbound and inbound traffic.
     */
    NatTrafficDirection["NONE"] = "NONE";
})(NatTrafficDirection = exports.NatTrafficDirection || (exports.NatTrafficDirection = {}));
/**
 * NAT providers
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 *
 */
class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     */
    static gateway(props = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_NatGatewayProps(props);
        return new NatGatewayProvider(props);
    }
    /**
     * Use NAT instances to provide NAT services for your VPC
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     */
    static instance(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_NatInstanceProps(props);
        return new NatInstanceProvider(props);
    }
}
exports.NatProvider = NatProvider;
_a = JSII_RTTI_SYMBOL_1;
NatProvider[_a] = { fqn: "@aws-cdk/aws-ec2.NatProvider", version: "1.145.0" };
/**
 * Provider for NAT Gateways
 */
class NatGatewayProvider extends NatProvider {
    constructor(props = {}) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        if (this.props.eipAllocationIds != null
            && !core_1.Token.isUnresolved(this.props.eipAllocationIds)
            && this.props.eipAllocationIds.length < options.natSubnets.length) {
            throw new Error(`Not enough NAT gateway EIP allocation IDs (${this.props.eipAllocationIds.length} provided) for the requested subnet count (${options.natSubnets.length} needed).`);
        }
        // Create the NAT gateways
        let i = 0;
        for (const sub of options.natSubnets) {
            const eipAllocationId = this.props.eipAllocationIds ? pickN(i, this.props.eipAllocationIds) : undefined;
            const gateway = sub.addNatGateway(eipAllocationId);
            this.gateways.add(sub.availabilityZone, gateway.ref);
            i++;
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az);
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.NAT_GATEWAY,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
}
/**
 * NAT provider which uses NAT Instances
 */
class NatInstanceProvider extends NatProvider {
    constructor(props) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_NatInstanceProps(props);
        if (props.defaultAllowedTraffic !== undefined && props.allowAllTraffic !== undefined) {
            throw new Error('Can not specify both of \'defaultAllowedTraffic\' and \'defaultAllowedTraffic\'; prefer \'defaultAllowedTraffic\'');
        }
    }
    configureNat(options) {
        var _d, _e, _f;
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ConfigureNatOptions(options);
        const defaultDirection = (_d = this.props.defaultAllowedTraffic) !== null && _d !== void 0 ? _d : (((_e = this.props.allowAllTraffic) !== null && _e !== void 0 ? _e : true) ? NatTrafficDirection.INBOUND_AND_OUTBOUND : NatTrafficDirection.OUTBOUND_ONLY);
        // Create the NAT instances. They can share a security group and a Role.
        const machineImage = this.props.machineImage || new NatInstanceImage();
        this._securityGroup = (_f = this.props.securityGroup) !== null && _f !== void 0 ? _f : new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
            allowAllOutbound: isOutboundAllowed(defaultDirection),
        });
        this._connections = new connections_1.Connections({ securityGroups: [this._securityGroup] });
        if (isInboundAllowed(defaultDirection)) {
            this.connections.allowFromAnyIpv4(port_1.Port.allTraffic());
        }
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: this._securityGroup,
                role,
                keyName: this.props.keyName,
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    /**
     * The Security Group associated with the NAT instances
     */
    get securityGroup() {
        if (!this._securityGroup) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'securityGroup\'');
        }
        return this._securityGroup;
    }
    /**
     * Manage the Security Groups associated with the NAT instances
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'connections\'');
        }
        return this._connections;
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1].instanceId }));
    }
    configureSubnet(subnet) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnet(subnet);
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az).instanceId;
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.INSTANCE,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
}
exports.NatInstanceProvider = NatInstanceProvider;
_b = JSII_RTTI_SYMBOL_1;
NatInstanceProvider[_b] = { fqn: "@aws-cdk/aws-ec2.NatInstanceProvider", version: "1.145.0" };
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
/**
 * Machine image representing the latest NAT instance image
 *
 *
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
_c = JSII_RTTI_SYMBOL_1;
NatInstanceImage[_c] = { fqn: "@aws-cdk/aws-ec2.NatInstanceImage", version: "1.145.0" };
function isOutboundAllowed(direction) {
    return direction === NatTrafficDirection.INBOUND_AND_OUTBOUND ||
        direction === NatTrafficDirection.OUTBOUND_ONLY;
}
function isInboundAllowed(direction) {
    return direction === NatTrafficDirection.INBOUND_AND_OUTBOUND;
}
/**
 * Token-aware pick index function
 */
function pickN(i, xs) {
    if (core_1.Token.isUnresolved(xs)) {
        return core_1.Fn.select(i, xs);
    }
    if (i >= xs.length) {
        throw new Error(`Cannot get element ${i} from ${xs}`);
    }
    return xs[i];
}
//# sourceMappingURL=data:application/json;base64,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