"""
svgen - A module for working with colors.
"""

# built-in
from typing import Dict, NamedTuple

# internal
from svgen.color.conversion import hsl_to_rgb, rgb_to_hsl
from svgen.color.hsl import Hsl
from svgen.color.rgb import Rgb

CSS_COLORS: Dict[str, str] = {
    # CSS Level 1
    "black": "#000000",
    "silver": "#c0c0c0",
    "gray": "#808080",
    "white": "#ffffff",
    "maroon": "#800000",
    "red": "#ff0000",
    "purple": "#800080",
    "fuchsia": "#ff00ff",
    "green": "#008000",
    "lime": "#00ff00",
    "olive": "#808000",
    "yellow": "#ffff00",
    "navy": "#000080",
    "blue": "#0000ff",
    "teal": "#008080",
    "aqua": "#00ffff",
    # CSS Level 2
    "orange": "#ffa500",
    # CSS Level 3
    "aliceblue": "#f0f8ff",
    "antiquewhite": "#faebd7",
    "aquamarine": "#7fffd4",
    "azure": "#f0ffff",
    "beige": "#f5f5dc",
    "bisque": "#ffe4c4",
    "blanchedalmond": "#ffebcd",
    "blueviolet": "#8a2be2",
    "brown": "#a52a2a",
    "burlywood": "#deb887",
    "cadetblue": "#5f9ea0",
    "chartreuse": "#7fff00",
    "chocolate": "#d2691e",
    "coral": "#ff7f50",
    "cornflowerblue": "#6495ed",
    "cornsilk": "#fff8dc",
    "crimson": "#dc143c",
    "darkblue": "#00008b",
    "darkcyan": "#008b8b",
    "darkgoldenrod": "#b8860b",
    "darkgray": "#a9a9a9",
    "darkgreen": "#006400",
    "darkkhaki": "#bdb76b",
    "darkmagenta": "#8b008b",
    "darkolivegreen": "#556b2f",
    "darkorange": "#ff8c00",
    "darkorchid": "#9932cc",
    "darkred": "#8b0000",
    "darksalmon": "#e9967a",
    "darkseagreen": "#8fbc8f",
    "darkslateblue": "#483d8b",
    "darkslategray": "#2f4f4f",
    "darkturquoise": "#00ced1",
    "darkviolet": "#9400d3",
    "deeppink": "#ff1493",
    "deepskyblue": "#00bfff",
    "dimgray": "#696969",
    "dodgerblue": "#1e90ff",
    "firebrick": "#b222222",
    "floralwhite": "#fffaf0",
    "forestgreen": "#228b22",
    "gainsboro": "#dcdcdc",
    "ghostwhite": "#f8f8ff",
    "gold": "#ffd700",
    "goldenrod": "#daa520",
    "greenyellow": "#adff2f",
    "honeydew": "#f0fff0",
    "hotpink": "#ff69b4",
    "indianred": "#cd5c5c",
    "indigo": "#4b0082",
    "ivory": "#fffff0",
    "khaki": "#f0e68c",
    "lavender": "#e6e6fa",
    "lavenderblush": "#fff0f5",
    "lawngreen": "#7cfc00",
    "lemonchiffon": "#fffacd",
    "lightblue": "#add8e6",
    "lightcoral": "#f08080",
    "lightcyan": "#e0ffff",
    "lightgoldenrodyellow": "#fafad2",
    "lightgray": "#d3d3d3",
    "lightgreen": "#90ee90",
    "lightpink": "#ffb6c1",
    "lightsalmon": "#ffa07a",
    "lightseagreen": "#20b2aa",
    "lightskyblue": "#87cefa",
    "lightslategray": "#778899",
    "lightsteelblue": "#b0c4de",
    "lightyellow": "#ffffe0",
    "limegreen": "#32cd32",
    "linen": "#faf0e6",
    "mediumaquamarine": "#66cdaa",
    "mediumblue": "#0000cd",
    "mediumorchid": "#ba55d3",
    "mediumpurple": "#9370db",
    "mediumseagreen": "#3cb371",
    "mediumslateblue": "#7b68ee",
    "mediumspringgreen": "#00fa9a",
    "mediumturquoise": "#48d1cc",
    "mediumvioletred": "#c71585",
    "midnightblue": "#191970",
    "mintcream": "#f5fffa",
    "mistyrose": "#ffe4e1",
    "moccasin": "#ffe4b5",
    "navajowhite": "#ffdead",
    "oldlace": "#fdf5e6",
    "olivedrab": "#6b8e23",
    "orangered": "#ff4500",
    "orchid": "#da70d6",
    "palegoldenrod": "#eee8aa",
    "palegreen": "#98fb98",
    "paleturquoise": "#afeeee",
    "palevioletred": "#db7093",
    "papayawhip": "#ffefd5",
    "peachpuff": "#ffdab9",
    "peru": "#cd853f",
    "pink": "#ffc0cb",
    "plum": "#dda0dd",
    "powderblue": "#b0e0e6",
    "rosybrown": "#bc8f8f",
    "royalblue": "#4169e1",
    "saddlebrown": "#8b4513",
    "salmon": "#fa8072",
    "sandybrown": "#f4a460",
    "seagreen": "#2e8b57",
    "seashell": "#fff5ee",
    "sienna": "#a0522d",
    "skyblue": "#87ceeb",
    "slateblue": "#6a5acd",
    "slategray": "#708090",
    "snow": "#fffafa",
    "springgreen": "#00ff7f",
    "steelblue": "#4682b4",
    "tan": "#d2b48c",
    "thistle": "#d8bfd8",
    "tomato": "#ff6347",
    "turquoise": "#40e0d0",
    "violet": "#ee82ee",
    "wheat": "#f5deb3",
    "whitesmoke": "#f5f5f5",
    "yellowgreen": "#9acd32",
    # CSS Color Module Level 4
    "rebeccapurple": "#663399",
}
CSS_COLORS["cyan"] = CSS_COLORS["aqua"]
CSS_COLORS["darkgrey"] = CSS_COLORS["darkgray"]
CSS_COLORS["darkslategrey"] = CSS_COLORS["darkslategray"]
CSS_COLORS["dimgrey"] = CSS_COLORS["dimgray"]
CSS_COLORS["grey"] = CSS_COLORS["gray"]
CSS_COLORS["lightgrey"] = CSS_COLORS["lightgray"]
CSS_COLORS["lightslategrey"] = CSS_COLORS["lightslategray"]
CSS_COLORS["magenta"] = CSS_COLORS["fuchsia"]
CSS_COLORS["slategrey"] = CSS_COLORS["slategray"]


class Color(NamedTuple):
    """A definition of a color."""

    rgb: Rgb
    hsl: Hsl

    def __eq__(self, other: object) -> bool:
        """Determine if two colors are equal."""

        if not isinstance(other, Color):
            return NotImplemented
        return self.rgb == other.rgb or self.hsl == other.hsl

    @classmethod
    def from_rgb(cls, color: Rgb) -> "Color":
        """Create a color from an rgb object."""

        return cls(color, rgb_to_hsl(color))

    @classmethod
    def from_hex(cls, value: str) -> "Color":
        """Create a color from a hex value."""

        return cls.from_rgb(Rgb.from_hex(value))

    @classmethod
    def from_hsl(cls, color: Hsl) -> "Color":
        """Create a color from an hsl object."""

        return cls(hsl_to_rgb(color), color)

    @classmethod
    def from_ctor(cls, value: str) -> "Color":
        """Create a color from an hsl or rgb constructor string."""

        # Check if this is a canonical color.
        lower = value.lower()
        if lower in CSS_COLORS:
            return cls.from_hex(CSS_COLORS[lower])

        assert "rgb" in value or "hsl" in value
        if "rgb" in value:
            return cls.from_rgb(Rgb.from_ctor(value))
        return cls.from_hsl(Hsl.from_ctor(value))
