from fractions import Fraction
import pickle
import copy
from math import log, exp

general = {
    "det()": ["matrix M", "Returns the determinant of the given matrix.", "det"],
    "transpose()": ["matrix M", "Returns the transpose of the given matrix.", "transpose"],
    "zeroMatrix()": ["rows m, columns n=m", "Returns a zero matrix of the given size.", "zeroMatrix"],
    "identityMatrix()": ["order m", "Returns the identity matrix of the given order.", "identityMatrix"],
    "matrify()": ["vector v, format='col'", "Converts a 1d list array vector into a 2d list array vector, of specified format.", "matrify"],
    "dot()": ["vector v1, vector v2", "Returns the dot product of two vectors.", "dot"],
    "multiplyMatrices()": ["matrix A, matrix B", "Multiplies two matrices and returns the resulting matrix.", "multiplyMatrices"],
    "norm()": ["vector v, norm n=2", "Returns the nth norm of the given vector.", "norm"],
    "scale()": ["matrix M, num alpha", "Scales the given matrix by some constant alpha and returns the result.", "scale"],
    "addMatrices()": ["matrix A, matrix B", "Adds two matrices of equal size.", "addMatrices"],
    "axpy()": ["matrix x, matrix y, num alpha", "Runs the AXPY operation, returning the vector or matrix resulting from alpha*x+y.", "axpy"],
    "linComb()": ["matrix x, matrix y, num alpha, num beta", "Performs a linear combination, returning the vector or matrix resulting from alpha*x+beta*y.", "linComb"],
    "rowSwap()": ["matrix M, idx row1, idx row2", "Performs the row swap operation row1<->row2, and returns the resulting matrix.", "rowSwap"],
    "rowAddition()": ["matrix M, idx row1, idx row2", "Performs the row addition operation row1+row2->row1, and returns the resulting matrix.", "rowAddition"],
    "rowScale()": ["matrix M, idx row1, num alpha", "Performs the row scale operation alpha*row1->row1, and returns the resulting matrix.", "rowScale"],
    "scaledRowAddition()": ["matrix M, idx row1, idx row2, num alpha", "Performs the scaled row addition operation row1+alpha*row2->row1, and returns the resulting matrix.", "scaledRowAddition"],
    "hAppend()": ["matrix A, matrix B", "Appends B to the right of A, returning the resulting matrix.", "hAppend"],
    "trace()": ["matrix M", "Returns the trace of the given matrix.", "trace"],
    "mPrint()": ["matrix M, num s=1, num l=0", "Prints the values of a matrix in a unique format, with s spaces between entries and l lines between rows.", "mPrint"],
    "ratMPrint()": ["matrix M, num s=1, num l=0", "Prints the values of a matrix in a unique format, with s spaces between entries and l lines between rows. Entries will be in rational (fractional) format.", "ratMPrint"],
    "line()": ["num n=1", "Prints out n blank lines for readability purposes.", "line"],
    "cutLeftSquare()": ["matrix M", "Cuts a non-square matrix into a square one, keeping the upper left side.", "cutLeftSquare"],
    "cutRightSquare()": ["matrix M", "Cuts a non-square matrix into a square one, keeping the upper right side.", "cutRightSquare"],
    "cofactor()": ["matrix M", "Returns the cofactor matrix associated with the given matrix.", "cofactor"],
    "adjugate()": ["matrix M", "Returns the adjugate (adjoint) matrix associated with the given one, which is the transpose of its cofactor.", "adjugate"],
    "inverse()": ["matrix M", "Returns the inverse of a matrix, computed using the determinant and adjugate.", "inverse"],
    "invSystem()": ["matrix A, vector x, vector b, num r=1000", "Computes the solutions to the system Ax=b, using the inverse-multiplication method. A must be a square matrix of coefficients, x and b should be column vectors in string and numerical formats respectively. r is the rational-denominator limit. The output will be a list, including both the solution matrix and a series of solution strings.", "invSystem"],
    "solutionPrint()": ["list S", "Prints the solutions provided in a list generated by a function such as invSystem.", "solutionPrint"],
    "sysSign()": ["num n", "Returns a + for positive numbers and - for negative numbers.", "sysSign"],
    "unityElim()": ["num n", "Essentially deletes the number from print if it is a 1. Used only for dependency purposes.", "unityElim"],
    "inputSystem()": ["num r=1000, num suppress=0, num readback=1", "A program for user-friendly input in order to solve and output solutions to a system of linear equations. r is the rational-denominator limit. Suppress will cancel solution output if ==1, and readback will cancel equation readback output if ==0.", "inputSystem"],
    "help()": ["num repress", "The help function for the matrix module.", "modulehelp"],
    "create()": ["num s=0, num p=1", "A program accepting user input to create and store a matrix for use while in the command prompt. String input will only be allowed if s==1. p will change the matrix print to rational if ==2, and will suppress printing if ==0.", "create"],
    "remove()": ["str key", "Removes the matrix stored under the given key from the matrices index."],
    "addIndex()": ["matrix M, str key", "Adds a new matrix to the matrices index, under a given key."],
    "listMatrices()": ["num p=1", "Prints all matrices stored in the matrices index. Print defaults to rational, but if p is changed it will be in decimal form.", "listMatrices"],
    "saveMatrices()": ["str filename='matrices.txt'", "A function which saves the matrices index into a text file.", "saveMatrices"],
    "loadMatrices()": ["str filename='matrices.txt'", "A function which returns an index from a text file. To update the current matrix index, use MatrixModule.index=MatrixModule.loadMatrices(filename).", "loadMatrices"],
    "rref()": ["matrix M", "Returns the Reduced Row Echelon Form (RREF) of the given matrix, computed using Gauss-Jordan elimination."],
    "rInverse()": ["matrix M", "Computes the inverse of a matrix by putting an augmented matrix into RREF using Gauss-Jordan elimination, and returns this inverse."],
    "elimSystem()": ["matrix A, vector x, vector b, num r=1000", "Computes the solutions to the system Ax=b, using Gauss-Jordan elimination. A must be a square matrix of coefficients, x and b should be column vectors in string and numerical formats respectively. r is the rational-denominator limit. The output will be a list, including both the solution matrix and a series of solution strings.", "elimSystem"],
    "isMatrix()": ["obj A", "Returns True if the passed object is a matrix, otherwise returns False."],
    "nthRegression()": ["num r=0, matrix A=['empty'], num n=1, num rcomp=1", "Computes an nth-degree regression by the least squares method and returns the best fit polynomial. Printing output restricted if r==1. Matrix input and use of n (degree preset) only intended for use by other dependent regression functions. R^2 coefficient of determination will be computed if rcomp==1."],
    "pwrRegression()": ["num r=0, num rcomp=1", "Computes a power regression by a logarithmic-linear least squares method and returns the best fitting function y=ax^b. Printing output restricted if r==1. R^2 coefficient of determination will be computed if rcomp==1."],
    "expRegression()": ["num r=0, num rcomp=1", "Computes an exponential regression by a logarithmic-linear least squares method and returns the best fitting function y=ab^x. Printing output restricted if r==1. R^2 coefficient of determination will be computed if rcomp==1."]
}


def det(mat):
    matrix = copy.deepcopy(mat)
    # Returns the determinant of the given matrix, so long as the input matrix is square.
    m = len(matrix)
    n = len(matrix[0])
    if (m != n):  # Makes sure input is square
        print("Failed: Input matrix must be square.")
        return
    if(m == 1):  # If 1x1 then return inner value
        return matrix[0][0]
    output = 0  # Initialize output
    for k in range(0, n):  # Iterate through each item in the first row
        inner = []
        for i in range(1, m):  # Iterate through each row other than the first
            inRow = []
            for j in range(0, n):  # Iterate through each value in the current row
                if (j != k):  # Set aside all values not in the same column as the current row 1 value for recursive calculation
                    inRow.append(matrix[i][j])
            inner.append(inRow)
        # We will multiply each first row value by the determinant of its corresponding inner matrix and add/subtract these new values. Starting at 0, even valued indexes are added and odds are subtracted.
        if(k % 2 == 0):
            output += matrix[0][k] * det(inner)
        else:
            output -= matrix[0][k] * det(inner)
    return output


def transpose(mat):
    matrix = copy.deepcopy(mat)
    # Returns the transpose of a given matrix.
    m = len(matrix)
    n = len(matrix[0])
    output = []
    for i in range(0, n):  # Iterates through each column
        row = []
        for j in range(0, m):  # Converts current column into a row by iterating through each value
            row.append(matrix[j][i])
        output.append(row)  # Puts these new rows into the output
    return output


def zeroMatrix(num1, num2=0):
    m, n = copy.copy(num1), copy.copy(num2)
    # Returns a zero matrix of specified size.
    if(n == 0):
        # If number of columns is not specified then make the matrix square.
        n = m
    output = []
    for i in range(0, m):  # Create m rows
        row = []
        for j in range(0, n):  # Each row should have n values/columns
            row.append(0)  # Append necessary 0 values
        output.append(row)
    return output


def identityMatrix(ordr):
    order = copy.copy(ordr)
    # Returns the identity matrix of specified size.
    output = zeroMatrix(order)  # Begin with a zero matrix
    for i in range(0, order):
        output[i][i] = 1  # Puts values of 1 along the main diagonal.
    return output


def matrify(vec, format="col"):
    vector = copy.deepcopy(vec)
    # Convert a vector in list (1d list array) format into a matrix (2d list array) for the purposes of vector algebra functions which assume matrix-format. Specify either row (1xn) or column (mx1) vector format for conversion; column is default.
    s = len(vector)
    if (format == "row"):
        output = [[]]  # Initializes output as 1xn
        for i in range(0, s):  # Iterate through values to create matrix adaptation in row vector format
            output[0].append(vector[i])
        return output
    elif(format == "col" or format == "column"):
        output = []  # Initializes output as mx1
        # Iterate through values to create matrix adaptation in column vector format
        for i in range(0, s):
            output.append([vector[i]])
        return output
    else:
        # Catches formats which are not either row or column.
        return(format + " is not a valid format.")


def dot(vc1, vc2):
    vec1, vec2 = copy.deepcopy(vc1), copy.deepcopy(vc2)
    # Returns the dot product of two vectors. Input vectors should both be of the same orientation (row or column) and the same size, and in matrix format.
    m1, n1, m2, n2 = len(vec1), len(vec1[0]), len(vec2), len(vec2[0])
    if (not (m1 == m2 and n1 == n2) or not(m1 == 1 or n1 == 1)):  # Test for size and formatting
        return "Vectors must be of equal size and properly formatted."
    output = 0  # Initialize constant output value
    if(m1 == 1):  # If the vectors are row vectors:
        for i in range(0, n1):  # Iterate through values and multiply correspondingly
            output += (vec1[0][i]*vec2[0][i])
        return output
    if(n1 == 1):  # If the vectors are column vectors:
        for i in range(0, m1):  # Iterate through values and multiply correspondingly
            output += (vec1[i][0]*vec2[i][0])
        return output


def multiplyMatrices(mat1, mat2):
    matrix1, matrix2 = copy.deepcopy(mat1), copy.deepcopy(mat2)
    # Multiplies two matrices and returns the resulting matrix.
    m1, n1, m2, n2 = len(matrix1), len(
        matrix1[0]), len(matrix2), len(matrix2[0])
    # Confirms that the number of columns in the first matrix is equal to the number of rows in the second.
    if (n1 != m2):
        return "Matrices are not of compatible sizes."
    # Initializes output matrix as a blank/zero matrix, with the number of rows from the first matrix and columns from the second.
    output = zeroMatrix(m1, n2)
    for i in range(0, m1):  # Iterate through each row of the first matrix.
        rowV = matrix1[i]
        # Iterate through each column of the second matrix.
        for j in range(0, n2):
            colV = []
            for k in range(0, m2):  # Set aside all values of a column into a vector
                colV.append(matrix2[k][j])
            # Take the dot product of these vectors, converted into corresponding matrix formats, and put it into the proper position in the resulting matrix.
            output[i][j] = dot(matrify(rowV), matrify(colV))
    return output


def norm(vec, nr=2):
    vector, nrm = copy.deepcopy(vec), copy.copy(nr)
    # Returns the nth norm of a vector. If norm is not specified, 2 is used as default, which is the same as the vector's magnitude. The vector must be in matrix list format.
    m = len(vector)
    n = len(vector[0])
    # Make sure the vector is of proper size and format.
    if not (m == 1 or n == 1):
        return "Input must be a vector."
    vec = []  # Initialize a list which is easy to iterate through.
    if(m == 1):  # If it's a row vector then just use the one row as the list.
        vec = vector[0]
    else:  # If it's a column vector then iterate to create the list.
        for i in range(0, m):
            vec.append(vector[i][0])
    value = 0
    if (nrm == "inf"):  # Support for inf-norm
        for j in range(0, len(vec)):  # Iterate to find max value.
            if (vec[j] > value):
                value = vec[j]
        return value
    else:  # For regular n-norms, calculate the norm value through iteration.
        for j in range(0, len(vec)):
            value += (vec[j] ** nrm)
        return (value**(1/nrm))


def scale(mat, a):
    matrix, alpha = copy.deepcopy(mat), copy.copy(a)
    # Returns the matrix scaled by the given constant value.
    m = len(matrix)
    n = len(matrix[0])
    output = matrix
    for i in range(0, m):  # Iterate through rows
        for j in range(0, n):  # Iterate through values in these rows
            output[i][j] *= alpha  # Perform individual scalar multiplication
    return output


def addMatrices(mat1, mat2):
    matrix1, matrix2 = copy.deepcopy(mat1), copy.deepcopy(mat2)
    # Adds two matrices and returns the resulting matrix.
    m1, n1, m2, n2 = len(matrix1), len(
        matrix1[0]), len(matrix2), len(matrix2[0])
    # Confirms that the matrices are the same size.
    if not (m1 == m2 and n1 == n2):
        return "Matrices must be the same sizes."
    output = matrix1
    for i in range(0, m1):  # Iterate through rows
        for j in range(0, n1):  # Iterate through values in these rows
            output[i][j] += matrix2[i][j]  # Add corresponding values.
    return output


def axpy(mx, my, alpha):
    x, y, a = copy.deepcopy(mx), copy.deepcopy(my), copy.copy(alpha)
    # Performs the AXPY operation, or single-scaled addition, on two matrices; returning the resulting matrix.
    return addMatrices(scale(x, a), y)


def linComb(mx, my, alpha, beta):
    x, y, a, b = copy.deepcopy(mx), copy.deepcopy(
        my), copy.copy(alpha), copy.copy(beta)
    # Returns the linear combination of two scaled matrices.
    return addMatrices(scale(x, a), scale(y, b))


def rowSwap(mat, rw1, rw2):
    matrix, r1, r2 = copy.deepcopy(mat), copy.copy(rw1), copy.copy(rw2)
    # Performs the operation R1<->R2. Inputs should be the 0-indices of the desired rows to swap. Returns resulting matrix.
    output = matrix
    output[r1], output[r2] = output[r2], output[r1]
    return output


def rowAddition(mat, rw1, rw2):
    matrix, r1, r2 = copy.deepcopy(mat), copy.copy(rw1), copy.copy(rw2)
    # Performs the operation R1+R2->R1. Inputs should be the 0-indices of the desired rows to add. Returns resulting matrix.
    output = matrix
    for i in range(0, len(output[r1])):  # Iterative addition
        output[r1][i] += output[r2][i]
    return output


def rowScale(mat, rw, alph):
    matrix, row, alpha = copy.deepcopy(mat), copy.copy(rw), copy.copy(alph)
    # Performs the operation AR1->R1. Input the 0-index of the desired row and the scalar Alpha. Returns resulting matrix.
    output = matrix
    for i in range(0, len(output[row])):  # Iterative scaling
        output[row][i] *= alpha
    return output


def scaledRowAddition(mat, rw1, rw2, alph):
    matrix, r1, r2, alpha = copy.deepcopy(mat), copy.copy(
        rw1), copy.copy(rw2), copy.copy(alph)
    # Performs the operation R1+AR2->R1. Inputs should be the 0-indices of the desired rows to add, as well as the scalar Alpha. Returns resulting matrix.
    output = matrix
    for i in range(0, len(output[r1])):  # Iterative scaled addition
        output[r1][i] += alpha*output[r2][i]
    return output


def hAppend(mat1, mat2):
    matrix1, matrix2 = copy.deepcopy(mat1), copy.deepcopy(mat2)
    # Appends matrix2 to the right of matrix, and returns the resulting matrix.
    m1 = len(matrix1)
    m2 = len(matrix2)
    # Confirms that the matrices have the same number of rows.
    if not (m1 == m2):
        return "Matrices must have the same number of rows."
    aC = len(matrix2[0])
    output = matrix1
    for i in range(0, m1):  # Iterates through each row
        # Appends each value of this row from the second matrix into the first.
        for j in range(0, aC):
            output[i].append(matrix2[i][j])
    return output


def trace(mat):
    matrix = copy.deepcopy(mat)
    # Returns the trace of a square matrix (the sum of elements along the main diagonal).
    m, n = len(matrix), len(matrix[0])
    if (m != n):  # Confiems that matrix is square.
        return "Input matrix must be square."
    output = 0
    for i in range(0, m):  # Iterates through and sums the main diagonal.
        output += matrix[i][i]
    return output


def mPrint(mat, s=1, l=0):
    matrix = copy.deepcopy(mat)
    # Prints the values of a matrix or vector in a specific format designed for readability.
    # There will be s spaces between each entry of a row, and l lines between rows.
    m, n = len(matrix), len(matrix[0])
    for i in range(0, m):
        str = ""
        spaces = ""
        for a in range(0, s):
            spaces += " "
        for j in range(0, n):
            str += ("%s%s") % (spaces, matrix[i][j])
        print(str)
        for k in range(0, l):
            print
    return


def ratMPrint(mat, s=1, l=0):
    matrix = copy.deepcopy(mat)
    # Prints the values of a matrix or vector in a specific format designed for readability, in rational (fractional) format rather than decimal.
    # There will be s spaces between each entry of a row, and l lines between rows.
    m, n = len(matrix), len(matrix[0])
    for i in range(0, m):
        str = ""
        spaces = ""
        for a in range(0, s):
            spaces += " "
        for j in range(0, n):
            str += ("%s%s") % (spaces, Fraction(matrix[i]
                                                [j]).limit_denominator())
        print(str)
        for k in range(0, l):
            print(" ")
    return


def line(n=1):
    # Prints n blank lines.
    for i in range(0, n):
        print(" ")
    return


def cutLeftSquare(mat):
    matrix = copy.deepcopy(mat)
    # Cuts a non square matrix into a square, focused on the upper left side, by removing rows or columns from the bottom or right as needed.
    m, n = len(matrix), len(matrix[0])
    output = []
    if (m > n):  # If there are too many rows, just recreate the matrix with fewer.
        for i in range(0, n):
            output.append(matrix[i])
    else:  # Otherwise, iterate through each row and only allow the proper number of values in
        for i in range(0, m):
            row = []
            for j in range(0, m):
                row.append(matrix[i][j])
            output.append(row)
    return output


def cutRightSquare(mat):
    matrix = copy.deepcopy(mat)
    # Cuts a non square matrix into a square, focused on the upper right side, by removing rows or columns from the bottom or left as needed.
    m, n = len(matrix), len(matrix[0])
    output = []
    if (m > n):  # If there are too many rows, just recreate the matrix with fewer.
        for i in range(0, n):
            output.append(matrix[i])
    else:  # Otherwise, iterate through each row and only allow the proper values in.
        for i in range(0, m):
            row = []
            # Make sure to iterate only on right-side values.
            for j in range(n-m, m+1):
                row.append(matrix[i][j])
            output.append(row)
    return output


def cofactor(mat):
    matrix = copy.deepcopy(mat)
    # Returns the cofactor matrix associated with the given matrix.
    m, n = len(matrix), len(matrix[0])
    if (m != n):
        return("Matrix must be square.")
    output = zeroMatrix(m)  # Initialize the output matrix
    for i in range(0, m):  # iterate through all rows
        for j in range(0, n):  # iterate through all entries in a row
            subMtx = []  # Initialize the associated submatrix for each entry
            for r in range(0, m):
                if (r != i):
                    subRow = []
                    for c in range(0, n):
                        if (c != j):  # Append the proper entries to the submatrix
                            subRow.append(matrix[r][c])
                    subMtx.append(subRow)
            # Calculate the minor and then the cofactor for the entry
            output[i][j] = ((-1)**(i+j) * det(subMtx))
    return output


def adjugate(mat):
    matrix = copy.deepcopy(mat)
    # Returns the adjugate (adjoint) matrix associated with the given matrix, which is the transpose of the associated cofactor matrix.
    return transpose(cofactor(matrix))


def inverse(mat):
    matrix = copy.deepcopy(mat)
    # Returns the inverse of a matrix, calculated using the determinant and adjugate of that matrix.
    return scale(adjugate(matrix), 1/det(matrix))


def invSystem(mA, vx, vb, rat=1000):
    A, x, b = copy.deepcopy(mA), copy.deepcopy(vx), copy.deepcopy(vb)
    # Calculates the solutions to the system Ax=b, using the method of inverse multiplication.
    # Returns a list, beginning with the solution matrix, followed by a series of strings expressing these solutions.
    # Only works on square systems, where there are as many coefficients in a row of A as there are unknowns in x.
    # A should be an mxm matrix. x and b both should be mx1 column vectors in matrix format.
    # A should be the matrix of coefficients, as numerical values. x should be the vector of unknowns, as string values. b should be the vector of constants, as numerical values.
    output = []  # Initialize the output list
    if not(len(A) == len(A[0]) == len(x) == len(b)):  # Confirms compatiblity
        output.append("Matrices are not of compatible sizes.")
    else:
        m = len(x)
        # Perform the actual calculation
        solMtx = multiplyMatrices(inverse(A), b)
        output.append(solMtx)  # Primary element of list inserted
        strList = []
        # Will provide the solution strings in rational fraction format, unless passed a 'rat' value of 0.
        for i in range(0, m):  # Iterates through each unknown
            if(rat != 0):
                # Creates readable string in rational fraction form for the solution to this unknown
                st = "%s = %s" % (x[i][0], Fraction(
                    solMtx[i][0]).limit_denominator(rat))
            else:
                # Creates readable string for the solution to this unknown
                st = "%s = %s" % (x[i][0], solMtx[i][0])
            strList.append(st)  # Appends this solution string into the list
        # Appends the solution string list into the output
        output.append(strList)
    return output


def solutionPrint(sols):
    solutions = copy.deepcopy(sols)
    # Prints the solutions to a system of equations generated by another function in a readable way.
    m = len(solutions[1])
    for i in range(0, m):
        print(solutions[1][i])
    return


def sysSign(n):
    num = copy.copy(n)
    if(num >= 0):
        return "+"
    elif(num < 0):
        return "-"
    else:
        return ""


def unityElim(n):
    num = copy.copy(n)
    if(num == 1):
        return ""
    else:
        return num


def help(rep=0):
    modulehelp(rep)
    return


def modulehelp(rep=0):
    if(rep == 0):
        print("This is the Matrix Module V0.0.11, a package created by Elliott Walker. Type 'funcs' for a list of available functions, 'details' for a more detailed list including parameters, or 'exit' to exit this help function.")
    else:
        print("Available commands: 'funcs', 'details', 'exit'")
    inp = input()
    if(inp == "func" or inp == "functions" or inp == "list" or inp == "funcs"):
        print("Available functions under Matrix Module:")
        for key in general.keys():
            print(key)
    elif(inp == "details" or inp == "det" or inp == "dets" or inp == "more"):
        print("Available functions under Matrix Module (with parameters and descriptions):")
        line()
        for key in general.keys():
            print("%s  Parameters: %s;  Description: %s" %
                  (key, general[key][0], general[key][1]))
            line()
    elif(inp == "exit"):
        return
    help(1)
    return


index={}

def create(string=0, p=1):
    pr=copy.copy(p)
    print("Enter size of matrix, e.g. '3x3':")
    sizeIn=input()
    sizeArr=sizeIn.split("x")
    if (len(sizeArr) != 2):
        print("Program failure: Improper formatting of input.")
        return
    m=int(sizeArr[0])
    n=int(sizeArr[1])
    print("Enter a key to store this %d x %d matrix in." % (m, n))
    key=input()
    if(key in index.keys()):
        print("Program failure: Key already exists in index.")
        return
    index[key]=zeroMatrix(m, n)
    print("Matrix stored in index['%s']." % (key))
    for i in range(m):
        for j in range(n):
            if(string == 1):
                index[key][i][j]=str(
                    input("Input item at position %d, %d: " % (i+1, j+1)))
            else:
                index[key][i][j]=int(
                    input("Input value at position %d, %d: " % (i+1, j+1)))
    print("Matrix created.")
    if pr == 1:
        mPrint(index[key])
    elif pr == 2:
        ratMPrint(index[key])
    return


def remove(key):
    if key in index:
        del index[key]
    return

def addIndex(matrix, key):
    if (matrix.isMatrix == True):
        if (key in index.keys()):
            print("Program failure: Key already exists in index.")
        else:
            index[key]=matrix
    else:
        print("Program failure: Object passed is not a matrix.")
    return

def listMatrices(rat=1):
    print("All stored matrices:")
    for key in index.keys():
        print("")
        print(" At index['%s']: " % (key))
        if rat == 1:
            ratMPrint(index[key])
        else:
            mPrint(index[key])
    print("")
    print("Access a stored matrix using MatrixModule.index['key'].")
    return

def saveMatrices(filename="matrices.txt"):
    # Saves the matrices index to a file using the pickle dump method.
    file=open(filename, "wb")
    file.truncate(0)
    pickle.dump(index, file)
    file.close()
    return


def loadMatrices(filename="matrices.txt"):
    # Loads a matrices index from a pickle-binary text file.
    file=open(filename, "rb")
    index=pickle.load(file)
    file.close()
    return index


def rref(mat):
    matrix=copy.deepcopy(mat)
    # Through Gauss-Jordan elimination, reduces a matrix to reduced row echelon form (RREF) and returns the resulting matrix. Only works if the matrix is square or fat: tall matrices (m>n) will not work with the given algorithm.
    m, n=len(matrix), len(matrix[0])
    if (m > n):
        print("Failure: tall matrices (m>n) cannot reduce through given method.")
        return
    output=matrix
    # Row Reduction
    for i in range(m):  # Iterate through each row.
        # If the row's lead value isn't 1, then scale the row by dividing by that value so that it is.
        if (output[i][i] != 1):
            output=rowScale(output, i, 1/output[i][i])
        # Now iterate through all of the other rows, and perform scaled addition to make the values in the column of the current leading 1 become zeroes.
        for j in range(i+1, m):
            output=scaledRowAddition(output, j, i, -output[j][i])
    # Back Substitution
    # Iterate through the rows, excluding the last one, backwards.
    for i in range(m-1, -1, -1):
        # For each element to the right of the diagonal, subtract the row with a leading 1 in that column (scaled by the value of the aforementioned element) from the current row.
        for j in range(m-1, i, -1):
            output=scaledRowAddition(output, i, j, -output[i][j])
    return output


def rInverse(mat):
    matrix=copy.deepcopy(mat)
    # Computes the inverse of a matrix using Gauss-Jordan elimination on an augmented matrix, and returns this resulting inverse matrix.
    if (det(matrix) == 0 or len(matrix) != len(matrix[0])):
        return("Matrix must be invertible.")
    return cutRightSquare(rref(hAppend(matrix, identityMatrix(len(matrix)))))


def elimSystem(mA, vx, vb, rat=1000):
    A, x, b=copy.deepcopy(mA), copy.deepcopy(vx), copy.deepcopy(vb)
    # Calculates the solutions to the system Ax=b, using the method of Gaussian elimination.
    # Returns a list, beginning with the solution matrix, followed by a series of strings expressing these solutions.
    # Only works on square systems, where there are as many coefficients in a row of A as there are unknowns in x.
    # A should be an mxm matrix. x and b both should be mx1 column vectors in matrix format.
    # A should be the matrix of coefficients, as numerical values. x should be the vector of unknowns, as string values. b should be the vector of constants, as numerical values.
    output=[]
    if not(len(A) == len(A[0]) == len(x) == len(b)):  # Confirms compatiblity
        output.append("Matrices are not of compatible sizes.")
    else:
        m=len(x)
        # Perform the actual calculation
        solMtx=rref(hAppend(A, b))
        output.append(solMtx)  # Primary element of list inserted
        strList=[]
        # Will provide the solution strings in rational fraction format, unless passed a 'rat' value of 0.
        for i in range(0, m):  # Iterates through each unknown
            if(rat != 0):
                # Creates readable string in rational fraction form for the solution to this unknown
                st="%s = %s" % (x[i][0], Fraction(
                    solMtx[i][len(x)]).limit_denominator(rat))
            else:
                # Creates readable string for the solution to this unknown
                st="%s = %s" % (x[i][0], solMtx[i][len(x)])
            strList.append(st)  # Appends this solution string into the list
        # Appends the solution string list into the output
        output.append(strList)
    return output


def inputSystem(rat=1000, suppress=0, readback=1):
    # Accepts input for a much more user-friendly method of solving a system of linear equations.
    print("Would you like to solve using Gauss-Jordan elimination (1), or computational inverse (2)?")
    methodIn=input()
    if (int(methodIn) == 2):
        method="Inv"
    else:
        method="GJ"
    print("How many unknowns?")
    m=int(input())
    x=zeroMatrix(m, 1)
    for i in range(0, m):
        print("What is the unknown at index %d?" % (i))
        x[i][0]=input()
    print("When inputting coefficients separate them with spaces. For example, input 4x+3y=9 as '4 3 9'.")
    A=zeroMatrix(m)
    b=zeroMatrix(m, 1)
    for j in range(0, m):
        print("What are the coefficients in the equation of index %d?" % (j))
        B=input().split()
        if (len(B) != (m+1)):
            print("Program failure: Incorrect number of coefficients entered.")
            return
        for k in range(0, m+1):
            if (k == m):
                b[j][0]=int(B[k])
            else:
                A[j][k]=int(B[k])
    if (readback == 1):
        line()
        print("Your input:")
        for row in range(0, m):
            equ=""
            for val in range(0, m+1):
                if (val == m):
                    equ += ("= %d" % (b[row][0]))
                elif (val == 0):
                    equ += ("%s%s " % (unityElim(A[row][val]), x[val][0]))
                else:
                    equ += ("%s %s%s " %
                            (sysSign(A[row][val]), unityElim(abs(A[row][val])), x[val][0]))
            print(equ)
    if (det(A) == 0):
        print("System dependent - method failed.")
        return
    if (method == "Inv"):
        output=invSystem(A, x, b, rat)
    else:
        output=elimSystem(A, x, b, rat)
    if (suppress == 0):
        line()
        print("Solution:")
        solutionPrint(output)
        line()
        input("Press return to exit.")
    return(output)

def isMatrix(obj):
    if (type(obj)==list):
        for i in range(len(obj)):
            if (obj[i].type() != list):
                return False
        return True
    else:
        return False

def nthRegression(restrict=0,inMatrix=["empty"],ndeg=1,rcomp=1):
    # Performs least-squares regression in order to find the best fitting nth-degree polynomial. For n > 6 the computation may take a long time / not work, as computational expense goes up exponentially with increasing n (inverse of (n+1)x(n+1) matrix must be taken...)
    npoints = 1
    if(inMatrix==["empty"]):
        print("What degree regression would you like to perform?")
        ndeg = int(input())
        print("Would you like to use Matrix input (1) or Manual input (2)?")
        inmode = input()
        dataMt = []
        if (int(inmode) == 1):
            # I don't have anything to handle input matrices that are too small at the moment.
            print("Enter index of the dataset matrix to input. For example, to access the matrix at location index['matA'], simply type 'matA' (no quotes).")
            indx = input()
            dataMt = index[str(indx)]
        else:
            print("How many datapoints would you like to input? You must have at least %d datapoints. When entering datapoints use the format x,y."%(ndeg))
            npoints = int(input())
            if(npoints < ndeg):
                print("Not enough datapoints. Failed.")
                return
            for i in range(npoints):
                print("Enter datapoint at index %d:"%(i))
                ptIn = input()
                ptArr = ptIn.split(",")
                if(len(ptArr) != 2):
                    print("Program failure: Improper formatting of input.")
                    return
                dataMt.append([float(ptArr[0]),float(ptArr[1])])
    else:
        dataMt = inMatrix
        npoints = len(dataMt)
    xset = []
    yset = []
    for i in range(npoints):
        yset.append([dataMt[i][1]])
        rowMat = []
        xcur = dataMt[i][0]
        for j in range(ndeg+1):
            rowMat.append(xcur**j)
        xset.append(rowMat)
    coeffs = multiplyMatrices(multiplyMatrices(inverse(multiplyMatrices(transpose(xset),xset)),transpose(xset)),yset)
    outpoly = "y = "
    for i in range(ndeg+1):
        outpoly += ("%sx^%d" % (Fraction(coeffs[i][0]).limit_denominator(), i)) # Readable format y = ax^0 + bx^1 + cx^2 + ...
        if(i < ndeg):
            outpoly += " + "
    if(restrict != 1):
        print("outpoly")
    R2 = 0
    if(rcomp == 1):
        def polyFunction(x):
            y = 0
            for i in range(ndeg+1):
                y += coeffs[i][0]*(x**i)
            return y
        # Compute R^2 value (coefficient of determination)
        # Begin by computing RSS (Residual Sum of Squares)
        RSS = 0
        for i in range(len(dataMt)):
            RSS += (dataMt[i][1]-polyFunction(dataMt[i][0]))**2 # Sum of squares of residuals (errors between dataset values and model prediction)
        # Now compute TSS (Total Sum of Squares)
        TSS = 0
        mu = 0 # Mean. Will sum all values (requiring starting value of 0) and then divide by number of values.
        for i in range(len(dataMt)):
            mu += dataMt[i][1]
        mu /= len(dataMt)
        for i in range(len(dataMt)):
            TSS += (dataMt[i][1]-mu)**2 # Total sum of squares, which is proportional to variance. Sum of squares of differences between y-values and mean of y-values.
        R2 = 1 - RSS/TSS # General definition for R^2 given RSS and TSS, see Coefficient of Determination on Wikipedia.
        print("R^2 = %s" % (R2))
    return [coeffs,outpoly,R2]
    
def pwrRegression(restrict=0,rcomp=1):
    # Computes the closest power-regression (form of y=ax^b) fit to a dataset using logarithmic least-squares
    print("Would you like to use Matrix input (1) or Manual input (2)?")
    inmode = input()
    lnMt = []
    if (int(inmode) == 1):
        # I don't have anything to handle input matrices that are too small at the moment.
        print("Enter index of the dataset matrix to input. For example, to access the matrix at location index['matA'], simply type 'matA' (no quotes).")
        indx = input()
        lnMt = index[str(indx)]
    else:
        print("How many datapoints would you like to input? You must have at least 3 datapoints. When entering datapoints use the format x,y.")
        npoints = int(input())
        if(npoints < 3):
            print("Not enough datapoints. Failed.")
            return
        for i in range(npoints):
            print("Enter datapoint at index %d:"%(i))
            ptIn = input()
            ptArr = ptIn.split(",")
            if(len(ptArr) != 2):
                print("Program failure: Improper formatting of input.")
                return
            lnMt.append([log(float(ptArr[0])),log(float(ptArr[1]))]) # Converts coordinates to (log(x),log(y)) to exploit fact that log(ax^b) = log(a) + b*log(x), which is linear.
    lnMt_reg = nthRegression(1,lnMt,1,0)[0] # Performs linear regression (n=1) on logarithmic dataset
    lnMt_reg[0][0] = exp(lnMt_reg[0][0]) # Accounts for fact that "y-intercept" value is still in natural logarithm form by performing exponentiation
    outeq = "y = %sx^(%s)" % (Fraction(lnMt_reg[0][0]).limit_denominator(),Fraction(lnMt_reg[1][0]).limit_denominator()) # Readable format y = ax^(b)
    if(restrict != 1):
        print(outeq)
    R2 = 0
    if(rcomp == 1):
        a = lnMt_reg[0][0]
        b = lnMt_reg[1][0]
        def pwrFunction(x):
            return a*(x**b)
        # Compute R^2 value (coefficient of determination)
        # Begin by computing RSS (Residual Sum of Squares)
        RSS = 0
        for i in range(len(lnMt)):
            RSS += (exp(lnMt[i][1])-pwrFunction(exp(lnMt[i][0])))**2 # Sum of squares of residuals (errors between dataset values and model prediction)
        # Now compute TSS (Total Sum of Squares)
        TSS = 0
        mu = 0 # Mean. Will sum all values (requiring starting value of 0) and then divide by number of values.
        for i in range(len(lnMt)):
            mu += exp(lnMt[i][1])
        mu /= len(lnMt)
        for i in range(len(lnMt)):
            TSS += (exp(lnMt[i][1])-mu)**2 # Total sum of squares, which is proportional to variance. Sum of squares of differences between y-values and mean of y-values.
        R2 = 1 - RSS/TSS # General definition for R^2 given RSS and TSS, see Coefficient of Determination on Wikipedia.
        print("R^2 = %s" % (R2))
    return [lnMt_reg,outeq,R2]

def expRegression(restrict=0,rcomp=1):
    # Computes the closest exponential-regression (form of y=ab^x) fit to a dataset using logarithmic least-squares
    print("Would you like to use Matrix input (1) or Manual input (2)?")
    inmode = input()
    lnMt = []
    if (int(inmode) == 1):
        # I don't have anything to handle input matrices that are too small at the moment.
        print("Enter index key of the dataset matrix to input.")
        indx = input()
        lnMt = index[str(indx)] # At the current stage of development, this needs to already be in (x,log(y)) format.
    else:
        print("How many datapoints would you like to input? You must have at least 3 datapoints. When entering datapoints use the format x,y.")
        npoints = int(input())
        if(npoints < 3):
            print("Not enough datapoints. Failed.")
            return
        for i in range(npoints):
            print("Enter datapoint at index %d:"%(i))
            ptIn = input()
            ptArr = ptIn.split(",")
            if(len(ptArr) != 2):
                print("Program failure: Improper formatting of input.")
                return
            lnMt.append([float(ptArr[0]),log(float(ptArr[1]))]) # Converts coordinates to (x,log(y)) to exploit fact that log(ab^x) = log(a) + x*log(b), which is linear.
    lnMt_reg = nthRegression(1,lnMt,1,0)[0] # Performs linear regression (n=1) on logarithmic dataset
    lnMt_reg[0][0] = exp(lnMt_reg[0][0]) # Account for a-value still being in natural logarithm form by exponentiating
    lnMt_reg[1][0] = exp(lnMt_reg[1][0]) # Account for b-value logarithm form, as above
    outeq = "y = %s*%s^x" % (Fraction(lnMt_reg[0][0]).limit_denominator(),Fraction(lnMt_reg[1][0]).limit_denominator()) # Readable format y = ax^(b)
    if(restrict != 1):
        print(outeq)
    R2 = 0
    if(rcomp == 1):
        a = lnMt_reg[0][0]
        b = lnMt_reg[1][0]
        def expFunction(x):
            return a*(b**x)
        # Compute R^2 value (coefficient of determination)
        # Begin by computing RSS (Residual Sum of Squares)
        RSS = 0
        for i in range(len(lnMt)):
            RSS += (exp(lnMt[i][1])-expFunction(lnMt[i][0]))**2 # Sum of squares of residuals (errors between dataset values and model prediction)
        # Now compute TSS (Total Sum of Squares)
        TSS = 0
        mu = 0 # Mean. Will sum all values (requiring starting value of 0) and then divide by number of values.
        for i in range(len(lnMt)):
            mu += exp(lnMt[i][1])
        mu /= len(lnMt)
        for i in range(len(lnMt)):
            TSS += (exp(lnMt[i][1])-mu)**2 # Total sum of squares, which is proportional to variance. Sum of squares of differences between y-values and mean of y-values.
        R2 = 1 - RSS/TSS # General definition for R^2 given RSS and TSS, see Coefficient of Determination on Wikipedia.
        print("R^2 = %s" % (R2))
    return [lnMt_reg,outeq,R2]