"""Модуль построения окна гуи"""

import binascii
import hashlib
import os
import sys

from PyQt5.QtCore import Qt
from PyQt5.QtGui import QStandardItemModel, QStandardItem
from PyQt5.QtWidgets import QMainWindow, QApplication, QMessageBox, \
    QPushButton, QTableView, QAction, qApp, QLabel, \
    QLineEdit, QFileDialog, QTableWidget, QDialog, QComboBox

import logs.server_log_config

class MainWindow(QMainWindow):
    """Главное окно серверного приложения"""

    def __init__(self, database, server, config):
        super().__init__()
        self.database = database
        self.server_thread = server
        self.config = config

        exitAction = QAction('Exit', self)
        exitAction.setShortcut('Ctrl+Q')
        exitAction.triggered.connect(qApp.quit)
        self.setWindowTitle('App Server')

        self.refresh_button = QAction('Update List', self)
        self.setting_server = QAction('Server Setting', self)
        self.clients_history = QAction('Client History', self)
        self.register_btn = QAction('Client registry', self)
        self.remove_btn = QAction('Remove client', self)

        self.tool_bar = self.addToolBar('MainBar')
        self.tool_bar.addAction(self.refresh_button)
        self.tool_bar.addAction(self.setting_server)
        self.tool_bar.addAction(self.clients_history)
        self.tool_bar.addAction(self.register_btn)
        self.tool_bar.addAction(self.remove_btn)
        self.tool_bar.addAction(exitAction)

        self.setFixedSize(425, 610)
        self.setWindowTitle('App Server alpha release')

        self.label = QLabel('List of connected clients:', self)
        self.label.setFixedSize(240, 15)
        self.label.move(10, 40)

        self.active_clients_table = QTableView(self)
        self.active_clients_table.move(0, 65)
        self.active_clients_table.setFixedSize(425, 525)

        self.refresh_button.triggered.connect(self.create_users_model)
        self.clients_history.triggered.connect(self.show_statistics)
        self.setting_server.triggered.connect(self.server_config)
        self.register_btn.triggered.connect(self.reg_user)
        self.remove_btn.triggered.connect(self.rem_user)

        self.show()

    def create_users_model(self):
        """
        Метод заполняющий таблицу активных пользователей.
        """

        list_users = self.database.active_users_list()
        list = QStandardItemModel()
        list.setHorizontalHeaderLabels(
            ['Client Name', 'IP Address', 'Port', 'Conn... time'])
        for row in list_users:
            user, ip, port, time = row
            user = QStandardItem(user)
            user.setEditable(False)
            ip = QStandardItem(ip)
            ip.setEditable(False)
            port = QStandardItem(str(port))
            port.setEditable(False)
            time = QStandardItem(str(time.replace(microsecond=0)))
            time.setEditable(False)
            list.appendRow([user, ip, port, time])
        self.active_clients_table.setModel(list)
        self.active_clients_table.resizeColumnsToContents()
        self.active_clients_table.resizeRowsToContents()

    def show_statistics(self):
        """
        Метод создающий окно со статистикой клиентов.
        """

        global stat_window
        stat_window = HistoryWindow(self.database)
        stat_window.show()

    def server_config(self):
        """
        Метод создающий окно с настройками сервера.
        """

        global config_window
        config_window = ConfigWindow(self.config)

    def reg_user(self):
        """
        Метод создающий окно регистрации пользователя.
        """

        global reg_window
        reg_window = RegisterUser(self.database, self.server_thread)
        reg_window.show()

    def rem_user(self):
        """
        Метод создающий окно удаления пользователя.
        """

        global rem_window
        rem_window = DelUserDialog(self.database, self.server_thread)
        rem_window.show()


class HistoryWindow(QMainWindow):
    """
    Класс - окно отображения истоии пользователей
    """

    def __init__(self, database):
        super().__init__()
        self.database = database
        self.initUI()

    def initUI(self):
        """
        Функция иницивализации окна истории
        """

        self.setWindowTitle('Client statistics')
        self.setFixedSize(600, 700)
        self.setAttribute(Qt.WA_DeleteOnClose)

        self.close_button = QPushButton('Close', self)
        self.close_button.move(250, 650)
        self.close_button.clicked.connect(self.close)

        self.history_table = QTableView(self)
        self.history_table.move(0, 10)
        self.history_table.setFixedSize(600, 630)

        self.create_stat_model()

    def create_stat_model(self):
        """
        Функция отображения истории
        """

        hist_list = self.database.message_history()

        list = QStandardItemModel()
        list.setHorizontalHeaderLabels(
            ['Client name', 'Last login time', 'Messages sent',
             'Messages received'])
        for row in hist_list:
            user, last_seen, sent, recvd = row
            user = QStandardItem(user)
            user.setEditable(False)
            last_seen = QStandardItem(str(last_seen.replace(microsecond=0)))
            last_seen.setEditable(False)
            sent = QStandardItem(str(sent))
            sent.setEditable(False)
            recvd = QStandardItem(str(recvd))
            recvd.setEditable(False)
            list.appendRow([user, last_seen, sent, recvd])
        self.history_table.setModel(list)
        self.history_table.resizeColumnsToContents()
        self.history_table.resizeRowsToContents()


class ConfigWindow(QMainWindow):
    """
    Класс - окна конфигурации сервера
    """

    def __init__(self, config):
        super().__init__()
        self.config = config
        self.initUI()

    def initUI(self):
        """
        Функция инициалтзации окна конфигурации
        """

        self.setFixedSize(415, 260)
        self.setWindowTitle('Server Setting')
        self.setAttribute(Qt.WA_DeleteOnClose)
        # self.setModal(True)

        self.db_path_label = QLabel('The path to the database file: ', self)
        self.db_path_label.move(10, 10)
        self.db_path_label.setFixedSize(240, 15)

        self.db_path = QLineEdit(self)
        self.db_path.setFixedSize(250, 20)
        self.db_path.move(10, 35)
        self.db_path.setReadOnly(True)

        self.db_path_select = QPushButton('View...', self)
        self.db_path_select.move(275, 28)

        self.db_path_select.clicked.connect(self.open_file_dialog)

        self.db_file_label = QLabel('Name of the database file: ', self)
        self.db_file_label.move(10, 68)
        self.db_file_label.setFixedSize(200, 15)

        self.db_file = QLineEdit(self)
        self.db_file.move(240, 66)
        self.db_file.setFixedSize(150, 20)

        self.port_label = QLabel('Port number for connections:', self)
        self.port_label.move(10, 108)
        self.port_label.setFixedSize(220, 15)

        self.port = QLineEdit(self)
        self.port.move(240, 108)
        self.port.setFixedSize(150, 20)

        self.ip_label = QLabel(
            'From which IP do we \n accept connections:', self)
        self.ip_label.move(10, 148)
        self.ip_label.setFixedSize(180, 30)

        self.ip_label_note = QLabel(
            'leave this field empty to \n accept connections from any addresses.',
            self)
        self.ip_label_note.move(10, 178)
        self.ip_label_note.setFixedSize(500, 40)

        self.ip = QLineEdit(self)
        self.ip.move(240, 148)
        self.ip.setFixedSize(150, 20)

        self.save_btn = QPushButton('Save', self)
        self.save_btn.move(190, 220)

        self.close_button = QPushButton('Close', self)
        self.close_button.move(275, 220)
        self.close_button.clicked.connect(self.close)

        self.show()

        self.db_path.insert(self.config['SETTINGS']['Database_path'])
        self.db_file.insert(self.config['SETTINGS']['Database_file'])
        self.port.insert(self.config['SETTINGS']['Default_port'])
        self.ip.insert(self.config['SETTINGS']['Listen_Address'])
        self.save_btn.clicked.connect(self.save_server_config)

    def open_file_dialog(self):
        '''Метод обработчик открытия окна выбора папки.'''
        global dialog
        dialog = QFileDialog(self)
        path = dialog.getExistingDirectory()
        path = path.replace('/', '\\')
        self.db_path.clear()
        self.db_path.insert(path)

    def save_server_config(self):
        """
        Метод сохранения настроек.
        Проверяет правильность введённых данных и
        если всё правильно сохраняет ini файл.
        """

        global config_window
        message = QMessageBox()
        self.config['SETTINGS']['Database_path'] = self.db_path.text()
        self.config['SETTINGS']['Database_file'] = self.db_file.text()
        try:
            port = int(self.port.text())
        except ValueError:
            message.warning(self, 'Ошибка', 'Порт должен быть числом')
        else:
            self.config['SETTINGS']['Listen_Address'] = self.ip.text()
            if 1023 < port < 65536:
                self.config['SETTINGS']['Default_port'] = str(port)
                dir_path = os.getcwd()
                dir_path = os.path.join(dir_path, '..')
                with open(f"{dir_path}/{'server.ini'}", 'w') as conf:
                    self.config.write(conf)
                    message.information(
                        self, 'OK', 'Настройки успешно сохранены!')
            else:
                message.warning(
                    self, 'Ошибка', 'Порт должен быть от 1024 до 65536')


class RegisterUser(QDialog):
    """
    Класс диалог регистрации пользователя на сервере.
    """

    def __init__(self, database, server):
        super().__init__()

        self.database = database
        self.server = server

        self.setWindowTitle('Регистрация')
        self.setFixedSize(175, 183)
        # self.setModal(True)
        self.setAttribute(Qt.WA_DeleteOnClose)

        self.label_username = QLabel('Введите имя пользователя:', self)
        self.label_username.move(10, 10)
        self.label_username.setFixedSize(150, 15)

        self.client_name = QLineEdit(self)
        self.client_name.setFixedSize(154, 20)
        self.client_name.move(10, 30)

        self.label_passwd = QLabel('Введите пароль:', self)
        self.label_passwd.move(10, 55)
        self.label_passwd.setFixedSize(150, 15)

        self.client_passwd = QLineEdit(self)
        self.client_passwd.setFixedSize(154, 20)
        self.client_passwd.move(10, 75)
        self.client_passwd.setEchoMode(QLineEdit.Password)
        self.label_conf = QLabel('Введите подтверждение:', self)
        self.label_conf.move(10, 100)
        self.label_conf.setFixedSize(150, 15)

        self.client_conf = QLineEdit(self)
        self.client_conf.setFixedSize(154, 20)
        self.client_conf.move(10, 120)
        self.client_conf.setEchoMode(QLineEdit.Password)

        self.btn_ok = QPushButton('Сохранить', self)
        self.btn_ok.move(10, 150)
        self.btn_ok.clicked.connect(self.save_data)

        self.btn_cancel = QPushButton('Выход', self)
        self.btn_cancel.move(90, 150)
        self.btn_cancel.clicked.connect(self.close)

        self.messages = QMessageBox()

        self.show()

    def save_data(self):
        """
        Метод проверки правильности ввода и сохранения в базу нового пользователя.
        """

        if not self.client_name.text():
            self.messages.critical(
                self, 'Ошибка', 'Не указано имя пользователя.')
            return
        elif self.client_passwd.text() != self.client_conf.text():
            self.messages.critical(
                self, 'Ошибка', 'Введённые пароли не совпадают.')
            return
        elif self.database.check_user(self.client_name.text()):
            self.messages.critical(
                self, 'Ошибка', 'Пользователь уже существует.')
            return
        else:
            passwd_bytes = self.client_passwd.text().encode('utf-8')
            salt = self.client_name.text().lower().encode('utf-8')
            passwd_hash = hashlib.pbkdf2_hmac(
                'sha512', passwd_bytes, salt, 10000)
            self.database.add_user(
                self.client_name.text(),
                binascii.hexlify(passwd_hash))
            self.messages.information(
                self, 'Успех', 'Пользователь успешно зарегистрирован.')
            self.server.service_update_lists()
            self.close()


class DelUserDialog(QDialog):
    """
    Класс - диалог выбора контакта для удаления.
    """

    def __init__(self, database, server):
        super().__init__()
        self.database = database
        self.server = server

        self.setFixedSize(350, 120)
        self.setWindowTitle('Удаление пользователя')
        self.setAttribute(Qt.WA_DeleteOnClose)
        self.setModal(True)

        self.selector_label = QLabel(
            'Выберите пользователя для удаления:', self)
        self.selector_label.setFixedSize(200, 20)
        self.selector_label.move(10, 0)

        self.selector = QComboBox(self)
        self.selector.setFixedSize(200, 20)
        self.selector.move(10, 30)

        self.btn_ok = QPushButton('Удалить', self)
        self.btn_ok.setFixedSize(100, 30)
        self.btn_ok.move(230, 20)
        self.btn_ok.clicked.connect(self.remove_user)

        self.btn_cancel = QPushButton('Отмена', self)
        self.btn_cancel.setFixedSize(100, 30)
        self.btn_cancel.move(230, 60)
        self.btn_cancel.clicked.connect(self.close)

        self.all_users_fill()

    def all_users_fill(self):
        """
        Метод заполняющий список пользователей.
        """

        self.selector.addItems([item[0]
                                for item in self.database.users_list()])

    def remove_user(self):
        """
        Метод - обработчик удаления пользователя.
        """

        self.database.remove_user(self.selector.currentText())
        if self.selector.currentText() in self.server.names:
            sock = self.server.names[self.selector.currentText()]
            del self.server.names[self.selector.currentText()]
            self.server.remove_client(sock)
        self.server.service_update_lists()
        self.close()


if __name__ == '__main__':
    pass
    # app = QApplication(sys.argv)
    # message = QMessageBox
    # # window = MainWindow()
    # # window.active_clients_table.setModel(gui_create_model([]))
    # # window = HistoryWindow()
    # # window.history_table.setModel(create_stat_model([]))
    # window = ConfigWindow()
    #
    # app.exec_()
