import numpy as np
import pandas as pd
import matplotlib.pyplot as plt 
from mhkit.wave.resource import significant_wave_height as _sig_wave_height
from mhkit.wave.resource import peak_period as _peak_period
from mhkit.river.graphics import _xy_plot


def plot_spectrum(S, ax=None):
    """
    Plots wave amplitude spectrum versus omega
    
    Parameters
    ------------
    S: pandas DataFrame
        Spectral density [m^2/Hz] indexed frequency [Hz]
    ax : matplotlib axes object
        Axes for plotting.  If None, then a new figure is created.
    Returns
    ---------
    ax : matplotlib pyplot axes
    
    """
    assert isinstance(S, pd.DataFrame), 'S must be of type pd.DataFrame'
    
    f = S.index
    
    ax = _xy_plot(f*2*np.pi, S/(2*np.pi), fmt='-', xlabel='omega [rad/s]', 
             ylabel='Spectral density [m$^2$s/rad]', ax=ax)
    
    """
    spectrum_type = S.columns
    if S.shape[1] == 1:
        Hm0 = _sig_wave_height(S).iloc[0,0]
        Tp0 = _peak_period(S).iloc[0,0]
        title = 'Spectrum: ' + spectrum_type[0] + ' \n Tp = {:0.2f}, Hs = {:0.2f}'.format(Tp0,Hm0)
        ax.set_title(title)
    else:
        ax.legend(spectrum_type)
    """   
    return ax

def plot_elevation_timeseries(eta, ax=None):
    """
    Plot wave surface elevation time-series
    
    Parameters
    ------------
    eta: pandas DataFrame
        Wave surface elevation [m] indexed by time [datetime or s]
    ax : matplotlib axes object
        Axes for plotting.  If None, then a new figure is created.
        
    Returns
    ---------
    ax : matplotlib pyplot axes
            
    """
    
    assert isinstance(eta, pd.DataFrame), 'eta must be of type pd.DataFrame'
    
    ax = _xy_plot(eta.index, eta, fmt='-', xlabel='Time', 
                  ylabel='$\eta$ [m]', ax=ax)
    
    return ax

def plot_matrix(M, xlabel='Te', ylabel='Hm0', zlabel=None, show_values=True, 
                ax=None):
    """
    Plots values in the matrix as a scatter diagram

    Parameters
    ------------
    M: pandas DataFrame
        Matrix with numeric labels for x and y axis, and numeric entries.
        An example would be the average capture length matrix generated by
        mhkit.device.wave, or something similar.
    xlabel: string (optional)
        Title of the x-axis
    ylabel: string (optional)
        Title of the y-axis
    zlabel: string (optional)
        Colorbar label
    show_values : bool (optional)
        Show values on the scatter diagram
    ax : matplotlib axes object
        Axes for plotting.  If None, then a new figure is created.
    
    Returns
    ---------
    ax : matplotlib pyplot axes
    
    """
    assert isinstance(M, pd.DataFrame), 'M must be of type pd.DataFrame'
    
    if ax is None:
        plt.figure()
        ax = plt.gca()
        
    im = ax.imshow(M, origin='lower', aspect='auto')
    
    # Add colorbar
    cbar = plt.colorbar(im)
    if zlabel:
        cbar.set_label(zlabel, rotation=270, labelpad=15)
    
    # Set x and y label
    ax.set_xlabel(xlabel)
    ax.set_ylabel(ylabel)
    
    # Show values in the plot
    if show_values:
        for i, col in enumerate(M.columns):
            for j, index in enumerate(M.index):
                if not np.isnan(M.loc[index,col]):
                    ax.text(i, j, format(M.loc[index,col], '.2f'), ha="center", va="center")
    
    # Reset x and y ticks
    ax.set_xticks(np.arange(len(M.columns)))
    ax.set_yticks(np.arange(len(M.index)))
    ax.set_xticklabels(M.columns)
    ax.set_yticklabels(M.index)

    return ax

