"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mode = exports.StaticSiteAuthorization = exports.SpaAuthorization = exports.Authorization = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const core_1 = require("@aws-cdk/core");
const cdk_lambda_at_edge_pattern_1 = require("@cloudcomponents/cdk-lambda-at-edge-pattern");
const auth_flow_1 = require("./auth-flow");
const retrieve_user_pool_client_secret_1 = require("./retrieve-user-pool-client-secret");
const secret_generator_1 = require("./secret-generator");
const user_pool_client_redirects_1 = require("./user-pool-client-redirects");
const user_pool_domain_1 = require("./user-pool-domain");
class Authorization extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.userPool = props.userPool;
        this.redirectPaths = (_a = props.redirectPaths) !== null && _a !== void 0 ? _a : {
            signIn: '/parseauth',
            authRefresh: '/refreshauth',
            signOut: '/',
        };
        this.signOutUrlPath = (_b = props.signOutUrl) !== null && _b !== void 0 ? _b : '/signout';
        this.httpHeaders = (_c = props.httpHeaders) !== null && _c !== void 0 ? _c : {
            'Content-Security-Policy': "default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self' 'unsafe-inline'; object-src 'none'; connect-src 'self'",
            'Strict-Transport-Security': 'max-age=31536000; includeSubdomains; preload',
            'Referrer-Policy': 'same-origin',
            'X-XSS-Protection': '1; mode=block',
            'X-Frame-Options': 'DENY',
            'X-Content-Type-Options': 'nosniff',
            'Cache-Control': 'no-cache',
        };
        this.oauthScopes = (_d = props.oauthScopes) !== null && _d !== void 0 ? _d : [aws_cognito_1.OAuthScope.PHONE, aws_cognito_1.OAuthScope.EMAIL, aws_cognito_1.OAuthScope.PROFILE, aws_cognito_1.OAuthScope.OPENID, aws_cognito_1.OAuthScope.COGNITO_ADMIN];
        this.cookieSettings = props.cookieSettings;
        this.userPoolClient = this.createUserPoolClient();
        this.nonceSigningSecret = this.generateNonceSigningSecret();
        this.cognitoAuthDomain = this.retrieveCognitoAuthDomain();
        this.authFlow = this.createAuthFlow((_e = props.logLevel) !== null && _e !== void 0 ? _e : cdk_lambda_at_edge_pattern_1.LogLevel.WARN);
    }
    updateUserPoolClientCallbacks(redirects) {
        const { callbackUrls, logoutUrls } = redirects;
        new user_pool_client_redirects_1.UserPoolClientRedirects(this, 'UserPoolClientRedirects', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            callbackUrls,
            logoutUrls,
        });
    }
    createDefaultBehavior(origin, options) {
        return {
            origin,
            forwardQueryString: true,
            compress: true,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            edgeLambdas: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
            ...options,
        };
    }
    createLegacyDefaultBehavior(options) {
        return {
            isDefaultBehavior: true,
            forwardedValues: {
                queryString: true,
            },
            compress: true,
            lambdaFunctionAssociations: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
            ...options,
        };
    }
    createAdditionalBehaviors(origin, options) {
        return {
            [this.redirectPaths.signIn]: {
                origin,
                forwardQueryString: true,
                compress: true,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.parseAuth],
                ...options,
            },
            [this.redirectPaths.authRefresh]: {
                origin,
                forwardQueryString: true,
                compress: true,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.refreshAuth],
                ...options,
            },
            [this.signOutUrlPath]: {
                origin,
                forwardQueryString: true,
                compress: true,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.signOut],
                ...options,
            },
        };
    }
    createLegacyAdditionalBehaviors(options) {
        return [
            {
                pathPattern: this.redirectPaths.signIn,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.parseAuth],
                ...options,
            },
            {
                pathPattern: this.redirectPaths.authRefresh,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.refreshAuth],
                ...options,
            },
            {
                pathPattern: this.signOutUrlPath,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.signOut],
                ...options,
            },
        ];
    }
    generateNonceSigningSecret() {
        const { secret } = new secret_generator_1.SecretGenerator(this, 'SecretGenerator');
        return secret;
    }
    retrieveCognitoAuthDomain() {
        const userPoolDomain = new user_pool_domain_1.UserPoolDomain(this, 'UserPoolDomain', {
            userPool: this.userPool,
        });
        return userPoolDomain.cognitoAuthDomain;
    }
}
exports.Authorization = Authorization;
class SpaAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.SPA;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: false,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: [aws_cognito_1.UserPoolClientIdentityProvider.COGNITO],
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; SameSite=Lax',
                accessToken: 'Path=/; Secure; SameSite=Lax',
                refreshToken: 'Path=/; Secure; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
}
exports.SpaAuthorization = SpaAuthorization;
class StaticSiteAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.STATIC_SITE;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: true,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: [aws_cognito_1.UserPoolClientIdentityProvider.COGNITO],
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        const clientSecret = this.retrieveUserPoolClientSecret();
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            clientSecret,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                accessToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                refreshToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
    retrieveUserPoolClientSecret() {
        const { clientSecret } = new retrieve_user_pool_client_secret_1.RetrieveUserPoolClientSecret(this, 'RetrieveUserPoolClientSecret', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
        });
        return clientSecret;
    }
}
exports.StaticSiteAuthorization = StaticSiteAuthorization;
var Mode;
(function (Mode) {
    Mode["SPA"] = "SPA";
    Mode["STATIC_SITE"] = "STATIC_SITE";
})(Mode = exports.Mode || (exports.Mode = {}));
//# sourceMappingURL=data:application/json;base64,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