
from datetime import datetime

from django import forms
from django.apps import apps
from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from django.core.exceptions import ValidationError
from django.contrib.admin.widgets import AutocompleteSelect
from django.contrib.admin.sites import site

from product_barcode.utils import validate_barcode
from djforms.fields import DatePickerField
from categories.models import Category

from products.models import Product


class ProductForm(forms.ModelForm):

    def clean_bar_code(self):

        bar_code = self.cleaned_data.get('bar_code')

        if bar_code:
            validate_barcode(bar_code, self.instance.pk)

        return bar_code

    def clean(self):

        data = self.cleaned_data

        if not data.get('price_wholesale') or not data.get('price_retail'):
            return data

        if data['price_wholesale'] > data['price_retail']:
            raise ValidationError(
                _('Wholesale price can`t be greater than retail price'))

        return data

    class Meta:
        widgets = {
            'manufacturer': AutocompleteSelect(
                Product._meta.get_field('manufacturer').remote_field,
                site,
                attrs={'style': 'width: 200px'}
            )
        }
        model = Product
        fields = '__all__'


class SearchProductForm(forms.Form):

    code = forms.CharField(required=False)

    bar_code = forms.CharField(required=False)

    query = forms.CharField(required=False)

    category = forms.ModelChoiceField(
        queryset=Category.objects.all(),
        required=False,
        widget=forms.HiddenInput)

    def clean(self):

        cleaned_data = {}

        for k, v in self.cleaned_data.items():
            if v:
                cleaned_data[k] = v

        return cleaned_data


class HistoryForm(forms.Form):

    date_from = DatePickerField(label=_('Date from'))

    date_to = DatePickerField(label=_('Date to'))

    def __init__(self, data):

        today = datetime.now().date().strftime(settings.DATE_INPUT_FORMATS[0])

        super().__init__(
            data={
                'date_from': data.get('date_from', today),
                'date_to': data.get('date_to', today)
            }
        )


class AddProductForm(forms.ModelForm):

    stock = forms.FloatField(label=_('Stock'), initial=0)

    class Meta:
        model = apps.get_model('products', 'Product')
        fields = [
            'category',
            'name',
            'stock',
            'price_retail',
            'manufacturer',
            'code',
            'bar_code',
            'warehouse',
            'unit_type',
            'min_stock',
            'initial_currency',
            'price_retail'
        ]
