"""
Module for identifiers.

Strings are not sufficient as identifiers, especially when (re)-generating
unique identifiers.
"""

from abc import ABC, abstractmethod
from typing import Optional

from .location import Location

class Id(ABC):
    """Abstract base class for identifiers."""

    @abstractmethod
    def names(self) -> set['NamedId']:
        """
        Return a set of all named identifiers contained in this identifier.

        This method is for compability with AST classes.
        """
        ...

    def is_equiv(self, other) -> bool:
        """
        Check if this identifier is equivalent to another identifier.

        This method is for compability with AST classes.
        """
        return self == other

class UnderscoreId(Id):
    """
    Placeholder identifier.

    When used in an assignment, the value is not bound.
    This identifier is illegal as a variable.
    """

    def __repr__(self):
        return 'UnderscoreId()'

    def __str__(self):
        return '_'

    def __eq__(self, other):
        return isinstance(other, UnderscoreId)

    def __hash__(self):
        return hash(())

    def names(self) -> set['NamedId']:
        return set()

class NamedId(Id):
    """
    Named identifier.

    A named identifier consists of a base name and an optional
    count to indicate its version.
    """

    base: str
    count: Optional[int]

    def __init__(self, base: str, count: Optional[int] = None):
        if not isinstance(base, str):
            raise TypeError(f'expected a str, for {base}')
        self.base = base
        self.count = count

    def __repr__(self):
        return f'NamedId(\'{str(self)}\')'

    def __str__(self):
        if self.count is None:
            return self.base
        else:
            return f'{self.base}{self.count}'

    def __eq__(self, other):
        if not isinstance(other, NamedId):
            return False
        return self.base == other.base and self.count == other.count

    def __hash__(self):
        return hash((self.base, self.count))

    def names(self) -> set['NamedId']:
        return { self }


class SourceId(NamedId):
    """
    Named identifier from the original source.

    A source identifier should only be generated by the parser.
    """

    loc: Location
    """source location"""

    def __init__(self, base: str, loc: Location, count: Optional[int] = None):
        super().__init__(base, count)
        self.loc = loc

    def __repr__(self):
        return f'SourceId(\'{str(self)}\')'

    def __eq__(self, other):
        if not isinstance(other, NamedId | SourceId):
            return False
        return self.base == other.base and self.count == other.count

    def __hash__(self):
        return hash((self.base, self.count))
