"""
This module is an example of a barebones numpy reader plugin for napari.

It implements the Reader specification, but your plugin may choose to
implement multiple readers or even other plugin contributions. see:
https://napari.org/plugins/guides.html?#readers
"""
import numpy as np
import h5py
    
def napari_get_reader(path):
    """A basic implementation of a Reader contribution.

    Parameters
    ----------
    path : str or list of str
        Path to file, or list of paths.

    Returns
    -------
    function or None
        If the path is a recognized format, return a function that accepts the
        same path or list of paths, and returns a list of layer data tuples.
    """
    if isinstance(path, list):
        # reader plugins may be handed single path, or a list of paths.
        # if it is a list, it is assumed to be an image stack...
        # so we are only going to look at the first file.
        path = path[0]
        # show_info(path)
        
    # if we know we cannot read the file, we immediately return None.
    if path.endswith(".h5"):
        return reader_h5
    elif path.endswith(".npy"):
        return reader_npy
    else:
        return None

def reader_h5(path):
    """Take a path or list of paths and return a list of LayerData tuples.

    Readers are expected to return data as a list of tuples, where each tuple
    is (data, [add_kwargs, [layer_type]]), "add_kwargs" and "layer_type" are
    both optional.

    Parameters
    ----------
    path : str or list of str
        Path to file, or list of paths.

    Returns
    -------
    layer_data : list of tuples
        A list of LayerData tuples where each tuple in the list contains
        (data, metadata, layer_type), where data is a numpy array, metadata is
        a dict of keyword arguments for the corresponding viewer.add_* method
        in napari, and layer_type is a lower-case string naming the type of layer.
        Both "meta", and "layer_type" are optional. napari will default to
        layer_type=="image" if not provided
    """
    # handle both a string and a list of strings
    paths = [path] if isinstance(path, str) else path
    
    # Load data

    f = h5py.File(paths[0])
    
    key = None
    keys = list(f.keys())
    Nkeys = len(keys)
    
    if Nkeys == 1:
        key = keys[0]
    else:
        for i in range(Nkeys):
            if 'data' in keys[i].lower() and 'meta' not in keys[i].lower() and 'analog' not in keys[i].lower():
                key = keys[i]
    if key is None:
        print('No data set found')
        return None

    # data_name = list(f.keys())[0]
    print(key)
    dset = f[key][0,:,:,:,:]
    dset = np.sum(dset, axis = -2)
    
    data = np.squeeze(dset)

    # optional kwargs for the corresponding viewer.add_* method
    add_kwargs = {'colormap': 'magma'}

    layer_type = "image"  # optional, default is "image"
    return [(data, add_kwargs, layer_type)]


def reader_npy(path):
    """Take a path or list of paths and return a list of LayerData tuples.

    Readers are expected to return data as a list of tuples, where each tuple
    is (data, [add_kwargs, [layer_type]]), "add_kwargs" and "layer_type" are
    both optional.

    Parameters
    ----------
    path : str or list of str
        Path to file, or list of paths.

    Returns
    -------
    layer_data : list of tuples
        A list of LayerData tuples where each tuple in the list contains
        (data, metadata, layer_type), where data is a numpy array, metadata is
        a dict of keyword arguments for the corresponding viewer.add_* method
        in napari, and layer_type is a lower-case string naming the type of layer.
        Both "meta", and "layer_type" are optional. napari will default to
        layer_type=="image" if not provided
    """
    # handle both a string and a list of strings
    paths = [path] if isinstance(path, str) else path
    
    # load all files into array
    arrays = [np.load(_path) for _path in paths]
    # stack arrays into single array
    data = np.squeeze(np.stack(arrays))

    # optional kwargs for the corresponding viewer.add_* method
    add_kwargs = {'colormap': 'magma'}

    layer_type = "image"  # optional, default is "image"
    return [(data, add_kwargs, layer_type)]
