#####################################################################################
# Class that represents a token obtained from Cherwell
######################################################################################
import json
import time
from time import strptime
import datetime


class AccessToken(object):

    """
    The AccessToken class serves to provid a token class to store and manage the Bearer token associated with
    a Cherwell REST API connection.

    ----------
    ***NOTE***
    ----------
        This class should idealy not be instantiated directly, but is instantiated and managed through the
        CherwellAPI.CherwellClient.Connection class.

        However it can be retrieved using the 'token' attribute of the CherwellAPI.CherwellClient.Connection object
        and saved to persistent storage - using pickle as an example. Then it can be retrieved and reinstantiated as
        an AccessToken and passed into a new CherwellAPI.CherwellClient.Connection object when it is
        instantiated as a new object. In this way the token can be shared between client instances.

        The CherwellAPI.CherwellClient.Connection handles token expiration and refreshing of the token. It does not
        require further management

    Parameters
    ----------

    username : str

        The name of the Cherwell user to authenticate to the cherwell REST API

    expires: str

        The token expiry date/time in GMT

    access_token : str

        The bearer token used for authorisation on REST API calls

    issued : str

        The date and time (GMT) the token was issued

    expires_in : str

        The amount of time till token expiry

    token_type : str

        The type name of the token

    client_id : str

        The Cherwell REST API client key generated by the Cherwell admin client

    refresh_token : str

        The token string that can be used to request a new token, when it expires

    minutes_before_expiry : int

        The amount of minutes before token expiry, that a token will be set to expired - this enables the token
        to be refreshed before expiry (default = 2 minutes)

    """

    def __init__(self, username=None, expires=None, access_token=None, issued=None, expires_in=None,
                 token_type=None, client_id=None, refresh_token=None, minutes_before_expiry=2):

        # Populate with values passed in
        self.username = username
        self.expires = expires
        self.access_token = access_token
        self.issued = issued
        self.expires_in = expires_in
        self.token_type = token_type
        self.client_id = client_id
        self.refresh_token = refresh_token
        self.minutes_before_expiry = minutes_before_expiry

    def refresh_token(self):

        """ Returns the refresh token string"""

        return self.refresh_token

    def __repr__(self):

        """ Returns the json representation of the token"""

        return json.dumps({
            "username": self.username,
            ".expires": self.expires,
            "access_token": self.access_token,
            ".issued": self.issued,
            "expires_in": self.expires_in,
            "token_type": self.token_type,
            "as:client_id": self.client_id,
            "refresh_token": self.refresh_token
        })

    def access_token(self):

        """ Returns the access token string value"""

        return self.access_token()

    def current_time_gmt(self):

        """ Returns the current GMT time"""

        current_time = time.strftime("%a, %d %b %Y %H:%M:%S GMT", time.gmtime())
        current_time = strptime(current_time, "%a, %d %b %Y %H:%M:%S %Z")

        return datetime.datetime(*current_time[:6])

    def token_expiry_gmt(self):

        """ Returns the token expiry time in GMT"""

        return datetime.datetime(*strptime(self.expires, "%a, %d %b %Y %H:%M:%S %Z")[:6])

    def expired(self):

        """ Returns 'True' if token has expired, else 'False'"""

        if self.current_time_gmt() > (self.token_expiry_gmt() - datetime.timedelta(minutes=self.minutes_before_expiry)):
            return True
        else:
            return False

    def load(self, token_json):

        """
        Loads the token from passed in json

        Parameters:
        -----------

            token_json : json

                Attributes of the token in json format
        """

        try:
            self.username = token_json["username"]
            self.expires = token_json[".expires"]
            self.access_token = token_json["access_token"]
            self.issued = token_json[".issued"]
            self.expires_in = token_json["expires_in"]
            self.token_type = token_json["token_type"]
            self.client_id = token_json["as:client_id"]
            self.refresh_token = token_json["refresh_token"]

        except:
            raise ValueError("Invalid Token Values")

