"""module to automatically wrap methods in autogenerated low-level code and re-expose them as high-level functions"""

import inspect
from urllib.parse import urljoin

from beartype import beartype
from box import Box
import do_sdk_platform

from deeporigin.auth import get_tokens
from deeporigin.config import get_value
from deeporigin.utils.core import _get_method


@beartype
def add_functions_to_module(
    module,
    api_name: str,
) -> list:
    """utility function to dynamically add functions to a module

    This function works by calling setattr on the module.

    Args:
        module (str): name of the module
        api_name (str): name of the API

    Returns:
        set of methods that were added
    """
    methods = _get_client_methods(
        _get_api_client(
            api_name=api_name,
            configure=False,
        )
    )

    sanitized_methods = []

    for method in methods:
        # clean up the name so that it's more readable
        sanitized_method_name = method.split("controller")[-1]

        sanitized_method_name = sanitized_method_name.replace(
            "_without_preload_content", ""
        ).lstrip("_")

        sanitized_methods.append(sanitized_method_name)

        # add this function as an attribute to this module
        # so that we can call it
        setattr(
            module,
            sanitized_method_name,
            _create_function(
                method_path=method,
                api_name=api_name,
            ),
        )

    return sanitized_methods


@beartype
def _get_api_client(*, api_name: str, configure: bool = True):
    """return a configured client for the API we want to access

    Args:
        api_name (str): name of the API

    Returns:
        configured client
    """

    if configure:
        from do_sdk_platform.configuration import Configuration

        configuration = Configuration(
            host=urljoin(get_value()["api_endpoint"], "/api"),
            access_token=get_tokens()["access"],
        )

        client = do_sdk_platform.ApiClient(configuration=configuration)
    else:
        client = do_sdk_platform.ApiClient()

    api_class = getattr(do_sdk_platform, api_name)
    client = api_class(api_client=client)
    return client


@beartype
def _get_client_methods(client) -> set:
    """utility function to get methods from the client that return raw responses from the server"""
    methods = set(
        [
            attr
            for attr in dir(client)
            if callable(getattr(client, attr))
            and not attr.startswith("_")
            and "without_preload_content" in attr
        ]
    )

    return methods


def _create_function(*, method_path: str, api_name: str):
    """utility function the dynamically creates functions
    that wrap low-level functions in the DeepOrigin data API"""

    # we're constructing a client solely for the purposes
    # of inspecting its methods and extracting
    # function signatures. So we don't need any
    # authentication

    client = _get_api_client(
        configure=False,
        api_name=api_name,
    )

    method = _get_method(client, method_path)

    signature = inspect.signature(method)

    def dynamic_function(
        *,
        client=None,
        **kwargs,
    ):
        """dynamic function that wraps low-level functions in the DeepOrigin platform API"""

        if client is None:
            client = _get_api_client(api_name=api_name)
        method = _get_method(client, method_path)

        # call the low level API method
        response = method(**kwargs)

        if 400 <= response.status < 600:
            content = response.read().decode("utf-8", errors="replace")

            raise ValueError(
                f"HTTP request failed with status: {response.status} - {response.reason} - {content}"
            )

        if not isinstance(response, dict):
            response = response.json()

        if "data" in response.keys():
            response = response["data"]
            if isinstance(response, list):
                response = [Box(item) for item in response]
            else:
                response = Box(response)
        else:
            response = Box(response)

        return response

    # attach the signature of the underlying method to the
    # function so that IDEs can display it properly
    dynamic_function.__signature__ = signature

    return dynamic_function
