"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ShellScriptAction = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
/**
 * Validate a revision using shell commands
 */
class ShellScriptAction {
    constructor(props) {
        var _a, _b;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: props.actionName,
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [],
            outputs: [],
        };
        if (Object.keys((_a = props.useOutputs) !== null && _a !== void 0 ? _a : {}).length + ((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).length === 0) {
            throw new Error('You must supply either \'useOutputs\' or \'additionalArtifacts\', since a CodeBuild Action must always have at least one input artifact.');
        }
    }
    /**
     * The CodeBuild Project's principal
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d, _e;
        const inputs = new Array();
        inputs.push(...(_a = this.props.additionalArtifacts) !== null && _a !== void 0 ? _a : []);
        const envVarCommands = new Array();
        const bashOptions = (_b = this.props.bashOptions) !== null && _b !== void 0 ? _b : '-eu';
        if (bashOptions) {
            envVarCommands.push(`set ${bashOptions}`);
        }
        for (const [varName, output] of Object.entries((_c = this.props.useOutputs) !== null && _c !== void 0 ? _c : {})) {
            const outputArtifact = output.artifactFile;
            // Add the artifact to the list of inputs, if it's not in there already. Determine
            // the location where CodeBuild is going to stick it based on whether it's the first (primary)
            // input or an 'extra input', then parse.
            let artifactIndex = inputs.findIndex(a => a.artifactName === outputArtifact.artifact.artifactName);
            if (artifactIndex === -1) {
                artifactIndex = inputs.push(outputArtifact.artifact) - 1;
            }
            const dirEnv = artifactIndex === 0 ? 'CODEBUILD_SRC_DIR' : `CODEBUILD_SRC_DIR_${outputArtifact.artifact.artifactName}`;
            envVarCommands.push(`export ${varName}="$(node -pe 'require(process.env.${dirEnv} + "/${outputArtifact.fileName}")["${output.outputName}"]')"`);
        }
        this._project = new codebuild.PipelineProject(scope, 'Project', {
            environment: { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0 },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    build: {
                        commands: [
                            ...envVarCommands,
                            ...this.props.commands,
                        ],
                    },
                },
            }),
        });
        for (const statement of (_d = this.props.rolePolicyStatements) !== null && _d !== void 0 ? _d : []) {
            this._project.addToRolePolicy(statement);
        }
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.props.actionName,
            input: inputs[0],
            extraInputs: inputs.slice(1),
            runOrder: (_e = this.props.runOrder) !== null && _e !== void 0 ? _e : 100,
            project: this._project,
        });
        // Replace the placeholder actionProperties at the last minute
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
    }
    /**
     * Project generated to run the shell script in
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after ShellScriptAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.ShellScriptAction = ShellScriptAction;
//# sourceMappingURL=data:application/json;base64,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