"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const crypto = require("crypto");
const path = require("path");
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const construct_internals_1 = require("../private/construct-internals");
const _util_1 = require("./_util");
/**
 * A standard synth with a generated buildspec
 */
class SimpleSynthAction {
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        if (this.props.installCommand && this.props.installCommands) {
            throw new Error('Pass either \'installCommand\' or \'installCommands\', but not both');
        }
        if (this.props.buildCommand && this.props.buildCommands) {
            throw new Error('Pass either \'buildCommand\' or \'buildCommands\', but not both');
        }
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * Create a standard NPM synth action
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Create a standard Yarn synth action
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Project generated to run the synth command
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after SimpleSynthAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d;
        const buildCommands = (_a = this.props.buildCommands) !== null && _a !== void 0 ? _a : [this.props.buildCommand];
        const installCommands = (_b = this.props.installCommands) !== null && _b !== void 0 ? _b : [this.props.installCommand];
        const testCommands = (_c = this.props.testCommands) !== null && _c !== void 0 ? _c : [];
        const synthCommand = this.props.synthCommand;
        const buildSpec = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    commands: _util_1.filterEmpty([
                        this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                        ...installCommands,
                    ]),
                },
                build: {
                    commands: _util_1.filterEmpty([
                        ...buildCommands,
                        ...testCommands,
                        synthCommand,
                    ]),
                },
            },
            artifacts: renderArtifacts(this),
        });
        const environment = { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0, ...this.props.environment };
        const environmentVariables = {
            ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
            ...this.props.environmentVariables,
        };
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = hash(core_1.Stack.of(scope).resolve({
            environment,
            buildSpecString: buildSpec.toBuildSpec(),
            environmentVariables,
        }));
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: this.props.projectName,
            environment,
            buildSpec,
            environmentVariables,
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._project = project;
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_d = this.props.additionalArtifacts) !== null && _d !== void 0 ? _d : []).map(a => a.artifact)],
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: {
                _PROJECT_CONFIG_HASH: { value: projectConfigHash },
            },
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            var _a;
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * The CodeBuild Project's principal
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
function hash(obj) {
    const d = crypto.createHash('sha256');
    d.update(JSON.stringify(obj));
    return d.digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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