"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdatePipelineAction = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const cpactions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const construct_internals_1 = require("../private/construct-internals");
/**
 * Action to self-mutate the pipeline
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to deploy the pipeline stack.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline.
 */
class UpdatePipelineAction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const selfMutationProject = new codebuild.PipelineProject(this, 'SelfMutation', {
            projectName: props.projectName,
            environment: { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0 },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g aws-cdk${installSuffix}`,
                    },
                    build: {
                        commands: [
                            // Cloud Assembly is in *current* directory.
                            `cdk -a ${construct_internals_1.embeddedAsmPath(scope)} deploy ${props.pipelineStackName} --require-approval=never --verbose`,
                        ],
                    },
                },
            }),
        });
        // allow the self-mutating project permissions to assume the bootstrap Action role
        selfMutationProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: ['arn:*:iam::*:role/*-deploy-role-*', 'arn:*:iam::*:role/*-publishing-role-*'],
        }));
        selfMutationProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStacks'],
            resources: ['*'],
        }));
        // S3 checks for the presence of the ListBucket permission
        selfMutationProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:ListBucket'],
            resources: ['*'],
        }));
        this.action = new cpactions.CodeBuildAction({
            actionName: 'SelfMutate',
            input: props.cloudAssemblyInput,
            project: selfMutationProject,
        });
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.UpdatePipelineAction = UpdatePipelineAction;
//# sourceMappingURL=data:application/json;base64,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