"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaiterStateMachine = void 0;
const aws_iam_1 = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 */
class WaiterStateMachine extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const role = new aws_iam_1.Role(this, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('states.amazonaws.com'),
        });
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.isCompleteHandler.functionArn],
        }));
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.timeoutHandler.functionArn],
        }));
        const definition = core_1.Stack.of(this).toJsonString({
            StartAt: 'framework-isComplete-task',
            States: {
                'framework-isComplete-task': {
                    End: true,
                    Retry: [{
                            ErrorEquals: ['States.ALL'],
                            IntervalSeconds: props.interval.toSeconds(),
                            MaxAttempts: props.maxAttempts,
                            BackoffRate: props.backoffRate,
                        }],
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'framework-onTimeout-task',
                        }],
                    Type: 'Task',
                    Resource: props.isCompleteHandler.functionArn,
                },
                'framework-onTimeout-task': {
                    End: true,
                    Type: 'Task',
                    Resource: props.timeoutHandler.functionArn,
                },
            },
        });
        const resource = new core_1.CfnResource(this, 'Resource', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        resource.node.addDependency(role);
        this.stateMachineArn = resource.ref;
    }
    grantStartExecution(identity) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
}
exports.WaiterStateMachine = WaiterStateMachine;
//# sourceMappingURL=data:application/json;base64,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