"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCustomResource = exports.AwsCustomResourcePolicy = exports.PhysicalResourceId = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
// don't use "require" since the typescript compiler emits errors since this
// file is not listed in tsconfig.json.
const metadata = JSON.parse(fs.readFileSync(path.join(__dirname, 'sdk-api-metadata.json'), 'utf-8'));
const awsSdkMetadata = metadata;
/**
 * Physical ID of the custom resource.
 */
class PhysicalResourceId {
    /**
     * @param responsePath Path to a response data element to be used as the physical id.
     * @param id Literal string to be used as the physical id.
     */
    constructor(responsePath, id) {
        this.responsePath = responsePath;
        this.id = id;
    }
    /**
     * Extract the physical resource id from the path (dot notation) to the data in the API call response.
     */
    static fromResponse(responsePath) {
        return new PhysicalResourceId(responsePath, undefined);
    }
    /**
     * Explicit physical resource id.
     */
    static of(id) {
        return new PhysicalResourceId(undefined, id);
    }
}
exports.PhysicalResourceId = PhysicalResourceId;
/**
 * The IAM Policy that will be applied to the different calls.
 */
class AwsCustomResourcePolicy {
    /**
     * @param statements statements for explicit policy.
     * @param resources resources for auto-generated from SDK calls.
     */
    constructor(statements, resources) {
        this.statements = statements;
        this.resources = resources;
    }
    /**
     * Explicit IAM Policy Statements.
     *
     * @param statements the statements to propagate to the SDK calls.
     */
    static fromStatements(statements) {
        return new AwsCustomResourcePolicy(statements, undefined);
    }
    /**
     * Generate IAM Policy Statements from the configured SDK calls.
     *
     * Each SDK call with be translated to an IAM Policy Statement in the form of: `call.service:call.action` (e.g `s3:PutObject`).
     *
     * @param options options for the policy generation
     */
    static fromSdkCalls(options) {
        return new AwsCustomResourcePolicy([], options.resources);
    }
}
exports.AwsCustomResourcePolicy = AwsCustomResourcePolicy;
/**
 * Use this constant to configure access to any resource.
 */
AwsCustomResourcePolicy.ANY_RESOURCE = ['*'];
/**
 * Defines a custom resource that is materialized using specific AWS API calls.
 *
 * Use this to bridge any gap that might exist in the CloudFormation Coverage.
 * You can specify exactly which calls are invoked for the 'CREATE', 'UPDATE' and 'DELETE' life cycle events.
 *
 */
class AwsCustomResource extends cdk.Construct {
    // 'props' cannot be optional, even though all its properties are optional.
    // this is because at least one sdk call must be provided.
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (!props.onCreate && !props.onUpdate && !props.onDelete) {
            throw new Error('At least `onCreate`, `onUpdate` or `onDelete` must be specified.');
        }
        for (const call of [props.onCreate, props.onUpdate]) {
            if (call && !call.physicalResourceId) {
                throw new Error('`physicalResourceId` must be specified for onCreate and onUpdate calls.');
            }
        }
        for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
            if ((_a = call === null || call === void 0 ? void 0 : call.physicalResourceId) === null || _a === void 0 ? void 0 : _a.responsePath) {
                AwsCustomResource.breakIgnoreErrorsCircuit([call], 'PhysicalResourceId.fromResponse');
            }
        }
        this.props = props;
        const provider = new lambda.SingletonFunction(this, 'Provider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: '679f53fa-c002-430c-b0da-5b7982bd2287',
            lambdaPurpose: 'AWS',
            timeout: props.timeout || cdk.Duration.minutes(2),
            role: props.role,
            logRetention: props.logRetention,
            functionName: props.functionName,
        });
        this.grantPrincipal = provider.grantPrincipal;
        // Create the policy statements for the custom resource function role, or use the user-provided ones
        const statements = [];
        if (props.policy.statements.length !== 0) {
            // Use custom statements provided by the user
            for (const statement of props.policy.statements) {
                statements.push(statement);
            }
        }
        else {
            // Derive statements from AWS SDK calls
            for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
                if (call) {
                    const statement = new iam.PolicyStatement({
                        actions: [awsSdkToIamAction(call.service, call.action)],
                        resources: props.policy.resources,
                    });
                    statements.push(statement);
                }
            }
        }
        const policy = new iam.Policy(this, 'CustomResourcePolicy', {
            statements: statements,
        });
        if (provider.role !== undefined) {
            policy.attachToRole(provider.role);
        }
        const create = props.onCreate || props.onUpdate;
        this.customResource = new cdk.CustomResource(this, 'Resource', {
            resourceType: props.resourceType || 'Custom::AWS',
            serviceToken: provider.functionArn,
            pascalCaseProperties: true,
            properties: {
                create: create && encodeBooleans(create),
                update: props.onUpdate && encodeBooleans(props.onUpdate),
                delete: props.onDelete && encodeBooleans(props.onDelete),
                installLatestAwsSdk: (_b = props.installLatestAwsSdk) !== null && _b !== void 0 ? _b : true,
            },
        });
        // If the policy was deleted first, then the function might lose permissions to delete the custom resource
        // This is here so that the policy doesn't get removed before onDelete is called
        this.customResource.node.addDependency(policy);
    }
    static breakIgnoreErrorsCircuit(sdkCalls, caller) {
        for (const call of sdkCalls) {
            if (call === null || call === void 0 ? void 0 : call.ignoreErrorCodesMatching) {
                throw new Error(`\`${caller}\`` + ' cannot be called along with `ignoreErrorCodesMatching`.');
            }
        }
    }
    /**
     * Returns response data for the AWS SDK call.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getDataString` for string attributes.
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseFieldReference(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getData');
        return this.customResource.getAtt(dataPath);
    }
    /**
     * Returns response data for the AWS SDK call as string.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseField(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getDataString');
        return this.customResource.getAttString(dataPath);
    }
}
exports.AwsCustomResource = AwsCustomResource;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 * Example: CloudWatchLogs with putRetentionPolicy => logs:PutRetentionPolicy
 *
 * TODO: is this mapping correct for all services?
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = (awsSdkMetadata[srv] && awsSdkMetadata[srv].prefix) || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
/**
 * Encodes booleans as special strings
 */
function encodeBooleans(object) {
    return JSON.parse(JSON.stringify(object), (_k, v) => {
        switch (v) {
            case true:
                return 'TRUE:BOOLEAN';
            case false:
                return 'FALSE:BOOLEAN';
            default:
                return v;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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