"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyStackSynthesizer = void 0;
const cxschema = require("../../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const cfn_fn_1 = require("../cfn-fn");
const construct_compat_1 = require("../construct-compat");
const asset_parameters_1 = require("../private/asset-parameters");
const _shared_1 = require("./_shared");
const stack_synthesizer_1 = require("./stack-synthesizer");
/**
 * The well-known name for the docker image asset ECR repository. All docker
 * image assets will be pushed into this repository with an image tag based on
 * the source hash.
 */
const ASSETS_ECR_REPOSITORY_NAME = 'aws-cdk/assets';
/**
 * This allows users to work around the fact that the ECR repository is
 * (currently) not configurable by setting this context key to their desired
 * repository name. The CLI will auto-create this ECR repository if it's not
 * already created.
 */
const ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY = 'assets-ecr-repository-name';
/**
 * Use the original deployment environment
 *
 * This deployment environment is restricted in cross-environment deployments,
 * CI/CD deployments, and will use up CloudFormation parameters in your template.
 *
 * This is the only StackSynthesizer that supports customizing asset behavior
 * by overriding `Stack.addFileAsset()` and `Stack.addDockerImageAsset()`.
 */
class LegacyStackSynthesizer extends stack_synthesizer_1.StackSynthesizer {
    constructor() {
        super(...arguments);
        this.cycle = false;
        /**
         * The image ID of all the docker image assets that were already added to this
         * stack (to avoid duplication).
         */
        this.addedImageAssets = new Set();
    }
    bind(stack) {
        this.stack = stack;
    }
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        // Backwards compatibility hack. We have a number of conflicting goals here:
        //
        // - We want put the actual logic in this class
        // - We ALSO want to keep supporting people overriding Stack.addFileAsset (for backwards compatibility,
        // because that mechanism is currently used to make CI/CD scenarios work)
        // - We ALSO want to allow both entry points from user code (our own framework
        // code will always call stack.deploymentMechanism.addFileAsset() but existing users
        // may still be calling `stack.addFileAsset()` directly.
        //
        // Solution: delegate call to the stack, but if the stack delegates back to us again
        // then do the actual logic.
        //
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddFileAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addFileAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        // See `addFileAsset` for explanation.
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddDockerImageAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addDockerImageAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    /**
     * Synthesize the associated stack to the session
     */
    synthesize(session) {
        _shared_1.assertBound(this.stack);
        this.synthesizeStackTemplate(this.stack, session);
        // Just do the default stuff, nothing special
        this.emitStackArtifact(this.stack, session);
    }
    doAddDockerImageAsset(asset) {
        var _a, _b;
        _shared_1.assertBound(this.stack);
        // check if we have an override from context
        const repositoryNameOverride = this.stack.node.tryGetContext(ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY);
        const repositoryName = (_b = (_a = asset.repositoryName) !== null && _a !== void 0 ? _a : repositoryNameOverride) !== null && _b !== void 0 ? _b : ASSETS_ECR_REPOSITORY_NAME;
        const imageTag = asset.sourceHash;
        const assetId = asset.sourceHash;
        // only add every image (identified by source hash) once for each stack that uses it.
        if (!this.addedImageAssets.has(assetId)) {
            const metadata = {
                repositoryName,
                imageTag,
                id: assetId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget,
                file: asset.dockerFile,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
            this.addedImageAssets.add(assetId);
        }
        return {
            imageUri: `${this.stack.account}.dkr.ecr.${this.stack.region}.${this.stack.urlSuffix}/${repositoryName}:${imageTag}`,
            repositoryName,
        };
    }
    doAddFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const httpUrl = `https://s3.${this.stack.region}.${this.stack.urlSuffix}/${bucketName}/${objectKey}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    get assetParameters() {
        _shared_1.assertBound(this.stack);
        if (!this._assetParameters) {
            this._assetParameters = new construct_compat_1.Construct(this.stack, 'AssetParameters');
        }
        return this._assetParameters;
    }
}
exports.LegacyStackSynthesizer = LegacyStackSynthesizer;
//# sourceMappingURL=data:application/json;base64,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