"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApp = void 0;
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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